/*
 * File:      world.cpp
 * Purpose:	  generate Traveller world
 * Author:	
 * Created:	
 * Updated:	
 * Copyright: LGPL.
 *            Traveller is a registered trademark of Far Future Enterprises.	
 */

/* rcsid[] = "$RCSfile: world.cpp,v $ $Revision: 1.7 $ $Author: man $ $Date: 2000/10/11 02:44:17 $" */

// For compilers that support precompilation, includes "wx.h".
#include "wx/wxprec.h"

#ifndef WX_PRECOMP
#include "wx/wx.h"
#endif

#include <stdlib.h>
#include <stdarg.h>
#include "world.h"
#include "syst.h"
#include "wx/confbase.h"
#include "wx/fileconf.h"
#include "t_res.h"
#include "t_help.h"

IMPLEMENT_APP(WorldApp)

#define BIG_BUFFER_SIZE			120

// button defines
enum {
	ID_TOOLBAR = 500,
	TB_OPEN,
	TB_CONFIG,
	TB_GEN_WORLD,
	TB_GEN_SAT,
	TB_REGEN,
	TB_TWEAK,
	TB_SAVE,
	TB_PRINT,
	TB_ABOUT,
	TB_EXIT
};

// misc defines
#define BUILD_FRAME_WIDTH         500
#define BUILD_FRAME_HEIGHT        400
#define BUILD_PANEL_HEIGHT        80
//#define ID_TOOLBAR					501

#ifdef __WXGTK__
#include "bitmaps/open.xpm"
#include "bitmaps/config.xpm"
#include "bitmaps/help.xpm"
#include "bitmaps/homeworld.xpm"
#include "bitmaps/sat.xpm"
#include "bitmaps/restart.xpm"
#include "bitmaps/tweak_world.xpm"
#include "bitmaps/print.xpm"
#include "bitmaps/save.xpm"
#include "bitmaps/exit.xpm"
#endif

// handy reference
WorldFrame *frame = NULL;
wxFont *NormalFont = NULL;

// ===============================================================
// `Main program' equivalent, creating windows and returning main app frame
//wxFrame *
bool
WorldApp::OnInit(void)
{
char config_file[MAX_FILE_LENGTH];
//char *cfg_ptr,*temp_ptr;
char *fname=NULL;
char last_arg=0;
int ndx=-1,i=1;
char str[MAX_FILE_LENGTH];
wxConfig *config;
wxString ptr;

    // load general resource
	config = new wxConfig((const wxString) RESOURCE_FILE);
	wxConfigBase::Set(config);
	wxConfigBase::Get();

	sprintf(str, "/%s/%s", SECTION_DATA, ENTRY_DATA_DIR);
	if(config->Read((const wxString) str, &ptr)) {
		strcpy(config_file, ptr.GetData());
		strcat(config_file, "/");
		sprintf(str, "/%s/%s", SECTION_DATA, ENTRY_DATA_FILE);
		if(config->Read((const wxString) str, &ptr))
			strcat(config_file, ptr.GetData());
		else
			config_file[0] = 0;
	}
	wxConfigBase::Set((wxConfigBase *)NULL);
	delete config;

	// create fonts
    NormalFont = new wxFont(11, wxMODERN, wxNORMAL, wxNORMAL);

	// process args
	while(i < argc) {
		if('-' == argv[i][0]) {
			if('f' == argv[i][1]) {
				if(argv[i][2] != 0)
					fname = &argv[i][2];
				else
					last_arg = 'f';
			}
			if('i' == argv[i][1]) {
				if(argv[i][2] != 0)
					ndx = atoi(&argv[i][2]);
				else
					last_arg = 'i';
			}
		} else if(last_arg != 0) {
			if('f' == last_arg)
				fname = &argv[i][0];
			if('i' == last_arg)
				ndx = atoi(&argv[i][0]);
			last_arg = 0;
		} else {
			// error!
		}
		i++;
	}

	// create the frame
   	frame = new WorldFrame(config_file);
	frame->CreateToolBar(wxNO_BORDER | wxHORIZONTAL | wxTB_DOCKABLE, 
			ID_TOOLBAR);
	frame->InitButtons(frame->GetToolBar());
#ifdef __WINDOWS__
	frame->SetIcon(wxIcon("world_icn"));
#endif
#ifdef __WXGTK__
	frame->SetIcon(wxIcon(homeworld_xpm));
#endif

	if(fname != NULL) {
		frame->LoadSystem(fname);
		if(ndx > -1)
			frame->Generate(ndx);
	}

	frame->Show(TRUE);

	return(TRUE);
}

// ===============================================================
BEGIN_EVENT_TABLE(WorldFrame, wxFrame)
	EVT_MENU(TB_OPEN, WorldFrame::Open)
	EVT_MENU(TB_CONFIG, WorldFrame::Config)
	EVT_MENU(TB_GEN_WORLD, WorldFrame::GenWorld)
	EVT_MENU(TB_GEN_SAT, WorldFrame::GenSat)
	EVT_MENU(TB_REGEN, WorldFrame::Regen)
	EVT_MENU(TB_TWEAK, WorldFrame::Tweak)
	EVT_MENU(TB_SAVE, WorldFrame::Save)
	EVT_MENU(TB_PRINT, WorldFrame::Print)
	EVT_MENU(TB_ABOUT, WorldFrame::OnAbout)
	EVT_MENU(TB_EXIT, WorldFrame::Exit)
	EVT_TOOL_ENTER(ID_TOOLBAR, WorldFrame::OnToolEnter)
END_EVENT_TABLE()

// ---------------------------------------------------------------
WorldFrame::WorldFrame(char *cfg_ptr) :
	wxFrame(NULL, -1, "Generate World Detail",
		wxPoint(-1, -1), wxSize(BUILD_FRAME_WIDTH, BUILD_FRAME_HEIGHT))
{
	codes = new TCodes(cfg_ptr);
	sys = new System(codes);
	detail = new Detail();

	choice = NULL;
	tweak_dlg = NULL;

	// now do the window stuff
	CreateStatusBar(2);

	canvas = new WorldCanvas(this);
    canvas->SetScrollbars(4, 4, 300, 1700, 0, 0);
	canvas->SetBackgroundColour(*wxWHITE);
}

WorldFrame::~WorldFrame()
{
	delete sys;
	delete codes;
	delete detail;
}

#define MAX_BUTTONS		10
static char *short_msgs[] = {
    "Load",
    "Enter",
	"World",
	"Satillite",
	"Regenerate",
	"Edit",
	"Save",
	"Print",
	"About",
	"Exit" 
	};

static char *long_msgs[] = {
    "Load System",
    "Enter System",
	"Generate World Detail",
	"Generate Satillite Detail",
	"Regenerate Current",
	"Edit Current",
	"Save Current",
	"Print Current",
	"About World Generator",
	"Exit Program" 
	};

void
WorldFrame::InitButtons(wxToolBar *tb)
{
int i;
wxBitmap *bm[MAX_BUTTONS];
int currentX = DLG_OFFSET;
#ifdef __WXMSW__
  int width = 24;
#else
  int width = 16;
#endif

	// Set up toolbar
	tb->SetMargins(DLG_OFFSET, DLG_OFFSET);

#ifdef __WXMSW__
	bm[0] = new wxBitmap("icon1");
	bm[1] = new wxBitmap("icon2");
	bm[2] = new wxBitmap("icon3");
	bm[3] = new wxBitmap("icon4");
	bm[4] = new wxBitmap("icon5");
	bm[5] = new wxBitmap("icon6");
	bm[6] = new wxBitmap("icon7");
	bm[7] = new wxBitmap("icon8");
	bm[8] = new wxBitmap("icon9");
	bm[9] = new wxBitmap("icon10");
#else
	bm[0] = new wxBitmap(open_xpm);
	bm[1] = new wxBitmap(config_xpm);
	bm[2] = new wxBitmap(homeworld_xpm);
	bm[3] = new wxBitmap(sat_xpm);
	bm[4] = new wxBitmap(restart_xpm);
	bm[5] = new wxBitmap(tweak_world_xpm);
	bm[6] = new wxBitmap(save_xpm);
	bm[7] = new wxBitmap(print_xpm);
	bm[8] = new wxBitmap(help_xpm);
	bm[9] = new wxBitmap(exit_xpm);
#endif

	for(i = 0;i < MAX_BUTTONS;i++) {
		tb->AddTool(TB_OPEN + i, *(bm[i]), wxNullBitmap, FALSE,
			currentX, -1, NULL, 
			short_msgs[i], long_msgs[i]);
		currentX += width + DLG_OFFSET;
	}
		
	tb->Realize();

	// Can delete the bitmaps since they're reference counted
	for (i = 0; i < MAX_BUTTONS; i++)
		delete bm[i];
}

void
WorldFrame::Generate(int ndx)
{
BaseOrbit *bo;
StarOrbit *so;
DetailStar *ds;

	sys->SetCurrentOrbit(ndx);
	so = sys->GetPrimOrbit();
	ds = so->GetStar();

	bo = sys->GetCurrentOrbit();
	if(bo->GetOrbType() & ORBIT_IS_SAT) {
		WorldOrbit *wo;
		BaseSize *ws;

		wo = (WorldOrbit *)bo->GetParent();
		detail->AddWorldDetail(wo, ds->GetStarType());
		ws = detail->GetSize();

		detail->AddParent(ws->get_t_hex_val(),
				ws->orb->GetOrbit(),
				ds->GetMass(),
				ws->orb->GetEccen(),
				ws->GetMass(),
				ws->GetUWPSize(),
				ws->GetNumSatillites());

		detail->ClearDetail();
		detail->AddSatDetail((SatOrbit *)bo,
			ds->GetStarType());
	}
	else
		detail->AddWorldDetail(
			(WorldOrbit *)sys->GetIndexedOrbit(ndx),
			ds->GetStarType());

	canvas->Refresh();
}

void
WorldFrame::LoadSystem(char *s)
{
	wxBeginBusyCursor();
	sys->Clear();
	if(sys->LoadFile(s) > -1) {
		DetailStar *ds[3];
		PrimOrbit *po;
		CompOrbit *co;
		TertOrbit *to;

		po = sys->GetPrimOrbit();
		ds[0] = po->GetStar();
		detail->AddStar(DSI_PRIMARY, 
				ds[0]->GetStarType(), 
				ds[0]->GetStarClass(), 
				ds[0]->GetStarSize(), 
				-2.0, -2.0);
		if((co = po->GetComp(0)) != NULL) {
			ds[1] = co->GetStar();
			detail->AddStar(DSI_1ST_SECONDARY, 
					ds[1]->GetStarType(), 
					ds[1]->GetStarClass(), 
					ds[1]->GetStarSize(), 
					co->BaseOrbit::GetOrbit(), 
					ds[0]->GetMass());
			if((to = po->GetTert(0)) != NULL) {
				ds[2] = to->GetStar();
				detail->AddStar(DSI_1ST_TERTARY, 
						ds[2]->GetStarType(), 
						ds[2]->GetStarClass(), 
						ds[2]->GetStarSize(), 
						to->BaseOrbit::GetOrbit(), 
						ds[1]->GetMass());
			}
		}
		if((co = po->GetComp(1)) != NULL) {
			ds[1] = co->GetStar();
			detail->AddStar(DSI_2ND_SECONDARY, 
					ds[1]->GetStarType(), 
					ds[1]->GetStarClass(), 
					ds[1]->GetStarSize(), 
					co->BaseOrbit::GetOrbit(), 
					ds[0]->GetMass());
			if((to = po->GetTert(1)) != NULL) {
				ds[2] = to->GetStar();
				detail->AddStar(DSI_2ND_TERTARY, 
						ds[2]->GetStarType(), 
						ds[2]->GetStarClass(), 
						ds[2]->GetStarSize(), 
						to->BaseOrbit::GetOrbit(), 
						ds[1]->GetMass());
			}
		}

		canvas->Refresh();
	}
	wxEndBusyCursor();
}

// -------------------------------------------------------------------------
void
WorldFrame::OnToolEnter(wxCommandEvent& event)
{
int i=TB_OPEN;
bool fail=TRUE;

	while(i <= TB_EXIT) {
		if(event.GetSelection() == i) {
			char str[BIG_BUFFER_SIZE];

			sprintf(str, "%s", long_msgs[i - TB_OPEN]);
			SetStatusText((const wxString) str, 0);
			fail = FALSE;
			break;
		}
		i++;
	}
	if(fail)
		SetStatusText("", 0);
}

// -------------------------------------------------------------------------
void 
WorldFrame::Open(wxCommandEvent& event)
{
char *s;
wxString str;

	str = wxFileSelector("Load System File", NULL, NULL, NULL, 
			"*.sys", wxHIDE_READONLY, this);
	s = (char *)str.GetData();
   	if(s[0] != 0)
		LoadSystem(s);
}

void 
WorldFrame::GenWorld(wxCommandEvent& event)
{
// FIXME: check for file or input
int i;

	if(NULL == choice)
		choice = new ChoiceDialog(this);

	if(choice->GetWorldChoice(sys, &i)) {
		StarOrbit *so;
		DetailStar *ds;

		sys->SetCurrentOrbit(i);
		so = sys->GetPrimOrbit();
		ds = so->GetStar();
		detail->ClearPar();
		detail->AddWorldDetail(
				(WorldOrbit *)sys->GetIndexedOrbit(i),
				ds->GetStarType());
		canvas->Refresh();
	}
}

void 
WorldFrame::GenSat(wxCommandEvent& event)
{
// FIXME: check for file or input
int i;
WorldOrbit *wo;

	if(detail->GetSize() == NULL)
		wxMessageBox("Detail World First!", "Error",
			wxOK | wxCENTRE | wxICON_EXCLAMATION, this);
	else {
		wo = (WorldOrbit *)sys->GetCurrentOrbit();
		if(choice->GetSatChoice(wo, &i)) {
			StarOrbit *so;
			DetailStar *ds;

			so = sys->GetPrimOrbit();
			ds = so->GetStar();

			if(detail->GetPar() == NULL) {
				BaseSize *ws;

				ws = detail->GetSize();
				detail->AddParent(ws->get_t_hex_val(),
						ws->orb->GetOrbit(),
						ds->GetMass(),
						ws->orb->GetEccen(),
						ws->GetMass(),
						ws->GetUWPSize(),
						ws->GetNumSatillites());
			}
			detail->ClearDetail();
			detail->AddSatDetail(wo->GetSat(i),
				ds->GetStarType());
			canvas->Refresh();
		}
	}
}

void 
WorldFrame::Regen(wxCommandEvent& event)
{
	if(detail->GetSize() == NULL)
		wxMessageBox("Detail World First!", "Error",
			wxOK | wxCENTRE | wxICON_EXCLAMATION, this);
	else {
		StarOrbit *so;
		DetailStar *ds;

		so = sys->GetPrimOrbit();
		ds = so->GetStar();
		detail->Generate(sys->GetCurrentOrbit(), ds->GetStarType());
		canvas->Refresh();
	}
}

void 
WorldFrame::Tweak(wxCommandEvent& event)
{
BaseSize *bs;
Atmos *atm;

	bs = detail->GetSize();
	if((bs != NULL) && 
		((bs->GetType() == ST_WORLD) || bs->GetType() == ST_SAT)) {
		bool alb,grn;
		float rot,alb_adj,grn_adj;
		WorldSize *ws;

		ws = (WorldSize *)bs;
		rot = ws->GetRotPeriod();

		atm = detail->GetAtmos();
		alb = (bool)atm->GetAlbT();
		if(alb)
			alb_adj = (atm->GetAlbedo() - atm->GetAdjAlbedo()) /
					atm->GetAlbedo();
		else
			alb_adj = 0.0;
		grn = (bool)atm->GetGrnT();
		if(grn)
			grn_adj = 
				(atm->GetGreenhouse() - atm->GetAdjGreenhouse()) /
					atm->GetGreenhouse();
		else
			grn_adj = 0.0;

		if(NULL == tweak_dlg)
			tweak_dlg = new TweakDialog(this);

		if(tweak_dlg->GetValues(&rot, &alb, &alb_adj, 
					&grn, &grn_adj)) {
			ws->SetRotPeriod(rot);

			atm->AdjAdjAlb(alb, alb_adj);
			atm->AdjAdjGrn(grn, grn_adj);

			atm->RecalcTemp(rot);
			canvas->Refresh();
		}
	} else
		wxMessageBox("Detail World First!", "Error",
			wxOK | wxCENTRE | wxICON_EXCLAMATION, this);
}

void 
WorldFrame::Save(wxCommandEvent& event)
{
	if(detail->GetSize() == NULL)
		wxMessageBox("Detail World First!", "Error",
			wxOK | wxCENTRE | wxICON_EXCLAMATION, this);
	else {
		char *s=NULL;
		wxString str;

		str = wxFileSelector("Save Detail", 
				NULL, 
				NULL,
				"*.txt",
				"*.txt",
				wxSAVE | wxOVERWRITE_PROMPT | wxHIDE_READONLY,
				this);
		s = (char *)str.GetData();

		if(s[0] != 0) {
			FILE *fpx;

			if((fpx = fopen(s, "a+")) != NULL) {
				WorldDisplay *wd;

				wd = new WorldDisplay(fpx);
				wd->DrawDetail(detail);

				delete wd;
			}
		}
	}
}

void 
WorldFrame::Print(wxCommandEvent& event)
{
wxPrinter *printer;
PrintOut *print_out;

	print_out = new PrintOut(detail);

	printer = new wxPrinter();
	printer->Print(this, print_out, TRUE);
		
	delete printer;
	delete print_out;
}

void 
WorldFrame::Exit(wxCommandEvent& event)
{
	Close(TRUE);
}

void 
WorldFrame::Config(wxCommandEvent& event)
{
	NotImplemented("World Detail");
}

void 
WorldFrame::OnAbout(wxCommandEvent& event)
{
AboutDialog *dlg;

	dlg = new AboutDialog(this);
	dlg->ShowAbout("World Detail");
	delete dlg;
}

// ---------------------------------------------------------------
void
WorldFrame::Draw(wxDC *dc)
{
WorldDisplay *dd;

	dd = new WorldDisplay(dc);
	dd->DrawDetail(detail);
	delete dd;
}

// ---------------------------------------------------------------
// ---------------------------------------------------------------
void
WorldFrame::UpdateStatus(void)
{
	SetStatusText("");
}

// -------------------------------------------------------------------------
// =================================================================
// ===============================================================
WorldPanel::WorldPanel(wxWindow *frame, int x, int y, int w, int h) :
	wxPanel(frame, -1, wxPoint(x, y), wxSize(w, h))
{
}

WorldPanel::~WorldPanel()
{
}

// ===============================================================
WorldCanvas::WorldCanvas(wxWindow *frame,
						   int x, int y, int w, int h,
						   long style) :
    wxScrolledWindow(frame, -1, wxPoint(x, y), wxSize(w, h), style)
{
}

void
WorldCanvas::OnDraw(wxDC& dc)
{
	if(frame != NULL) {
		dc.Clear();
		frame->Draw(&dc);
	}
	else
	    dc.Clear();
}

// =================================================================
WorldDisplay::~WorldDisplay()
{
}

int
WorldDisplay::DrawDetail(Detail *d,
		char *msg1, char *msg2,
		int start_count, int total_count)
{
int ret=0;

	Start(msg1);

	ret += DrawSystem(d, ret, start_count, total_count);
	ret += DrawParent(d, ret, start_count, total_count);
	ret += DrawDetail(d, ret, start_count, total_count);

	Finish(msg2);
	return(ret);
}

int 
WorldDisplay::DrawSystem(Detail *d, 
		int cur_count, int start_count, int total_count)
{
int i;

	for(i = 0;i < MAX_STARS;i++) {
		StarInfo *si;

		if((si = d->GetStar(i)) != NULL) {
			char buff[120];

			DrawLine(&cur_count, start_count, total_count, 
					"                    Orbit: %s", 
							si->GetFormatedOrbit(buff));
			if((i != DSI_PRIMARY) && (si->orb->GetOrbit() > 0.0)) {
				DrawLine(&cur_count, start_count, total_count, 
					"                   Period: %s", 
							si->orb->FormatPeriod(buff));
				DrawLine(&cur_count, start_count, total_count, 
					"             Eccentricity: %.3f", si->orb->GetEccen());
			}
			DrawLine(&cur_count, start_count, total_count, 
					"                First Orb: %d", si->GetFirstOrbit());
			DrawLine(&cur_count, start_count, total_count, 
					"                Habitible: %d", si->GetHabZone());
			DrawLine(&cur_count, start_count, total_count, 
					"                    Class: %s", si->GetClassDesc(buff));
			DrawLine(&cur_count, start_count, total_count, 
					"                     Size: %s", si->GetSizeDesc(buff));
			DrawLine(&cur_count, start_count, total_count, 
					"                    Lumin: %f Sol", si->GetLumin());
			DrawLine(&cur_count, start_count, total_count, 
					"                     Temp: %f K", si->GetTemp());
			DrawLine(&cur_count, start_count, total_count, 
					"                   Radius: %f Sol", si->GetRad());
			DrawLine(&cur_count, start_count, total_count, 
					"                     Mass: %f Sol", si->GetMass());
			DrawLine(&cur_count, start_count, total_count, " ");
		}
	}

	return(cur_count);
}

int 
WorldDisplay::DrawParent(Detail *d, 
		int cur_count, int start_count, int total_count)
{
ParentSize *pa;

	if((pa = d->GetPar()) != NULL) {
		DrawLine(&cur_count, start_count, total_count, 
			"             Primary Mass: %.3f", pa->GetMass());
		DrawOrbit(pa->orb, &cur_count, start_count, total_count);
		DrawLine(&cur_count, start_count, total_count, " ");
	}

	return(cur_count);
}

int 
WorldDisplay::DrawDetail(Detail *d, 
		int cur_count, int start_count, int total_count)
{
int i;
float rot;
BaseSize *as;
Atmos *atm;
Hydro *hyd;
Resources *res;
WorldSize *ws;
SatSize *ss;
SocList *sl;
SocialData *sd;

	as = d->GetSize();
	if(as != NULL) {
		switch(as->GetType()) {
			case ST_GG: {
				DrawGGSize((GGSize *) as, &cur_count, start_count, total_count);
				rot = 0;
				break;
			}
			case ST_BELT: {
				DrawBeltSize((BeltSize *) as, &cur_count, 
							start_count, total_count);
				rot = 0;
				break;
			}
			case ST_WORLD: {
				DrawWorldSize((WorldSize *) as, &cur_count,
							start_count,total_count);
				ws = (WorldSize *) as;
				rot = ws->GetRotPeriod();
				break;
			}
			case ST_SAT: {
				DrawSatSize((SatSize *) as, &cur_count, 
							start_count, total_count);
				ss = (SatSize *) as;
				rot = ss->GetEffPeriod();
				break;
			}
			case ST_PARENT:
				break;
		}
		DrawLine(&cur_count, start_count, total_count, " ");
	}

	atm = d->GetAtmos();
	if(atm != NULL)
		DrawAtmos(atm, &cur_count, start_count, total_count);

	hyd = d->GetHydro();
	if(hyd != NULL)
		DrawHydro(hyd, &cur_count, start_count, total_count);

	if(atm != NULL)
		DrawTempW(atm, rot, &cur_count, start_count, total_count);

	res = d->GetResources();
	if(res != NULL)
		DrawResources(res, &cur_count, start_count, total_count);

// total pop?
	sl = d->GetSocList();
	if(sl->Number() > 1) {
		DrawLine(&cur_count, start_count, total_count, 
			"         Total Population: %.0f", sl->GetPop());
		DrawLine(&cur_count, start_count, total_count, 
			"               Government: 7; Balkanized");
		DrawLine(&cur_count, start_count, total_count, " ");
	}
	i=0;
	while((sd = d->GetSocial(i++)) != NULL)
		DrawSocial(sd, &cur_count, start_count, total_count);

	return(cur_count);
}

int 
WorldDisplay::DrawOrbit(OrbitInfo *orb, 
					int *cur_count, int start_count, int total_count)
{
char buff[120];

	DrawLine(cur_count, start_count, total_count, 
			"                    Orbit: %.3f(%.1fAU)",
				orb->GetOrbit(), orb->GetAUOrbit());
	DrawLine(cur_count, start_count, total_count, 
			"           Orbital Period: %s",
				orb->FormatPeriod(buff));
	DrawLine(cur_count, start_count, total_count, 
			"             Eccentricity: %.3f",
				orb->GetEccen());
	return(*cur_count);
}

int 
WorldDisplay::DrawGGSize(GGSize *gs, 
					int *cur_count, int start_count, int total_count)
{
	DrawOrbit(gs->orb, cur_count, start_count, total_count);
	DrawLine(cur_count, start_count, total_count, 
			"                  Density: %.3f", gs->GetDensity());
	DrawLine(cur_count, start_count, total_count, 
			"                     Mass: %.3f", gs->GetMass());
	DrawLine(cur_count, start_count, total_count, 
			"                 UWP Size: %d", gs->GetUWPSize());
	DrawLine(cur_count, start_count, total_count, 
			"     Number of Satillites: %d", gs->GetNumSatillites());

	return(*cur_count);
}

int 
WorldDisplay::DrawBeltSize(BeltSize *bs, 
					int *cur_count, int start_count, int total_count)
{
char buff[80];

	DrawOrbit(bs->orb, cur_count, start_count, total_count);
	DrawLine(cur_count, start_count, total_count, 
			"Predominate Body Diameter: %s", bs->FormatD(0, buff));
	if(bs->GetPredom() != bs->GetMax())
		DrawLine(cur_count, start_count, total_count, 
			"    Maximum Body Diameter: %s", bs->FormatD(1, buff));
	DrawLine(cur_count, start_count, total_count, 
			"              Nickle Zone: %d%%", bs->GetN());
	DrawLine(cur_count, start_count, total_count, 
			"               Mixed Zone: %d%%", bs->GetM());
	DrawLine(cur_count, start_count, total_count, 
			"    Carbonaceous/Ice Zone: %d%%", bs->GetC());
	DrawLine(cur_count, start_count, total_count, 
			"              Orbit Width: %.2fAU", bs->GetWidth());

	return(*cur_count);
}

int 
WorldDisplay::DrawGenSize(WorldSize *ws, 
					int *cur_count, int start_count, int total_count)
{
char buff[80];

	DrawLine(cur_count, start_count, total_count, 
			"                 Diameter: %ldKm", ws->GetDiameter());
	DrawLine(cur_count, start_count, total_count, 
			"                  Density: %.3f", ws->GetDensity());
	DrawLine(cur_count, start_count, total_count, 
			"                     Mass: %.3f", ws->GetMass());
	DrawLine(cur_count, start_count, total_count, 
			"                  Gravity: %.3f", ws->GetGravity());
	DrawLine(cur_count, start_count, total_count, 
			"                     Core: %s", ws->GetCore());
	DrawLine(cur_count, start_count, total_count, 
			"                     Tilt: %.3f degrees", ws->GetTilt());
	DrawLine(cur_count, start_count, total_count, 
			"                   Stress: %d", ws->GetStress());
	DrawLine(cur_count, start_count, total_count, 
			"          Rotation Period: %s", ws->FormatRotPeriod(buff));
	return(*cur_count);
}

int 
WorldDisplay::DrawWorldSize(WorldSize *ws, 
					int *cur_count, int start_count, int total_count)
{
	DrawOrbit(ws->orb, cur_count, start_count, total_count);
	DrawGenSize(ws, cur_count, start_count, total_count);
	DrawLine(cur_count, start_count, total_count, 
			"     Number of Satillites: %d", ws->GetNumSatillites());
	return(*cur_count);
}

int 
WorldDisplay::DrawSatSize(SatSize *ss, 
					int *cur_count, int start_count, int total_count)
{
char buff[80];


	DrawLine(cur_count, start_count, total_count, 
			"                    Orbit: %d(%.3fKm)\n", ss->GetSatOrbit(),
			ss->GetKmOrbit());
	DrawLine(cur_count, start_count, total_count, 
			"           Orbital Period: %s\n", ss->FormatSatPeriod(buff));

	DrawGenSize(ss, cur_count, start_count, total_count);
	DrawLine(cur_count, start_count, total_count, 
			"         Effective Period: %s", ss->FormatEffPeriod(buff));
	return(*cur_count);
}

int
WorldDisplay::DrawAtmos(Atmos *atm, 
				int *cur_count, int start_count, int total_count)
{
int j=0;
char buff[120];

	DrawLine(cur_count, start_count, total_count, 
			"               Atmosphere: %s",
			atm->get_full_desc(buff));
	DrawLine(cur_count, start_count, total_count, 
			"              Composition: %s", atm->GetAtmosCompStr(buff));

	if(atm->GetGasMask())
		DrawLine(cur_count, start_count, total_count, 
			"                    Gases: ", atm->GetGasStr(buff));

	DrawLine(cur_count, start_count, total_count, 
			"                 Pressure: %s %.3f",
			atm->GetTrueAtmosStr(), atm->GetPressure());

	DrawLine(cur_count, start_count, total_count, 
			"                   Albedo: %.3f", atm->GetAlbedo());
	DrawLine(cur_count, start_count, total_count, 
			"               Greenhouse: %.3f", atm->GetGreenhouse());

	if(atm->GetAlbedo() != atm->GetAdjAlbedo()) {
		j = 1;
		DrawLine(cur_count, start_count, total_count, 
				"      Terraforming Albedo: %.3f", atm->GetAdjAlbedo());
	}
	if(atm->GetGreenhouse() != atm->GetAdjGreenhouse()) {
		j = 1;
		DrawLine(cur_count, start_count, total_count, 
				"  Terraforming Greenhouse: %.3f", atm->GetAdjGreenhouse());
	}
	if(j) {
		DrawTemp(atm->GetBaseTemp(),FALSE, cur_count, start_count, total_count);
		DrawTemp(atm->GetTerrTemp(), TRUE, cur_count, start_count, total_count);
	} else
		DrawTemp(atm->GetBaseTemp(), TRUE, cur_count, start_count, total_count);

	DrawLine(cur_count, start_count, total_count, 
			" Orbital Eccentricity Mod: +/-%.3f C",
			atm->GetEccenMod());
	DrawLine(cur_count, start_count, total_count, 
			"     Axial Tilt Modifiers: %.3f C/%.3f C",
			atm->GetAxModPlus(), atm->GetAxModMinus());
	DrawLine(cur_count, start_count, total_count, 
			"       Latitude Modifiers: %d C per hex row",
			atm->GetLatMod());

	DrawTerr(cur_count, start_count, total_count, 
			"              Native Live: ", 
			atm->GetNative(), atm->GetNativeDM());

	DrawTerr(cur_count, start_count, total_count, 
			"  Atmosphere Terraforming: ",
			atm->GetAtmT(), atm->GetAtmTDM());

	DrawTerr(cur_count, start_count, total_count, 
			"  Greenhouse Terraforming: ",
			atm->GetGrnT(), atm->GetGrnTDM());

	DrawTerr(cur_count, start_count, total_count, 
			"      Albedo Terraforming: ",
			atm->GetAlbT(), atm->GetAlbTDM());

	DrawLine(cur_count, start_count, total_count, " ");

	return(*cur_count);
}

int 
WorldDisplay::DrawTerr(int *cur_count, int start_count, int total_count,
					char *msg, int terr, int dm)
{
char c;

	if(terr) c = 'Y';
	else c = 'N';
	DrawLine(cur_count, start_count, total_count, 
			"%s%c(%d)", msg, c, dm);

	return(*cur_count);
}

int 
WorldDisplay::DrawTemp(Temp *t, bool full_flag,
					int *cur_count, int start_count, int total_count)
{
static char *t_frmt[] = {
"               Luminosity: ",
"         Base Temperature: ",
"     Maximum Day Modifier: ",
"   Maximum Night Modifier: "
};

	if(t != NULL) {
		if(t->GetBaseTemp(0) == t->GetBaseTemp(1)) {
			if(full_flag)
				DrawLine(cur_count, start_count, total_count, 
					"%s%.3f", t_frmt[0],
					t->GetLumin(1));
			DrawLine(cur_count, start_count, total_count, 
					"%s%.3f C", t_frmt[1],
					t->GetBaseTemp(1) + 273.0);
			if(full_flag) {
				DrawLine(cur_count, start_count, total_count, 
					"%s%.3f C", t_frmt[2],
					t->GetDayRot(1));
				DrawLine(cur_count, start_count, total_count, 
					"%s%.3f C", t_frmt[3],
					-t->GetNightRot(1));
			}
		} else {
			if(full_flag)
				DrawLine(cur_count, start_count, total_count, 
					"%s%.3f/%.3f/%.3f", t_frmt[0],
					t->GetLumin(0), t->GetLumin(1), t->GetLumin(2));

			DrawLine(cur_count, start_count, total_count, 
					"%s%.3f C/%.3f C/%.3f C", t_frmt[1],
					t->GetBaseTemp(0) + 273.0, t->GetBaseTemp(1) + 273.0, 
					t->GetBaseTemp(2) + 273.0);

			if(full_flag) {
				DrawLine(cur_count, start_count, total_count, 
					"%s%.3f C/%.3f C/%.3f C", t_frmt[2],
					t->GetDayRot(0), t->GetDayRot(1), t->GetDayRot(2));
				DrawLine(cur_count, start_count, total_count, 
					"%s%.3f C/%.3f C/%.3f C", t_frmt[3],
					-t->GetNightRot(0), -t->GetNightRot(1), -t->GetNightRot(2));
			}
		}
	}
		
	return(*cur_count);
}

int
WorldDisplay::DrawTempW(Atmos *atm, float rot,
			int *cur_count, int start_count, int total_count)
{
int i,j,k;
char buff1[120];
char buff2[120];
static char *t_fmt[] = {
"---Winter---- ---Summer----",
"Base   Night  Day    Night  Day"
};

	if(atm->GetBaseTemp(0) == atm->GetBaseTemp(1))
		k = 1;
	else
		k = 0;

	if(k) {
		DrawLine(cur_count, start_count, total_count, 
			"              %s", t_fmt[0]);
		DrawLine(cur_count, start_count, total_count, 
			" Hex    %s", t_fmt[1]);
	} else {
		DrawLine(cur_count, start_count, total_count, 
			"              %s         %s", t_fmt[0], t_fmt[0]);
		DrawLine(cur_count, start_count, total_count, 
			" Hex    %s     %s",
			t_fmt[1], t_fmt[1]);
	}

	for(i = 0;i < 11;i++) {
		sprintf(buff1, "%d", i+1);
		strfixsize(buff1, 7);
		
		for(j = 0;j < 5;j++) {
			sprintf(buff2, "%.3f", atm->CalcTempAtRow(i, k, j, rot) + 273.0);
			strfixsize(buff2, 7);
			strcat(buff1, buff2);
		}

		if(!k) {
			strcat(buff1, " ");
			for(j = 0;j < 5;j++) {
				sprintf(buff2, "%.3f", 
					atm->CalcTempAtRow(i, 2, j, rot) + 273.0);
				strfixsize(buff2, 7);
				strcat(buff1, buff2);
			}
		}
		DrawLine(cur_count, start_count, total_count, buff1);
	}
	DrawLine(cur_count, start_count, total_count, " ");
		
	return(*cur_count);
}

int 
WorldDisplay::DrawHydro(Hydro *hyd, 
					int *cur_count, int start_count, int total_count)
{
char c;

	DrawLine(cur_count, start_count, total_count, 
			"  Hydrographic Percentage: %d%%", hyd->GetPerc());
	DrawLine(cur_count, start_count, total_count, 
			" Hydrographic Composition: %s", hyd->GetCompStr());
	DrawLine(cur_count, start_count, total_count, 
			"Number of Tectonic Plates: %d", hyd->GetPlates());
	DrawLine(cur_count, start_count, total_count, 
			"       Number of Volcanos: %d", hyd->GetVolc());
	if(hyd->GetPerc() > 50) {
		DrawLine(cur_count, start_count, total_count, 
			"    # of Major Continents: %d", hyd->GetMajorCont());
		DrawLine(cur_count, start_count, total_count, 
			"    # of Minor Continents: %d", hyd->GetMinorCont());
		DrawLine(cur_count, start_count, total_count, 
			"       # of Major Islands: %d", hyd->GetIslands());
		DrawLine(cur_count, start_count, total_count, 
			"       # of Archipelagoes: %d", hyd->GetArch());
	} else {
		DrawLine(cur_count, start_count, total_count, 
			"        # of Major Oceans: %d", hyd->GetMajorCont());
		DrawLine(cur_count, start_count, total_count, 
			"        # of Minor Oceans: %d", hyd->GetMinorCont());
		DrawLine(cur_count, start_count, total_count, 
			"          # of Small Seas: %d", hyd->GetIslands());
		DrawLine(cur_count, start_count, total_count, 
			"     # of Scattered Lakes: %d", hyd->GetArch());
	}

	DrawTerr(cur_count, start_count, total_count, 
			" Hydrosphere Terraforming: ",
			hyd->GetHydTer(), hyd->GetHydDM());

	DrawTerr(cur_count, start_count, total_count, 
			"     Terrian Terraforming: ",
			hyd->GetTerTer(), hyd->GetTerDM());

	if(hyd->GetWeatherControl()) c = 'Y';
	else c = 'N';
	DrawLine(cur_count, start_count, total_count, 
			"          Weather Control: %c", c);
	DrawLine(cur_count, start_count, total_count, " ");

	return(*cur_count);
}

int 
WorldDisplay::DrawResources(Resources *res, 
					int *cur_count, int start_count, int total_count)
{
char buff[120];
static char *t_fmt[] = {
	"        Natural Resources: ",
	"      Processed Resources: ",
	"    Manufactured Products: ",
	"     Information Products: ",
};

	DrawLine(cur_count, start_count, total_count, 
			"%s%s", t_fmt[0], res->GetNatResStr(buff));
	DrawLine(cur_count, start_count, total_count, 
			"%s%s", t_fmt[1], res->GetProResStr(buff));
	DrawLine(cur_count, start_count, total_count, 
			"%s%s", t_fmt[2], res->GetManResStr(buff));
	DrawLine(cur_count, start_count, total_count, 
			"%s%s", t_fmt[3], res->GetInfResStr(buff));
	DrawLine(cur_count, start_count, total_count, " ");

	return(*cur_count);
}

int 
WorldDisplay::DrawSocial(SocialData *sd, 
					int *cur_count, int start_count, int total_count)
{
Population *pop;

	// total pop
	pop = sd->GetTotPop();
	DrawLine(cur_count, start_count, total_count, 
		"         Total Population: %.0f", pop->GetPop());
	DrawLine(cur_count, start_count, total_count, 
		"         Rural Population: %.0f", sd->GetRuralPop());

	// cities
	DrawCities(sd, cur_count, start_count, total_count);

	// customs
	DrawCustoms(sd, cur_count, start_count, total_count);

	// outlook
	DrawLine(cur_count, start_count, total_count, 
			" Progressiveness Attitude: %s", sd->GetProgAttStr());
	DrawLine(cur_count, start_count, total_count, 
			"   Progressiveness Action: %s", sd->GetProgActStr());
	DrawLine(cur_count, start_count, total_count, 
			"  Aggressiveness Attitude: %s", sd->GetAggAttStr());
	DrawLine(cur_count, start_count, total_count, 
			"    Aggressiveness Action: %s", sd->GetAggActStr());
	DrawLine(cur_count, start_count, total_count, 
			"     Extensiveness Global: %s", sd->GetExtGlobStr());
	DrawLine(cur_count, start_count, total_count, 
			" Extensiveness Interstelr: %s", sd->GetExtIntStr());
	DrawLine(cur_count, start_count, total_count, " ");

	// govt
	DrawGovt(sd->GetGovt(), cur_count, start_count, total_count);

	// law
	DrawLaw(sd->GetLaw(), cur_count, start_count, total_count);

	// tech
	DrawTech(sd->GetTech(), cur_count, start_count, total_count);

	return(*cur_count);
}

int 
WorldDisplay::DrawCustoms(SocialData *sd, 
					int *cur_count, int start_count, int total_count)
{
int i=0;
char buff[120];
CustomData *cd;

	DrawLine(cur_count, start_count, total_count, "  Customs:");

	while((cd = sd->GetCustom(i++)) != NULL) {
		DrawLine(cur_count, start_count, total_count, "     %s", 
				cd->GetCustom(buff));
	}
	DrawLine(cur_count, start_count, total_count, " ");

	return(*cur_count);
}

int 
WorldDisplay::DrawCities(SocialData *sd, 
					int *cur_count, int start_count, int total_count)
{
char buff[120];
int i,j=0,k;
BaseCity *bc;

	for(i = 0;i < 4;i++) {
		k = j = 0;

		while((bc = sd->GetCity(j)) != NULL) {
			if(bc->GetCityType() == (CITY_TYPE) i) {
				if(k)
					DrawLine(cur_count, start_count, total_count, 
						"                           %s", 
						bc->GetCityFormat(buff));
				else
					DrawLine(cur_count, start_count, total_count, 
						"         %s: %s", bc->GetCityClass((CITY_TYPE) i),
						bc->GetCityFormat(buff));
				k++;
			}
			j++;
		}
	}
	DrawLine(cur_count, start_count, total_count, " ");

	return(*cur_count);
}

int 
WorldDisplay::DrawGovt(Govt *govt, 
					int *cur_count, int start_count, int total_count)
{
char buff[120];
int i;

	govt->get_short_desc(buff);
	DrawLine(cur_count, start_count, total_count, 
				"               Government: %s", buff);

	if(govt->get_t_hex_val()) {
		for(i = 0;i < govt->GetDivision();i++) {
			if(!i)
				DrawLine(cur_count, start_count, total_count, 
					"        Primary Authority: %s(%s)",
					govt->GetAuthStr(i), govt->GetOrgStr(i));
			else
				DrawLine(cur_count, start_count, total_count, 
					"          Other Authority: %s(%s)",
					govt->GetAuthStr(i), govt->GetOrgStr(i));
		}
	}
	DrawLine(cur_count, start_count, total_count, " ");

	return(*cur_count);
}

int 
WorldDisplay::DrawLaw(Law *law, 
					int *cur_count, int start_count, int total_count)
{
char buff[120];
int i;

	DrawLine(cur_count, start_count, total_count, 
			"   Law Profile Detail:");
	law->get_short_desc(buff);
	DrawLine(cur_count, start_count, total_count, 
			"                  Overall: %s", buff);

	for(i = 0;i < MAX_LAW;i++) {
		law_desc *ld;

		ld = law->GetLaw(i);
		ld->get_short_desc(buff);
		DrawLine(cur_count, start_count, total_count, 
			"         %s: %s", law->GetDesc(i), buff);
	}

	DrawLine(cur_count, start_count, total_count, 
			"               Uniformity: %s", law->GetUniformityStr());
	DrawLine(cur_count, start_count, total_count, " ");

	return(*cur_count);
}

int 
WorldDisplay::DrawTech(Tech *tech, 
					int *cur_count, int start_count, int total_count)
{
char buff[120];
int i;

	DrawLine(cur_count, start_count, total_count, 
			"   Technology Profile Detail:");
	for(i = 0;i < MAX_TECH;i++) {
		tech_desc *td;

		td = tech->GetTech(i);
		td->get_short_desc(buff);
		DrawLine(cur_count, start_count, total_count, 
				"     %s: %s", tech->GetDesc(i), buff);
	}
	DrawLine(cur_count, start_count, total_count, " ");

	return(*cur_count);
}


int 
WorldDisplay::DrawLine(int *cur_count, int start_count, int total_count,
					char *format...)
{
char buff[256];
va_list ap;

	if((*cur_count >= start_count) && (*cur_count < total_count)) {
		va_start(ap, format);
		vsprintf(buff, format, ap);
		Line(buff);
	}

	*cur_count = *cur_count + 1;
	return(1);
}

// =================================================================
#define LINES_PER_PAGE		90
//PrintOut::PrintOut(conf_struct *c)
PrintOut::PrintOut(Detail *d)
{
int i,line_count=0;
ParentSize *pa;
BaseSize *as;
Atmos *atm;
Hydro *hyd;
Resources *res;
SocList *sl;
SocialData *sd;

	page_max = 0;

	for(i = 0;i < MAX_STARS;i++) {
		StarInfo *si;

		if((si = d->GetStar(i)) != NULL) {
			line_count += 9;
			if(i) line_count += 2;
		}
	}
	
	if((pa = d->GetPar()) != NULL)
		line_count += 5;

	if((as = d->GetSize()) != NULL) {
		line_count += 4;
		if(as->GetType() == ST_GG)
			line_count += 4;
		else if(as->GetType() == ST_BELT) {
			BeltSize *bs;

			bs = (BeltSize *)as;
			line_count += 5;
			if(bs->GetPredom() != bs->GetMax())
				line_count++;
		} else {
			line_count += 8;
			if(as->GetType() == ST_WORLD)
				line_count++;
		}
	}

	if((atm = d->GetAtmos()) != NULL) {
		bool temp_2=FALSE;

		line_count += 32;
		if(atm->GetAlbedo() != atm->GetAdjAlbedo()) {
			line_count++;
			temp_2=TRUE;
		}
		if(atm->GetGreenhouse() != atm->GetAdjGreenhouse()) {
			line_count++;
			temp_2=TRUE;
		}
		if(temp_2) line_count++;
	}
		
	if((hyd = d->GetHydro()) != NULL)
		line_count += 12;

	if((res = d->GetResources()) != NULL)
		line_count += 5;

	sl = d->GetSocList();
	if(sl->Number() > 1)
		line_count += 3;
	i=0;
	while((sd = d->GetSocial(i++)) != NULL) {
		int j=0;
		BaseCity *bc;
		CustomData *cd;
		Govt *govt;

		line_count += 18 + MAX_LAW + MAX_TECH;
		while((bc = sd->GetCity(j++)) != NULL)
			line_count++;
		
		j = 0;
		while((cd = sd->GetCustom(j++)) != NULL)
			line_count++;

		govt = sd->GetGovt();
		if(govt->get_t_hex_val())
			line_count += govt->GetDivision();

	}

	page_max = line_count / LINES_PER_PAGE;

	if(line_count % LINES_PER_PAGE) page_max++;
}

bool
PrintOut::HasPage(int page)
{
	if(page <= page_max)
		return(TRUE);
	else
		return(FALSE);
}

void
PrintOut::GetPageInfo(int *minPage, int *maxPage,
						int *selPageFrom, int *selPageTo)
{
	*minPage = *selPageFrom = 1;
	*maxPage = *selPageTo = page_max;
}

bool 
PrintOut::OnPrintPage(int page)
{
wxDC *dc;
bool ret=FALSE;
char buff1[80];
char buff2[80];

	dc = GetDC();
	if((dc) && (page > 0)) {
		int line_count,cur_count;
		int w,h,a_h,a_w;
		float scale;
		WorldDisplay *wd;
		Detail *dtl;
		System *sys;
		BaseOrbit *bo;
		planet *p;

		dtl = frame->GetDetail();
		sys = frame->GetSys();
		bo = sys->GetCurrentOrbit();
		if(bo->GetOrbType() & ORBIT_IS_SAT) {
			SatOrbit *so;

			so = (SatOrbit *)bo;
			p = so->GetSat();
		} else {
			WorldOrbit *wo;

			wo = (WorldOrbit *)bo;
			p = wo->GetWorld();
		}

		// figure out how much there is to work with
		dc->GetSize(&w,&h);
		a_h = h - 55;
		a_w = w - 35;
		
		wd = new WorldDisplay(dc, 60, 20);

		buff1[0] = 0;
		
		sprintf(buff1, "World Detail For %s", p->GetName());
		strstrip(buff1, FALSE);
		if(sys->GetLoc() != NULL) {
			char buff[80];

			sprintf(buff, "; %s", sys->GetLoc());
			strcat(buff1, buff);
		}
		if(sys->GetSectName() != NULL) {
			char buff[80];

			sprintf(buff, " %s", sys->GetSectName());
			strcat(buff1, buff);
		}

		sprintf(buff2, "Page %d of %d", page, page_max);

		// fairly arbitrary, i would like about 100 lines......
		scale = (float)a_h / (float)(LINES_PER_PAGE * TEXT_Y_INCR);
		line_count = LINES_PER_PAGE - 4;
		dc->SetUserScale(scale, scale);

		cur_count = line_count * (page - 1);
		line_count = cur_count + (LINES_PER_PAGE - 4);
		wd->DrawDetail(dtl, buff1, buff2, cur_count, line_count);

		delete wd;
		ret = TRUE;
	}

	return(ret);
}

