/*
 * File:      trade_table.cpp
 * Purpose:	  tables to generate Traveller trade
 * Author:	
 * Created:	
 * Updated:	
 * Copyright: LGPL.
 *            Traveller is a registered trademark of Far Future Enterprises.	
 */

/* rcsid[] = "$RCSfile: trade_table.cpp,v $ $Revision: 1.2 $ $Author: man $ $Date: 2002/01/15 19:51:15 $" */

#include <stdlib.h>
#include <ctype.h>
#include "trade_table.h"
#include "string.h"
#include "str_util.h"
#include "parse.h"
#include "t_dice.h"

// define the file names
#define TRANSPORT_FILE	"transport.dat"
#define SPEC_TRADE_FILE	"spec_trade.dat"

// =============================================================================
// Speculative trade (book 2, pg 46)
SpecTradeEntry::SpecTradeEntry(char *line) :
	ListData()
{
char *ptrs[10];
GenericParse gp;

	gp.Parse(line, ptrs, 10);
	throw1 = atoi(ptrs[0]);
	throw2 = atoi(ptrs[1]);
	name = new char[strlen(ptrs[2]) + 1];
	strcpy(name, ptrs[2]);
	price = atol(ptrs[3]);
	purchase = new TradeDMs(ptrs[4]);
	resale = new TradeDMs(ptrs[5]);
	quantity = atoi(ptrs[6]);
	mult = atoi(ptrs[7]);
	if('0' == *ptrs[8]) {
		ind = FALSE;
	} else {
		ind = TRUE;
		tonnage = atoi(ptrs[8]);
	}

#ifdef SPEC_DEBUG_DUMP
	fprintf(stderr, "%d%d <%s>\n", throw1, throw2, name);
#endif
}

SpecTradeEntry::~SpecTradeEntry()
{
	if(name != NULL)
		delete name;
	if(purchase != NULL)
		delete purchase;
	if(resale != NULL)
		delete resale;
}

void
SpecTradeEntry::GetThrows(int *t1, int *t2)
{
	*t1 = throw1;
	*t2 = throw2;
}

void 
SpecTradeEntry::GenItem(char *world_codes, char *skills[], int levels[],
			char *item, int *quant, bool *indiv, int *value, int *ton)
{
int i,v,dm,mod,b_mod=0;
Dice d;

	strcpy(item, name);
	*quant = d.Roll(6, quantity) * mult;
	*indiv = ind;
	v = price;
	dm = purchase->GetDM(world_codes);
	i = 0;
	while((skills[i] != NULL) && (skills[i][0] != 0)) {
		if(strcasecmp(skills[i], "Broker") == 0) {
			dm += levels[i];
			b_mod = levels[i] * 5;
		} else if((strcasecmp(skills[i], "Admin") == 0) || 
				(strcasecmp(skills[i], "Bribery") == 0)) {
			dm += levels[i];
		} 
		i++;
	}

	// re-use i
	i = d.Roll(6, 2, dm);
	if(i < 3) mod = 40;
	else if(i < 4) mod = 50;
	else if(i < 11) mod = (i + 3) * 10;
	else if(i < 12) mod = 150;
	else if(i < 13) mod = 170;
	else if(i < 14) mod = 200;
	else if(i < 15) mod = 300;
	else mod = 400;

	*value = (price * (mod - b_mod)) / 100;
	
	if(ind)
		*ton = tonnage;
	else
		*ton = 0;
}

// -----------------------------------------------------------------------------
SpecTradeTable::SpecTradeTable(char *filename)
{
char buffer[BIG_BUFFER_SIZE];
int t1,t2;
FILE *fp;
SpecTradeEntry *ste;

	max_throw1=max_throw2=0;
	if((fp = fopen(filename, "r")) != NULL) {
		while(fgets(buffer, BIG_BUFFER_SIZE, fp) != NULL) {
			if('\n' == buffer[strlen(buffer) - 1]) 
				buffer[strlen(buffer) - 1] = 0;
			if(('#' == buffer[0]) || (strlen(buffer) == 0))
				continue;
			ste = new SpecTradeEntry(buffer);
			Append(ste);
			ste->GetThrows(&t1, &t2);
			if(t1 > max_throw1)
				max_throw1 = t1;
			if(t2 > max_throw2)
				max_throw2 = t2;
		}
	}
}

SpecTradeTable::~SpecTradeTable()
{
SpecTradeEntry *ste;
ListNode *n;

	n = First();
	while(n != NULL) {
		ste = (SpecTradeEntry *)n->Data();
		delete ste;
		DeleteNode(n);
		n = First();
	}
}

char *
SpecTradeTable::GetItem(int ndx)
{
SpecTradeEntry *ste;

	ste = (SpecTradeEntry *)NthData(ndx);
	if(ste != NULL)
		return(ste->GetName());
	else
		return(NULL);
}

void 
SpecTradeTable::GenItem(char *codes, char *sks[], int lvls[],
			char *item, int *quantity, bool *ind, int *value, int *tonnage)
{
int r1,r2,t1,t2;
SpecTradeEntry *ste;
ListNode *n;
Dice d;

	strcpy(item, "Oh-oh!");
	r1 = d.Roll(max_throw1, 1);
	r2 = d.Roll(max_throw2, 1);
	// XXX forgot about pop mod on first die.....
	n = First();
	while(n != NULL) {
		ste = (SpecTradeEntry *)n->Data();
		ste->GetThrows(&t1, &t2);
		if((t1 == r1) && (t2 == r2)) {
			// found it
			ste->GenItem(codes, sks, lvls, item, quantity, ind, value, tonnage);
			return;
		}
		n = Next();
	}
	// XXX
fprintf(stderr, "PROBLEM!!!!  Roll: %d %d\n", r1, r2);
}

// =============================================================================
TradeDMs::TradeDMs(char *dms)
{
char *ptrs[3];
int i=0;
GenericParse gp;

	gp.Parse(dms, ptrs, 3, ':');
	for(i = 0;i < 3;i++) {
		if(ptrs[i] != NULL) {
			code[i][0] = ptrs[i][0];
			code[i][1] = ptrs[i][1];
			dm[i] = atoi(&ptrs[i][2]);
		} else {
			code[i][0] = code[i][1] = code[i][2] = 0;
			dm[i] = 0;
		}
	}
}

TradeDMs::~TradeDMs()
{
}

int 
TradeDMs::GetDM(char *codes)
{
int i,ret=0;

	for(i = 0;i < 3;i++) {
		if((code[0] != 0) && (strstr(codes, code[i])))
			ret += dm[i];
	}
	
	return(ret);
}

// =============================================================================
CargoPass::CargoPass(char *n, FILE *fp)
{
char *ptrs[5],buffer[BIG_BUFFER_SIZE];
int data_count=0;
GenericParse gp;

	name = new char[strlen(n) + 1];
	strcpy(name, n);

	while(fgets(buffer, BIG_BUFFER_SIZE, fp) != NULL) {
		if('\n' == buffer[strlen(buffer) - 1]) 
			buffer[strlen(buffer) - 1] = 0;
		if(('#' == buffer[0]) || (strlen(buffer) == 0))
			continue;

		gp.Parse(buffer, ptrs, 5, ':');
		if(strcasecmp("Titles", ptrs[0]) == 0) {
			AllocString(names, &ptrs[1]);
		} else if(strcasecmp("Data", ptrs[0]) == 0) {
			if(data_count < 11) {
				AllocGen(data_count, &ptrs[2]);
				data_count++;
			} else {
				// error
			}
		} else if(strcasecmp("LoPop", ptrs[0]) == 0) {
			AllocInt(pop_thresh_lo, &ptrs[1]);
		} else if(strcasecmp("LoMod", ptrs[0]) == 0) {
			AllocInt(pop_dm_lo, &ptrs[1]);
		} else if(strcasecmp("HiPop", ptrs[0]) == 0) {
			AllocInt(pop_thresh_hi, &ptrs[1]);
		} else if(strcasecmp("HiMod", ptrs[0]) == 0) {
			AllocInt(pop_dm_hi, &ptrs[1]);
		} else if(strcasecmp("Red", ptrs[0]) == 0) {
			AllocInt(r_zone_dm, &ptrs[1]);
		} else if(strcasecmp("Amber", ptrs[0]) == 0) {
			AllocInt(a_zone_dm, &ptrs[1]);
		} else if(strcasecmp("Skills", ptrs[0]) == 0) {
			AllocString(skills, &ptrs[1]);
		} else if(strcasecmp("EndTable", ptrs[0]) == 0) {
			break;
		} else {
			// error
		}
	}

#ifdef DEBUG_DUMP
	{
	int i,j;

	fprintf(stderr, "=================================================\n");
	fprintf(stderr, "<%s>\n", name);
	for(i = 0;i < MAX_NUMS;i++) {
		fprintf(stderr, "%s ", names[i]);
	}
	fprintf(stderr, "\n");
	for(i = 0;i < 11;i++) {
		for(j = 0;j < MAX_NUMS;j++) {
			fprintf(stderr, "%dD(%d)(%d) ", num_gen[i][j].base,
				num_gen[i][j].d_minus, num_gen[i][j].num_minus);
		}
		fprintf(stderr, "\n");
	}
	for(i = 0;i < MAX_NUMS;i++) {
		fprintf(stderr, "pop[%d %d %d %d] ", pop_thresh_lo[i],
			pop_dm_lo[i], pop_thresh_hi[i], pop_dm_hi[i]);
	}
	fprintf(stderr, "\n");
	for(i = 0;i < MAX_NUMS;i++) {
		fprintf(stderr, "zone[%d %d] ", r_zone_dm[i],
			a_zone_dm[i]);
	}
	fprintf(stderr, "\n");
	for(i = 0;i < MAX_NUMS;i++) {
		fprintf(stderr, "<%s> ", skills[i]);
	}
	fprintf(stderr, "\n");

	}
#endif
}

CargoPass::~CargoPass()
{
int i;

	if(name != NULL)
		delete name;
	for(i = 0;i < MAX_NUMS;i++) {
		if(names[i] != NULL)
			delete names[i];
		if(skills[i] != NULL)
			delete skills[i];
	}
}

void
CargoPass::AllocString(char *dest[], char *src[])
{
int i;

	for(i = 0;i < MAX_NUMS;i++) {
		dest[i] = new char[strlen(src[i]) + 1];
		strcpy(dest[i], src[i]);
	}
}

void
CargoPass::AllocInt(int dest[], char *src[])
{
int i;

	for(i = 0;i < MAX_NUMS;i++)
		dest[i] = atoi(src[i]);
}

void
CargoPass::AllocGen(int ndx, char *src[])
{
int i,sign;

	for(i = 0;i < MAX_NUMS;i++) {
		num_gen[ndx][i].base = num_gen[ndx][i].d_minus = 
			num_gen[ndx][i].num_minus = 0;
		strstrip(src[i], TRUE);
		strstrip(src[i], FALSE);
		if(strlen(src[i]) < 2)
			continue;
		if(isdigit(src[i][0]))
			num_gen[ndx][i].base = src[i][0] - '0';
		if(0 == src[i][2])
			continue;
		if('-' == src[i][2])
			sign = -1;
		else
			sign = 1;
		if('D' == src[i][4]) {
			num_gen[ndx][i].d_minus = (src[i][3] - '0') * sign;
		} else {
			num_gen[ndx][i].num_minus = (src[i][3] - '0') * sign;
		}
	}
}

void
CargoPass::GetAvail(char *skill_list[], int *skill_levels, 
		char *source_uwp, char *dest_uwp, char d_zone, int *ret_num)
{
int s,d,i,j,tl_mod;
Dice dice;

	// figure tl_mod
	if(source_uwp[7] > 'A')
		s = source_uwp[7] - '0';
	else
		s = source_uwp[7] - 'A' -10;
	if(dest_uwp[7] > 'A')
		d = dest_uwp[7] - '0';
	else
		d = dest_uwp[7] - 'A' -10;
	tl_mod = s - d;
	// get pop's
	s = source_uwp[4] - '0';
	d = dest_uwp[4] - '0';
	if(s > 10) s = 10;
	if(d > 10) d = 10;
	for(i = 0;i < MAX_NUMS;i++) {
		ret_num[i] = 0;
		if(num_gen[s][i].base > 0) {
			ret_num[i] = dice.Roll(6, num_gen[s][i].base) + tl_mod;
			if(num_gen[s][i].d_minus > 0)
				ret_num[i] -= dice.Roll(6, num_gen[s][i].d_minus);
			if(num_gen[s][i].num_minus > 0)
				ret_num[i] -= num_gen[s][i].num_minus;
			if((pop_thresh_lo[i] >= 0)  && (d < pop_thresh_lo[i]))
				ret_num[i] += pop_dm_lo[i];
			if((pop_thresh_hi[i] >= 0)  && (d < pop_thresh_hi[i]))
				ret_num[i] += pop_dm_hi[i];
			if(('R' == d_zone) || ('r' == d_zone))
				ret_num[i] += r_zone_dm[i];
			if(('A' == d_zone) || ('a' == d_zone))
				ret_num[i] += a_zone_dm[i];
			j = 0;
			while(skill_list[j] != NULL) {
				if(strcmp(skill_list[j], skills[i]))
					ret_num[i] += skill_levels[i];
				j++;
			}
			if(ret_num[i] < 0)
				ret_num[i] = 0;
		}
	}
}

// =============================================================================
Trade::Trade(char *dir)
{
char buffer[BIG_BUFFER_SIZE],*ptrs[2];
FILE *fp;
GenericParse gp;

	sprintf(buffer, "%s%s", dir, SPEC_TRADE_FILE);
	spec = new SpecTradeTable(buffer);

	cargo = pass = NULL;
	sprintf(buffer, "%s%s", dir, TRANSPORT_FILE);
	if((fp = fopen(buffer, "r")) != NULL) {
		while(fgets(buffer, BIG_BUFFER_SIZE, fp) != NULL) {
			if('\n' == buffer[strlen(buffer) - 1]) 
				buffer[strlen(buffer) - 1] = 0;
			if(('#' == buffer[0]) || (strlen(buffer) == 0))
				continue;
			gp.Parse(buffer, ptrs, 2, ':');
			if(strcasecmp(ptrs[0], "Table") == 0) {
				if((strcasecmp(ptrs[1], "Passengers") == 0) &&
						(NULL == pass)) 
					pass = new CargoPass(ptrs[1], fp);
				else if((strcasecmp(ptrs[1], "Cargo") == 0) &&
						(NULL == cargo)) 
					cargo = new CargoPass(ptrs[1], fp);
				else {
					// XXX error!
				}
			}
		}
	}
}

Trade::~Trade()
{
	delete spec;
	if(cargo != NULL)
		delete cargo;
	if(pass != NULL)
		delete pass;
}

// =============================================================================
SpecTradeData::SpecTradeData(BaseSector *sect, int x, int y, 
				char *sks[], int lvls[], Trade *trade, bool b7) :
	ListData()
{
char buffer[BIG_BUFFER_SIZE],codes[20];
HexData *hd;
SpecTradeTable *spec;

	hd = sect->GetHex(x, y);
	sect->GetCodeStr(hd->world, codes);
	if(b7) {
		ind = FALSE;
		tonnage = 0;
		value = 4000;
		if(strcmp(codes, "Ag") == 0) 
			value -= 1000;
		if(strcmp(codes, "As") == 0) 
					value -= 1000;
		if(strcmp(codes, "Hi") == 0) 
			value -= 1000;
		if(strcmp(codes, "In") == 0) 
			value -= 1000;
		if(strcmp(codes, "Po") == 0)
			value -= 1000;
		if(strcmp(codes, "Ba") == 0)
			value += 1000;
		if(strcmp(codes, "De") == 0)
			value += 1000;
		if(strcmp(codes, "Fl") == 0)
			value += 1000;
		if(strcmp(codes, "Lo") == 0)
			value += 1000;
		if(strcmp(codes, "Ni") == 0)
			value += 1000;
		if(strcmp(codes, "Ri") == 0)
			value += 1000;
		if(strcmp(codes, "Va") == 0)
			value += 1000;
		value += (hd->world->get_tech_val() * 100);
		switch(hd->world->get_port()) {
			case 'A':
				value -= 1000;
				break;
			case 'C':
				value += 1000;
				break;
			case 'D':
				value += 2000;
				break;
			case 'E':
				value += 3000;
				break;
			case 'X':
				value += 5000;
				break;
		}
		strstrip(codes, TRUE);
		strstrip(codes, FALSE);
		item = new char[strlen(codes) + 20];
		sprintf(item, "%c-%c %s Cr%d", hd->world->get_port(), 
			hd->world->get_tech(), codes, value);
		quantity = 0; 		// XXX
	} else {
		spec = trade->GetSpecTrade();
		spec->GenItem(codes, sks, lvls, buffer, &quantity, &ind, 
				&value, &tonnage);
		item = new char[strlen(buffer) + 1];
		strcpy(item, buffer);
	}
}

SpecTradeData::~SpecTradeData()
{
	if(item != NULL)
		delete item;
}

// =============================================================================
AvailList::AvailList(BaseSector *sect, int x, int y, 
				char *sks[], int lvls[], Trade *trade, bool b7, bool enhanced)
{
char buff1[12],buff2[12];
int car[3],pass[3],i,j,k,l,x1,x2,y1,y2;
HexData *src,*dest;
CargoPass *c,*p;
TradeAvail *ta;
SpecTradeData *st;
Dice d;

	spec_trade = new LinkedList();
	trade_avail = new LinkedList();

	// cargo & passengers
	c = trade->GetCargo();
	p = trade->GetPass();
	src = sect->GetHex(x, y);
	for(i = 0;i < 6;i++) {						// direction
		for(j = 1;j < 7;j++) {					// mag
			if((sect->TranslateCoord(x, y, i, j, &x1, &y1)) &&
						((dest = sect->GetHex(x1, y1)) != NULL) &&
						(dest->world != NULL)) {
				c->GetAvail(sks, lvls, src->world->GetUWPStr(buff1),
					dest->world->GetUWPStr(buff2), dest->world->GetZone(), car);
				p->GetAvail(sks, lvls, src->world->GetUWPStr(buff1),
					dest->world->GetUWPStr(buff2), dest->world->GetZone(), pass);

				ta = new TradeAvail(pass, car, x1, y1, j);
				trade_avail->Append(ta);

				l = i + 2;
				if(l > 5) l -= 5;
				for(k = 1;k < j;k++) {		// back fill
					if((sect->TranslateCoord(x1, y1, l, k, &x2, &y2)) &&
						((dest = sect->GetHex(x2, y2)) != NULL) &&
						(dest->world != NULL)) {
						c->GetAvail(sks, lvls, src->world->GetUWPStr(buff1),
							dest->world->GetUWPStr(buff2), 
							dest->world->GetZone(), car);
						p->GetAvail(sks, lvls, src->world->GetUWPStr(buff1),
							dest->world->GetUWPStr(buff2), 
							dest->world->GetZone(), pass);

						ta = new TradeAvail(pass, car, x2, y2, j);
						trade_avail->Append(ta);
					}
				}
			}
		}
	}

	// spec. trade
	if(enhanced) {
		j = d.Roll(6, 1) - 4;
		for(i = 0;i < MAX_TRADE_SKILLS;i++) {
			if(NULL == sks[i])
				continue;
			if(strcmp(sks[i], "Trader") == 0)
				j += lvls[i];
			else if(strcmp(sks[i], "Streetwise") == 0)
				j += lvls[i] / 2;
		}
		if(src->world->get_pop_val() < 3)
			j -= 2;
		else if(src->world->get_pop_val() < 6)
			j--;
		if(src->world->get_pop_val() > 9)
			j += 2;
		else if(src->world->get_pop_val() > 7)
			j++;
		if(src->world->get_port() == 'A')
			j += 2;
		else if(src->world->get_port() == 'B')
			j++;
	} else
		j = 1;

	for(i = 0;i < j;i++) {
		st = new SpecTradeData(sect, x, y, sks, lvls, trade, b7);
		spec_trade->Append(st);
	}
}

AvailList::~AvailList()
{
	Clear();
	delete spec_trade;
	delete trade_avail;
}

void
AvailList::Clear(void)
{
TradeAvail *ta;
SpecTradeData *st;
ListNode *n;

	n = spec_trade->First();
	while(n != NULL) {
		st = (SpecTradeData *)n->Data();
		delete st;
		spec_trade->DeleteNode(n);
		n = spec_trade->First();
	}

	n = trade_avail->First();
	while(n != NULL) {
		ta = (TradeAvail *)n->Data();
		delete ta;
		trade_avail->DeleteNode(n);
		n = trade_avail->First();
	}
}

SpecTradeData *
AvailList::GetSpecTrade(int ndx)
{
ListNode *n;

	n = spec_trade->Nth(ndx);
	if(n != NULL)
		return((SpecTradeData *)n->Data());

	return(NULL);
}

TradeAvail *
AvailList::GetTradeAvail(int ndx)
{
ListNode *n;

	n = trade_avail->Nth(ndx);
	if(n != NULL)
		return((TradeAvail *)n->Data());

	return(NULL);
}

// =============================================================================
TradeAvail::TradeAvail(int p[], int c[], int l_x, int l_y, int l_d) :
	ListData()
{
	p1 = p[0];
	p2 = p[1];
	p3 = p[2];
	c1 = c[0];
	c2 = c[1];
	c3 = c[2];
	x = l_x;
	y = l_y;
	z = 0;
	dist = l_d;
}

TradeAvail::~TradeAvail()
{
}

void 
TradeAvail::GetAvailCargo(int c[])
{
	c[0] = c1;
	c[1] = c2;
	c[2] = c3;
}

void 
TradeAvail::GetAvailPass(int p[])
{
	p[0] = p1;
	p[1] = p2;
	p[2] = p3;
}

void 
TradeAvail::GetLoc(int *x1, int *y1)
{
	*x1 = x;
	*y1 = y;
}

// =============================================================================

