/*
 * File:	system.cpp
 * Purpose:	
 * Author:	
 * Created:	
 * Updated:	
 * Copyright: LGPL.
 *            Traveller is a registered trademark of Far Future Enterprises.	
 */

/* rcsid[] = "$RCSfile: system.cpp,v $ $Revision: 1.18 $ $Author: man $ $Date: 2003/05/04 15:12:39 $" */

// For compilers that support precompilation, includes "wx.h".
#include "wx/wxprec.h"

#ifndef WX_PRECOMP
#include "wx/wx.h"
#endif

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include <sys/stat.h>
#include <unistd.h>
#include "system.h"
#include "str_util.h"
#include "t_res.h"
#include "t_help.h"

#if defined(__WXGTK__)
// general icon
#include "system.xpm"
// menu/toolbar icons
#include "bitmaps/config.xpm"
#include "bitmaps/critter.xpm"
#include "bitmaps/exit.xpm"
#include "bitmaps/help.xpm"
#include "bitmaps/mag.xpm"
#include "bitmaps/mag_a.xpm"
#include "bitmaps/mag_m.xpm"
#include "bitmaps/mag_o.xpm"
#include "bitmaps/mag_p.xpm"
#include "bitmaps/open.xpm"
#include "bitmaps/orb.xpm"
#include "bitmaps/orb_a.xpm"
#include "bitmaps/orb_d.xpm"
#include "bitmaps/orb_m.xpm"
#include "bitmaps/print.xpm"
#include "bitmaps/save.xpm"
// tree icons
#include "bitmaps/aa_world.xpm"
#include "bitmaps/h0a0_world.xpm"
#include "bitmaps/h0a4_world.xpm"
#include "bitmaps/h4a0_world.xpm"
#include "bitmaps/h4a4_world.xpm"
#include "bitmaps/h8_world.xpm"
#include "bitmaps/lgg.xpm"
#include "bitmaps/s0_world.xpm"
#include "bitmaps/sgg.xpm"
#include "bitmaps/star_temp.xpm"
#endif

IMPLEMENT_APP(SystemApp)

// misc defines
#define BORDER_STYLE		wxSP_NOBORDER

#define FRAME_WIDTH             600
#define FRAME_HEIGHT            400

#define MINIMUM_WINDOW_SIZE		10

#define MAP_WIDTH               800
#define MAP_HEIGHT              600

#define MAP_SCROLL_STEPS         4
#define MAP_SCROLL_PIX_H         20
#define MAP_SCROLL_PIX_V         20
#define MAP_SCROLL_LEN_H         (MAP_WIDTH / MAP_SCROLL_PIX_H)
#define MAP_SCROLL_LEN_V         (MAP_HEIGHT / MAP_SCROLL_PIX_V)

#define LARGEST_ORBIT           293
#define SMALLEST_ORBIT          25
#define ORBIT_WIDTH             1
#define ORBIT_WIDTH_BELT        1

#define STAR_SPIKE             2
#define MAIN_STAR_RAD          4
#define SECOND_STAR_RAD        2
#define WORLD_RAD              3
#define SMALL_GG_RAD           5
#define LARGE_GG_RAD           6

#define MAP_X_OFFSET            (((MAP_WIDTH - (LARGEST_ORBIT * 2)) / 2) + 15)
#define MAP_Y_OFFSET            ((MAP_HEIGHT - (LARGEST_ORBIT * 2)) /2)

// these are not used the same:
#define MAP_TEXT_OFFSET_Y       14
#define MAP_TEXT_OFFSET_Y1      (MAP_HEIGHT - 36)
#define MAP_TEXT_OFFSET_Y2      (MAP_HEIGHT - 16)
#define MAP_TEXT_OFFSET_X       (MAP_X_OFFSET + 10)
#define MAP_TEXT_OFFSET_X1      5
#define MAP_TEXT_OFFSET_X2      ((LARGEST_ORBIT * 2) + MAP_X_OFFSET - 95)

#define SCALE_HEIGHT            16
#define SCALE_LENGTH            152
#define BOX_HEIGHT              (SCALE_HEIGHT / 2)
#define BOX_LENGTH              (SCALE_LENGTH / 4)

#define MAX_FILES				4

#define BITMAP_SIZE             16

#define TRIG_TABLE_SIZE         13

#define BIG_BUFFER_SIZE        120

#define LINE_STAR_RETURN       10
#define LINE_GG_RETURN         3
#define LINE_OTHER_RETURN      20

#define MAX_BUTTONS		16

// ID for the menu quit command
enum {
	ID_TOOLBAR = 500,
	TREE_CTRL,
	SYSTEM_OPEN,
	SYSTEM_SAVE,
	SYSTEM_PRINT,
	SYSTEM_HELP,
	SYSTEM_EXIT,
	SYSTEM_FILE1,
	SYSTEM_FILE2,
	SYSTEM_FILE3,
	SYSTEM_FILE4,
	SYSTEM_FILE5,
	SYSTEM_EDIT_SYSTEM,
	SYSTEM_EDIT_ORBIT,
	SYSTEM_EDIT_MOVE,
	SYSTEM_EDIT_ADD,
	SYSTEM_EDIT_DEL,
	SYSTEM_EDIT_DETAIL,
	SYSTEM_EDIT_ENC,
	SYSTEM_ZOOM_IN,
	SYSTEM_ZOOM_OUT,
	SYSTEM_ZOOM_SELECT,
	SYSTEM_ZOOM_ORBIT,
};

#define ICON_HEIGHT		13
#define ICON_WIDTH		ICON_HEIGHT
// -----------------------------------------------------------------------
enum {
	icon_dummy=0,
	icon_large_gg,
	icon_small_gg,
	icon_size0_world,
	icon_atmosa_world,
	icon_hydroa_world,
	icon_h4a4_world,
	icon_h4a0_world,
	icon_h0a4_world,
	icon_h0a0_world,
	icon_star0,
	icon_star1,
	icon_star2,
	icon_star3,
   	icon_star4
};

// -----------------------------------------------------------------------
// global fonts
wxFont  *textBigFont;
wxFont  *textHeaderFont;
wxFont  *textWindowFont;
wxFont  *NormalFont;
wxFont  *UnderlineFont;
wxFont  *BoldFont;
wxFont  *BUFont;
wxFont  *mapBigFont;
wxFont  *mapHeaderFont;
wxFont  *mapWindowFont;

// -----------------------------------------------------------------------
// trig table -- for drawing worlds on an orbit
static float x_table[TRIG_TABLE_SIZE], y_table[TRIG_TABLE_SIZE];

// -----------------------------------------------------------------------
// menu/button struct
static struct MENU_STRUCT {
	int id;
	char	*desc;
	char 	*short_help;
	char	*long_help;
} menu_struct[] = {
	{ SYSTEM_OPEN, "&Open", "Open", "Open a system file" },
	{ SYSTEM_SAVE, "&Save", "Save", "Save system file" },
	{ SYSTEM_PRINT, "&Print", "Print", "Print system" },
	{ SYSTEM_HELP, "&About", "About", "Get Some Help" },
	{ SYSTEM_EXIT, "E&xit", "Exit", "Exit program" },
	{ SYSTEM_EDIT_SYSTEM, "&System", "Edit Sys.", "Edit System Parameters" },
	{ SYSTEM_EDIT_ORBIT, "&Orbit", "Edit Orb.", "Edit Current Orbit" },
	{ SYSTEM_EDIT_MOVE, "&Move", "Move", "Move Current Orbit" },
	{ SYSTEM_EDIT_ADD, "&Add", "Add", "Add Orbit to Current Orbit" },
	{ SYSTEM_EDIT_DEL, "De&lete", "Delete", "Delete Current Orbit" },
	{ SYSTEM_EDIT_DETAIL, "&Detail", "Detail", "Generate Detail For Current Orbit" },
	{ SYSTEM_EDIT_ENC, "&Encounter", "Enc.", "Generate Encounter Tables For Current Orbit" },
	{ SYSTEM_ZOOM_IN, "Zoom &In", "Zoom In", "Zoom In" },
	{ SYSTEM_ZOOM_OUT, "Zoom &Out", "Zoom Out", "Zoom Out" },
	{ SYSTEM_ZOOM_SELECT, "Zoom &Selection", "Zoom Sel.", "Zoom Current Orbit As Inner Orbit" },
	{ SYSTEM_ZOOM_ORBIT, "&Zoom Orbit", "Zoom Orbit", "Zoom Current Orbit" },
	{ -1, "", "", "" }
};

// -----------------------------------------------------------------------
// seem to need these
SystemFrame *frame=NULL;
TempFrame *t_frame=NULL;
SystemApp *app=NULL;

// ======================================================================
bool 
SystemApp::OnInit(void)
{
bool normal=TRUE;
int i;
char *ptr=NULL;
struct stat stat_buff;

	app = this;

	// NOTE NOTE NOTE NOTE NOTE NOTE
	// check argc and argv:
	//  No args or one arg means SystemFrame should be created.
	//  More than one arg means further investigation needs to be done.
	//  In general, extra args are in case sysgen needs to be run and
	//  will be passed on.
	//  Further behavior depends on the following conditions:
	//  1) the given filename does not exist 
	//     - if it does exist, go ahead and create SystemFrame.
	//  2) Create TempFrame to see if the user says it's ok to run sysgen.  
	//     - if not ok to run sysgen, just exit.
	//     - if sysgen should be run, prompt for a language file, and 
	//       whether to run it interactive.  Exec sysgen with '-E' and exit.
	if(2 == argc)
		ptr = argv[1];
	else if(argc > 2) {
		for(i = 1;i < (argc);i++) {
			// find the filename
			if(strncmp(argv[i], "-o", 2) == 0) {
				if(argv[i][2] != 0)
					ptr = &argv[i][2];
				else if(i < (argc))
					ptr = argv[i+1];
				break;
			}
		}
		// no filename at this point is an error
		if(NULL == ptr) {
			fprintf(stderr, "%s Usage:\n", argv[0]);
			fprintf(stderr, "    %s <filename>\n        -- or --\n",
				argv[0]);
			fprintf(stderr, "args given for the sysgen program.\n");
			exit(-1);
		}
		// stat the filenme
		// XXX is this too unix specific?
		if(stat(ptr, &stat_buff) < 0)
			normal = FALSE;
	}

	if(normal) {
    	for(i = 0;i < TRIG_TABLE_SIZE; i++) {
			double ang;
			ang = ((3 * 3.1415927) / 2) - ((3.1415927 / 12) * i);
			x_table[i] = cos(ang);
			y_table[i] = sin(ang);
		}
	
		// Create fonts
		textBigFont = new wxFont(18, wxMODERN, wxNORMAL, wxBOLD, TRUE);
    	textHeaderFont = new wxFont(14, wxMODERN, wxNORMAL, wxBOLD, TRUE);
    	textWindowFont = new wxFont(11, wxMODERN, wxNORMAL, wxNORMAL);
		mapBigFont = new wxFont(18, wxMODERN, wxNORMAL, wxBOLD);
    	mapHeaderFont = new wxFont(14, wxMODERN, wxNORMAL, wxBOLD);
    	mapWindowFont = new wxFont(13, wxMODERN, wxNORMAL, wxNORMAL);
    	UnderlineFont = new wxFont(11, wxMODERN, wxNORMAL, wxNORMAL, TRUE);
    	NormalFont = new wxFont(11, wxSWISS, wxNORMAL, wxNORMAL);
    	BoldFont = new wxFont(12, wxSWISS, wxNORMAL, wxBOLD);
    	BUFont = new wxFont(12, wxMODERN, wxNORMAL, wxBOLD, TRUE);



		frame = new SystemFrame(
			NULL,
			"System",
			-1, -1,
			FRAME_WIDTH,
			FRAME_HEIGHT
			);

		frame->CreateToolBar(wxNO_BORDER | wxHORIZONTAL | wxTB_DOCKABLE, 
			ID_TOOLBAR);
		frame->InitButtons(frame->GetToolBar());
		frame->SetMenuMode(FALSE);

		// load?
		if(ptr != NULL) 
			frame->Load(ptr);

		// Show the frame
		frame->Show(TRUE);
		SetTopWindow(frame);
	} else {
		t_frame = new TempFrame(ptr);
		t_frame->Show(TRUE);
		SetTopWindow(t_frame);
	}
  
	// Return the main frame window
	return(TRUE);
}

int
SystemApp::OnExit(void)
{
wxWindow *w;

	while((w = GetTopWindow()) != NULL)
		delete w;
	return(0);
}

// ======================================================================
BEGIN_EVENT_TABLE(SystemFrame, wxFrame)
	EVT_MENU(SYSTEM_FILE1, SystemFrame::LoadFile1)
	EVT_MENU(SYSTEM_FILE2, SystemFrame::LoadFile2)
	EVT_MENU(SYSTEM_FILE3, SystemFrame::LoadFile3)
	EVT_MENU(SYSTEM_FILE4, SystemFrame::LoadFile4)
	EVT_MENU(SYSTEM_FILE5, SystemFrame::LoadFile5)
	EVT_MENU(SYSTEM_OPEN, SystemFrame::LoadFileCommand)
	EVT_MENU(SYSTEM_HELP, SystemFrame::OnAbout)
	EVT_MENU(SYSTEM_SAVE, SystemFrame::SaveSystem)
	EVT_MENU(SYSTEM_PRINT, SystemFrame::PrintSystem)
	EVT_MENU(SYSTEM_EXIT, SystemFrame::Quit)
	EVT_MENU(SYSTEM_EDIT_SYSTEM, SystemFrame::EditSystem)
	EVT_MENU(SYSTEM_EDIT_ORBIT, SystemFrame::EditOrbit)
	EVT_MENU(SYSTEM_EDIT_MOVE, SystemFrame::MoveOrbit)
	EVT_MENU(SYSTEM_EDIT_ADD, SystemFrame::AddOrbit)
	EVT_MENU(SYSTEM_EDIT_DEL, SystemFrame::DeleteOrbit)
	EVT_MENU(SYSTEM_EDIT_DETAIL, SystemFrame::DetailOrbit)
	EVT_MENU(SYSTEM_EDIT_ENC, SystemFrame::EncOrbit)
	EVT_MENU(SYSTEM_ZOOM_IN, SystemFrame::ZoomIn)
	EVT_MENU(SYSTEM_ZOOM_OUT, SystemFrame::ZoomOut)
	EVT_MENU(SYSTEM_ZOOM_SELECT, SystemFrame::ZoomSelect)
	EVT_MENU(SYSTEM_ZOOM_ORBIT, SystemFrame::ZoomOrbit)
	EVT_CLOSE(SystemFrame::OnCloseWindow)
	EVT_TOOL_ENTER(ID_TOOLBAR, SystemFrame::OnToolEnter)
END_EVENT_TABLE()

// ----------------------------------------------------------------------
// System frame constructor
SystemFrame::SystemFrame(wxFrame* frame, char* title,
						 int x, int y, int w, int h):
	wxFrame(frame, -1, title, wxPoint(x, y), wxSize(w, h))
{
int i,j;
wxString ptr;
char str[MAX_FILE_LENGTH];
wxConfig *config;

    map_center = 0;
    map_first = 0;

	sys_dlg = NULL;
	star_dlg = NULL;
	gg_dlg = NULL;
	world_dlg = NULL;
	move_dlg = NULL;
	add_dlg = NULL;
	print_dlg = NULL;

	dirs = new MTUFile();
	config = dirs->GetConfig();

	// set the icon
#ifdef __WINDOWS__
	SetIcon(wxIcon("system_icn"));
#endif
#ifdef __WXGTK__
	SetIcon(wxIcon(system_xpm));
#endif

	// set up a menubar
    //  file
	file_menu = new wxMenu;
	AddToMenu(file_menu, SYSTEM_OPEN);
	AddToMenu(file_menu, SYSTEM_SAVE);
	file_menu->AppendSeparator();
	AddToMenu(file_menu, SYSTEM_PRINT);
	file_menu->AppendSeparator();
	AddToMenu(file_menu, SYSTEM_EXIT);

	j = 0;
	for(i = 0;i < MAX_FILES;i++) {
		ptr = "";
		sprintf(str, "/%s/%s%d", SECTION_HISTORY, ENTRY_SYST_FILES, i+1);
		if(!config->Read((const wxString) str, &ptr))
			break;

		if(j == 0)
			file_menu->AppendSeparator();
		file_menu->Append(SYSTEM_FILE1+i, ptr,
					"Load Previous File");
		j++;
	}

    //  edit
	edit_menu = new wxMenu;
	AddToMenu(edit_menu, SYSTEM_EDIT_SYSTEM);
	AddToMenu(edit_menu, SYSTEM_EDIT_ORBIT);
	AddToMenu(edit_menu, SYSTEM_EDIT_MOVE);
	AddToMenu(edit_menu, SYSTEM_EDIT_ADD);
	AddToMenu(edit_menu, SYSTEM_EDIT_DEL);
	edit_menu->AppendSeparator();
	AddToMenu(edit_menu, SYSTEM_EDIT_DETAIL);
	AddToMenu(edit_menu, SYSTEM_EDIT_ENC);

    //  map
	map_menu = new wxMenu;
	AddToMenu(map_menu, SYSTEM_ZOOM_IN);
	AddToMenu(map_menu, SYSTEM_ZOOM_OUT);
	AddToMenu(map_menu, SYSTEM_ZOOM_SELECT);
	AddToMenu(map_menu, SYSTEM_ZOOM_ORBIT);

	//  help
	wxMenu *help_menu = new wxMenu;
	AddToMenu(help_menu, SYSTEM_HELP);

	menu_bar = new wxMenuBar(wxMB_DOCKABLE);
	menu_bar->Append(file_menu, "&File");
	menu_bar->Append(edit_menu, "&Edit");
	menu_bar->Append(map_menu, "&Map");
	menu_bar->Append(help_menu, "&Help");

	SetMenuBar(menu_bar);
	
	// create the 2 splitters
    splitter1 = new wxSplitterWindow(this, -1,
			wxPoint(0, 0), wxSize(400, 400),
            BORDER_STYLE);

    splitter2 = new wxSplitterWindow(splitter1, -1,
			wxPoint(0, 0), wxSize(400, 400),
            BORDER_STYLE);

	// create the canvases
	list = new ListCanvas(splitter1, textWindowFont, 5);

	detail = new SystemDetail(splitter2);
	detail->SetBackgroundColour(*wxWHITE);
	detail->SetScrollbars(18, 18, 8, 24, 0, 0);

	map = new MapCanvas(splitter2);
	map->SetBackgroundColour(*wxWHITE);
	map->SetScrollbars(MAP_SCROLL_PIX_H,
				  MAP_SCROLL_PIX_V,
				  MAP_SCROLL_LEN_H,
				  MAP_SCROLL_LEN_V,
				  0,
				  0);

	splitter1->SplitVertically(list, splitter2, (FRAME_WIDTH / 2));
    splitter1->SetMinimumPaneSize(MINIMUM_WINDOW_SIZE);
	
	splitter2->SplitHorizontally(detail, map, (FRAME_HEIGHT / 4));
    splitter2->SetMinimumPaneSize(MINIMUM_WINDOW_SIZE);

    CreateStatusBar();

	codes = new TCodes(dirs->GetMTUDataDir());
	sys = new System(codes);
}

SystemFrame::~SystemFrame()
{
    delete sys;
	delete codes;
	if(dirs != NULL)
		delete dirs;
}

void
SystemFrame::InitButtons(wxToolBar *tb)
{
int i;
int currentX = DLG_OFFSET;
#ifdef __WXMSW__
  int width = 24;
#else
  int width = 16;
#endif
wxBitmap* bm[MAX_BUTTONS];

	// Set up my toolbar
	tb->SetMargins(2, 2);
#ifdef __WXMSW__
	bm[0] = new wxBitmap("icon1");
	bm[1] = new wxBitmap("icon2");
	bm[2] = new wxBitmap("icon3");
	bm[3] = new wxBitmap("icon4");
	bm[4] = new wxBitmap("icon5");
	bm[5] = new wxBitmap("icon6");
	bm[6] = new wxBitmap("icon7");
	bm[7] = new wxBitmap("icon8");
	bm[8] = new wxBitmap("icon9");
	bm[9] = new wxBitmap("icon10");
	bm[10] = new wxBitmap("icon11");
	bm[11] = new wxBitmap("icon12");
	bm[12] = new wxBitmap("icon13");
	bm[13] = new wxBitmap("icon14");
	bm[14] = new wxBitmap("icon15");
	bm[15] = new wxBitmap("icon16");
#else
	bm[0] = new wxBitmap(open_xpm);
	bm[1] = new wxBitmap(save_xpm);
	bm[2] = new wxBitmap(print_xpm);
	bm[3] = new wxBitmap(help_xpm);
	bm[4] = new wxBitmap(exit_xpm);

	bm[5] = new wxBitmap(config_xpm);
	bm[6] = new wxBitmap(orb_xpm);
	bm[7] = new wxBitmap(orb_m_xpm);
	bm[8] = new wxBitmap(orb_a_xpm);
	bm[9] = new wxBitmap(orb_d_xpm);
	bm[10] = new wxBitmap(mag_xpm);
	bm[11] = new wxBitmap(critter_xpm);

	bm[12] = new wxBitmap(mag_p_xpm);
	bm[13] = new wxBitmap(mag_m_xpm);
	bm[14] = new wxBitmap(mag_a_xpm);
	bm[15] = new wxBitmap(mag_o_xpm);
#endif

	for(i = 0;i < MAX_BUTTONS;i++) {
		tb->AddTool(menu_struct[i].id, *(bm[i]), wxNullBitmap, FALSE,
			currentX, -1, NULL, 
			menu_struct[i].short_help, menu_struct[i].long_help);
		currentX += width + DLG_OFFSET;
		if((4 == i) || (11 == i))
			tb->AddSeparator();
	}
		
	tb->Realize();

	// Can delete the bitmaps since they're reference counted
	for (i = 0; i < MAX_BUTTONS; i++)
		delete bm[i];
}

void
SystemFrame::OnCloseWindow(wxCloseEvent &event)
{
wxString ptr;
int i,j;
char temp_str[MAX_FILE_LENGTH];
char *ptr1;
wxMenuItem *mn;
wxConfig *config;

	config = dirs->GetConfig();
	// init params
	config = new wxConfig((const wxString) RESOURCE_FILE);
	wxConfigBase::Set(config);
	wxConfigBase::Get();

	j = 0;
	for(i = 0; i < MAX_FILES;i++) {
		char *ptr2;

		mn = file_menu->FindItem(SYSTEM_FILE1 + j);
		if((NULL == mn) || ((ptr1 = (char *)mn->GetText().GetData()) == NULL))
			continue;
		// WTF?????
		while((ptr2 = strchr(ptr1, '\\')) != NULL) {
			*ptr2 = '/';
		}
		sprintf(temp_str, "%s/%s%d", SECTION_HISTORY, ENTRY_SYST_FILES, j+1);
		config->Write((const wxString) temp_str, (const wxString) ptr1);
		j++;
	}

	Show(FALSE);
	app->OnExit();
}

void 
SystemFrame::AddToMenu(wxMenu *mn, int id)
{
int i=0;

	while(menu_struct[i].id != 0) {
		if(id == menu_struct[i].id) {
			mn->Append(id, menu_struct[i].desc, menu_struct[i].long_help);
			return;
		}
		i++;
	}

	// oh-oh
	mn->Append(999, "Bad", "This is wrong!");
}

// add a file name to the 'file' menu
void 
SystemFrame::AppendMenu(char *s)
{
char *t;
int i,j;
wxMenuItem *mn;

	// look for the just opened file					
	j = 0;
	for(i = 0;i < MAX_FILES;i++) {
		if((mn = file_menu->FindItem(SYSTEM_FILE1 + i)) == NULL)
			continue;
		if((t = (char *)mn->GetText().GetData()) == NULL)
			continue;
		if(stupid_file_name_cmp(s, t) == 0)
			j |= 1;
	}
	// file not found in out list, add it
	if(j == 0) {
		// brand new one
		mn = file_menu->FindItem(SYSTEM_FILE1);
		if((NULL == mn) || (mn->GetText().IsNull())) {
			file_menu->AppendSeparator();
			file_menu->Append(SYSTEM_FILE1, s, "Load Previous File");
		} else {
			// find last one
			j = -1;
			for(i = SYSTEM_FILE1;i < (SYSTEM_FILE1 + MAX_FILES);i++) {
				mn = file_menu->FindItem(i);
				if((NULL == mn) || (mn->GetText().IsNull())) {
					j = i;
					break;
				}
			}
			
			// not full?
			if(j != -1) {
				mn = file_menu->FindItem(j - 1);
				t = (char *)mn->GetText().GetData();
				file_menu->Append(j, t, "Load Previous File");
			} else {
				// ok now, point to last one
				j = SYSTEM_FILE1 + MAX_FILES - 1;
			}

			// move 'em all down
			for(i = j; i > SYSTEM_FILE1;i--) {
				mn = file_menu->FindItem(i - 1);
				file_menu->SetLabel(i, mn->GetText().GetData());
			}

			// append the new one
			file_menu->SetLabel(SYSTEM_FILE1, s);
		}
	}
}

void
SystemFrame::OnToolEnter(wxCommandEvent& event)
{
int i=0;
bool fail=TRUE;

	while(menu_struct[i].id != 0) {
		if(event.GetSelection() == menu_struct[i].id) {
			wxString str;
			str.Printf("%s", menu_struct[i].long_help);
			SetStatusText(str, 0);
			fail = FALSE;
			break;
		}
		i++;
	}
	if(fail)
		SetStatusText("", 0);
}

void 
SystemFrame::Load(char *s)
{
char buff[BIG_BUFFER_SIZE],*ptr2;

	if(s) {
		sprintf(buff, s);
		while((ptr2 = strchr(buff, '\\')) != NULL)
			*ptr2 = '/';
		if(sys->LoadFile(buff) > -1) {
#ifdef SYS_DEBUG
sys->dump(stderr);
#endif
			dirs->CrackNames(buff);
			list->DeleteAllItems();
			UpdateTree();
			detail->Refresh();
			list->Refresh();
			map->Refresh();
			AppendMenu(buff);
			// reuse s
			if((s = sys->GetName()) != NULL)
				sprintf(buff, "System Viewer: %s", s);
			else
				sprintf(buff, "System Viewer");
			SetTitle(buff);
			SetMenuMode(TRUE);
		}
	}
}

void 
SystemFrame::LoadFileCommand(wxCommandEvent& event)
{
wxString ptr;
char *s;

	ptr = wxFileSelector("Load System File", NULL, NULL, "sys",
		"*.sys", wxOPEN | wxHIDE_READONLY, this);
	s = (char *)ptr.GetData();
	if(s[0] != 0)
		Load(s);
}

void
SystemFrame::LoadFile(const int id)
{
wxMenuItem *mn;

	mn = file_menu->FindItem(id);
	Load((char *)mn->GetText().GetData());
}

void 
SystemFrame::LoadFile1(wxCommandEvent& event)
{
	LoadFile(SYSTEM_FILE1);
}

void 
SystemFrame::LoadFile2(wxCommandEvent& event)
{
	LoadFile(SYSTEM_FILE2);
}

void 
SystemFrame::LoadFile3(wxCommandEvent& event)
{
	LoadFile(SYSTEM_FILE3);
}

void 
SystemFrame::LoadFile4(wxCommandEvent& event)
{
	LoadFile(SYSTEM_FILE4);
}

void 
SystemFrame::LoadFile5(wxCommandEvent& event)
{
	LoadFile(SYSTEM_FILE5);
}

void 
SystemFrame::SaveSystem(wxCommandEvent& event)
{
wxString ptr;
char *s;

	ptr = wxFileSelector("Save System File", NULL, NULL, "sys", 
					   "*.sys", wxSAVE | wxHIDE_READONLY, this);
	s = (char *)ptr.GetData();
	if(s[0] != 0)
		sys->SaveSystem(s);
}

void 
SystemFrame::PrintSystem(wxCommandEvent& event)
{
	if(print_dlg == NULL)
	    print_dlg = new PrintDialog(this);
	print_dlg->DoPrint(sys);
}

void 
SystemFrame::EditSystem(wxCommandEvent& event)
{
char buff[BIG_BUFFER_SIZE];
char buff1[BIG_BUFFER_SIZE],buff2[BIG_BUFFER_SIZE];
char buff3[BIG_BUFFER_SIZE],buff4[BIG_BUFFER_SIZE];
char *ptr;
			
	if(sys_dlg == NULL)
	    sys_dlg = new SystemDialog(this);

	buff1[0] = buff2[0] = buff3[0] = buff4[0] = 0;
	if((ptr = sys->GetName()) != NULL)
	    strcpy(buff1, ptr);
	if((ptr = sys->GetSectName()) != NULL)
	    strcpy(buff2, ptr);
	if((ptr = sys->GetSSName()) != NULL)
	    strcpy(buff3, ptr);
	if((ptr = sys->GetLoc()) != NULL)
	    strcpy(buff4, ptr);

	if(sys_dlg->GetSystemParams(buff1, buff2, buff3, buff4)) {
		sys->SetName(buff1);
		sys->SetSectName(buff2);
		sys->SetSSName(buff3);
		sys->SetLoc(buff4);
		if((ptr = sys->GetName()) != NULL)
			sprintf(buff, "System Viewer: %s", ptr);
	}
}

void 
SystemFrame::EditOrbit(wxCommandEvent& event)
{
char buff[BIG_BUFFER_SIZE];
BaseOrbit *bo;

	bo = sys->GetCurrentOrbit();

	if(bo->IsStar()) {
		int ty,cl,sz;
		DetailStar *ds;
				
		if(star_dlg == NULL)
		    star_dlg = new EditStarDialog(this);

		ds = ((StarOrbit *)bo)->GetStar();
		ds->GetName(buff);
		ty = ds->GetStarType();
		cl = ds->GetStarClass();
		sz = ds->GetStarSize();
		if(star_dlg->GetStar(buff, &ty, &cl, &sz)) {
			ds->SetName(buff);
			ds->SetStarType(ty);
			ds->SetStarClass(cl);
			ds->SetStarSize(sz);
			detail->Refresh();
			list->DeleteAllItems();
			UpdateTree();
			list->Refresh();
			map->Refresh();
		}
	} else if(bo->IsLargeGG() || bo->IsSmallGG()) {
		char name[BIG_BUFFER_SIZE];
		int g;
		planet *p;

		if(gg_dlg == NULL)
		    gg_dlg = new EditGGDialog(this);

		p = ((WorldOrbit *) bo)->GetWorld();
		p->GetName(name);
		if(bo->IsLargeGG())
		    g = 1;
		else
		    g = 0;
	    if(gg_dlg->GetGG(name, &g)) {
			p->SetName(name);
			if(g)
			    p->SetSizeVal(SIZE_LARGE_GG);
			else
			    p->SetSizeVal(SIZE_SMALL_GG);					    
			detail->Refresh();
			list->DeleteAllItems();
			UpdateTree();
			list->Refresh();
			map->Refresh();
	    }
	} else {
		if(world_dlg == NULL)
		    world_dlg = new EditWorldDialog(this, codes->GetSystTable());

		if(world_dlg->GetWorld(bo, codes->GetBaseTable(),
			   codes->GetAllegTable())) {
			detail->Refresh();
			list->DeleteAllItems();
			UpdateTree();
			list->Refresh();
			map->Refresh();
		}
	}
}

void 
SystemFrame::MoveOrbit(wxCommandEvent& event)
{
	if(move_dlg == NULL)
	    move_dlg = new MoveDialog(this);

	if(move_dlg->GetMove(sys)) {
		detail->Refresh();
		list->DeleteAllItems();
		UpdateTree();
		list->Refresh();
		map->Refresh();
	}
}

void 
SystemFrame::AddOrbit(wxCommandEvent& event)
{
char *ptr;
STAR_INDEX si;
ORBIT_TYPE ot;
float orbit;
			
	if(add_dlg == NULL)
	    add_dlg = new AddDialog(this);

	if(add_dlg->GetAdd(sys->GetCurrentOrbit(), &si, &ot, &orbit)) {
		switch(sys->CheckOrbit(si, orbit, ot)) {
			case OC_TOO_MANY_STARS:
			    ptr = "Too many star orbits.";
				break;
			case OC_NO_ORB:
			    ptr = "Orbit not available.";
				break;
			case OC_CONFLICTS:
			    ptr = "Star confliction.";
				break;
			case OC_STAR_CONFLICTS:
			    ptr = "Orbit conflicts with existing star.";
				break;
			case OC_NO_SAT:
			    ptr = "Can not be a satillite.";
				break;
			case OC_NOT_A_WORLD:
			    ptr = "Can not add that type of orbit.";
				break;
			default:
			    ptr = NULL;
				break;
		}
				
		if(ptr != NULL)
		    wxMessageBox(ptr, "Add Error",
						 wxOK | wxCENTRE | wxICON_EXCLAMATION);
		else {
			if(CreateOrbit(ot, orbit)) {
				detail->Refresh();
				list->DeleteAllItems();
				UpdateTree();
				list->Refresh();
				map->Refresh();
		    } else
			    wxMessageBox("Strange error?!?!?!", "Add Error",
						 wxOK | wxCENTRE | wxICON_EXCLAMATION);
		}
	}
}

// XXX this one is pretty ugly
char *
SystemFrame::GetFilename(char *buff, bool for_enc)
{
char *ptr,*ext,*filename,*d_name;

	// use buff for scratch....
	sprintf(buff, "%s", sys->GetFileName());
	// XXX unix specific:
	ptr = strrchr(buff, '/');

	if(NULL == ptr)
		ptr = buff;
	else
		ptr++;

	filename = new char[strlen(ptr) + 5];
	strcpy(filename, ptr);
	ptr = strrchr(filename, '.');
	if(ptr != NULL)
		*ptr = 0;

	if(for_enc) {
		d_name = "sys";
		ext = ".enc";
	} else {
		d_name = "world";
		ext = ".wld";
	}
	if(buff[0] != 0) {
		strcat(buff, "/");
		strcat(buff, d_name);
	}
	strcat(buff, filename);
	strcat(buff, ext);
	delete filename;

	return(buff);
}

void 
SystemFrame::DetailOrbit(wxCommandEvent& event)
{
char buff[512],buff1[256];
char *exe;

	exe = dirs->GetWorld();
	sprintf(buff1, "%s", sys->GetFileName());
	sprintf(buff, "%s -f%s -i%d -o%s", exe,
		sys->GetFileName(), sys->GetOrbitIndex(sys->GetCurrentOrbit()),
		GetFilename(buff1, FALSE));
	wxExecute(buff);
}

void 
SystemFrame::EncOrbit(wxCommandEvent& event)
{
BaseOrbit *bo;
char buff[512],buff1[256];
char lbuff1[12],lbuff2[80];
char *exe;

	bo = sys->GetCurrentOrbit();
	if(bo->IsStar() || bo->IsLargeGG() || bo->IsSmallGG()) {
	    wxMessageBox("Cannot generate encounters for this orbit", 
			"Encounter Error", wxOK | wxCENTRE | wxICON_EXCLAMATION);
		return;
	}

	if(bo->IsBelt() || bo->IsRing()) {
		if(wxMessageBox(
			"This may produce strange results.\nDo you wish to continue?",
			"Confirmation", wxYES_NO | wxICON_QUESTION | wxCENTRE) !=
				wxYES)
			return;
	}

	exe = dirs->GetEncTbl();
	sprintf(buff1, "%s", sys->GetFileName());
	bo->FormatUWP(lbuff1, 11);
	strstrip(lbuff1, TRUE);
	bo->FormatName(lbuff2, 79);
	strstrip(lbuff2, TRUE);
	sprintf(buff, "%s -u%s -n%s -o%s", exe,
		lbuff1, lbuff2, GetFilename(buff1, TRUE));
	wxExecute(buff);
}

int resp;
void 
SystemFrame::DeleteOrbit(wxCommandEvent& event)
{
int resp;

	resp = wxMessageBox(
			    " Are you sure you want to delete the current world? ",
				"Confirmation", wxYES_NO | wxICON_QUESTION | wxCENTRE);
	if(resp == wxYES) {
		if(!sys->DeleteOrbit(sys->GetCurrentOrbit())) {
			wxMessageBox(" Unable to delete this world!",
						 "Delete Error",
						 wxOK | wxCENTRE | wxICON_EXCLAMATION);
	    } else {
			detail->Refresh();
			list->DeleteAllItems();
			UpdateTree();
			list->Refresh();
			map->Refresh();
		}
	 }
}

void 
SystemFrame::ZoomIn(wxCommandEvent& event)
{
	if(map_first > 0)
	    map_first--;
	map->Refresh();
}

void 
SystemFrame::ZoomOut(wxCommandEvent& event)
{
	map_first++;
	map->Refresh();
}

void 
SystemFrame::ZoomSelect(wxCommandEvent& event)
{
int map_temp;

	map_center = sys->TranslateParentVIndex(list->GetSelection());
	map_temp = sys->TranslateVIndex(list->GetSelection());
	map_first = sys->TranslateRelIndex(map_temp, map_center);
	map->Refresh();
}

void 
SystemFrame::ZoomOrbit(wxCommandEvent& event)
{
	map_first = 0;
	map_center = sys->TranslateVIndex(list->GetSelection());
	map->Refresh();
}

void 
SystemFrame::Quit(wxCommandEvent& event)
{
	Close(TRUE);
}

void 
SystemFrame::OnAbout(wxCommandEvent& event)
{
AboutDialog *dlg;

	dlg = new AboutDialog(this);
	dlg->ShowAbout("System Editor");
	delete dlg;
}

// --------------------------------------------------------------------
void
SystemFrame::SetMenuMode(bool flag)
{
wxToolBar *tb;

	tb = GetToolBar();
	tb->EnableTool(SYSTEM_SAVE, flag);
	tb->EnableTool(SYSTEM_PRINT, flag);

	tb->EnableTool(SYSTEM_EDIT_SYSTEM, flag);
	tb->EnableTool(SYSTEM_EDIT_ORBIT, flag);
	tb->EnableTool(SYSTEM_EDIT_MOVE, flag);
	tb->EnableTool(SYSTEM_EDIT_ADD, flag);
	tb->EnableTool(SYSTEM_EDIT_DEL, flag);
	tb->EnableTool(SYSTEM_EDIT_DETAIL, flag);
	tb->EnableTool(SYSTEM_EDIT_ENC, flag);

	tb->EnableTool(SYSTEM_ZOOM_IN, flag);
	tb->EnableTool(SYSTEM_ZOOM_OUT, flag);
	tb->EnableTool(SYSTEM_ZOOM_SELECT, flag);
	tb->EnableTool(SYSTEM_ZOOM_ORBIT, flag);

	file_menu->Enable(SYSTEM_SAVE, flag);
	file_menu->Enable(SYSTEM_PRINT, flag);

	edit_menu->Enable(SYSTEM_EDIT_SYSTEM, flag);
	edit_menu->Enable(SYSTEM_EDIT_ORBIT, flag);
	edit_menu->Enable(SYSTEM_EDIT_MOVE, flag);
	edit_menu->Enable(SYSTEM_EDIT_ADD, flag);
	edit_menu->Enable(SYSTEM_EDIT_DEL, flag);
	edit_menu->Enable(SYSTEM_EDIT_DETAIL, flag);
	edit_menu->Enable(SYSTEM_EDIT_ENC, flag);

	map_menu->Enable(SYSTEM_ZOOM_IN, flag);
	map_menu->Enable(SYSTEM_ZOOM_OUT, flag);
	map_menu->Enable(SYSTEM_ZOOM_SELECT, flag);
	map_menu->Enable(SYSTEM_ZOOM_ORBIT, flag);
}

// --------------------------------------------------------------------
// orbit creation helpers
bool
SystemFrame::CreateOrbit(ORBIT_TYPE ot, float orbit)
{
    switch(ot) {
		case OT_STAR:
			return(CreateStar(orbit));
			break;
		case OT_GG:
			return(CreateGG(orbit));
			break;
		case OT_BELT:
			return(CreateBelt(orbit));
			break;
		case OT_WORLD:
			return(CreateWorld(orbit));
			break;
		case OT_SATILLITE:
			return(CreateSat(orbit));
			break;
		case OT_RING:
			return(CreateRing(orbit));
			break;
		case OT_UNKNOWN:
//			wxMessageBox("Unknown error!", "Add Error",
//						 wxOK | wxCENTRE | wxICON_EXCLAMATION);
			return(FALSE);
			break;
	}
	return(FALSE);
}

bool
SystemFrame::CreateStar(float o)
{
unsigned short mask,pmask;
SimpleOrbit *so;
BaseOrbit *bo;
char name[BIG_BUFFER_SIZE];
int ty,cl,sz;
DetailStar *ds;

    bo = sys->GetCurrentOrbit();
	pmask = bo->GetOrbMask();

	// identify the parent
	if(pmask & ORBIT_IS_PRIMARY) {
		PrimOrbit *po = (PrimOrbit *)bo;

		mask = ORBIT_IS_STAR | ORBIT_ORBITS_PRIMARY;
		if(po->GetComp(0) == NULL) {
		}
		else if(po->GetComp(1) == NULL)
			mask |= ORBIT_SECOND_STAR;
		else
		    return(FALSE);

		so = po;
	} else if(pmask & ORBIT_ORBITS_PRIMARY) {
		CompOrbit *co = (CompOrbit *)bo;
		
		mask = ORBIT_IS_STAR | ORBIT_ORBITS_COMP;
		if(co->GetTert() != NULL)
		    return(FALSE);
		if(pmask & ORBIT_SECOND_STAR)
			mask |= ORBIT_SECOND_STAR;
		so = co;
	}
	
	if(star_dlg == NULL)
	    star_dlg = new EditStarDialog(this);

	ty = cl = sz = 0;
	name[0] = 0;
	if(star_dlg->GetStar(name, &ty, &cl, &sz)) {
		ds = new DetailStar(name, ty, cl, sz);
		if(so->AppendToOrbitTable(mask, o, ds) != NULL)
		    return(TRUE);
		delete ds;
	}

	return(FALSE);
}

bool
SystemFrame::CreateGG(float o)
{
unsigned short mask,pmask;
char name[BIG_BUFFER_SIZE];
int sz;
SimpleOrbit *so;
BaseOrbit *bo;
planet *p;

    bo = sys->GetCurrentOrbit();
	pmask = bo->GetOrbMask();

	// identify the parent
	if(!(pmask & ORBIT_IS_STAR))
	    return(FALSE);
	else {
		mask = ORBIT_IS_WORLD;
		if(pmask & ORBIT_IS_PRIMARY)
		    mask |= ORBIT_ORBITS_PRIMARY;
		else if(pmask & ORBIT_ORBITS_PRIMARY)
		    mask |= ORBIT_ORBITS_COMP;
		else if(pmask & ORBIT_ORBITS_COMP)
		    mask |= ORBIT_ORBITS_TERT;
		else
		    return(FALSE);
	}

	so = (SimpleOrbit *)bo;

	if(gg_dlg == NULL)
	    gg_dlg = new EditGGDialog(this);

	sz = 0;
	name[0] = 0;
	if(gg_dlg->GetGG(name, &sz)) {
		char *ptr;
		
		if(sz)
		    ptr = "Large GG";
		else
		    ptr = "Small GG";

		p = new planet(name, ptr, 0, 0l, 0l, 0l, 0, "  ");
		if(so->AppendToOrbitTable(mask, o, p) != NULL)
		    return(TRUE);
		delete p;
	}

	return(FALSE);
}

bool
SystemFrame::CreateOrbit(char *u, float o)
{
unsigned short mask,pmask;
SimpleOrbit *so;
BaseOrbit *bo;
planet *p;

    bo = sys->GetCurrentOrbit();
	pmask = bo->GetOrbMask();

	// identify the parent
	if(!(pmask & ORBIT_IS_STAR))
		mask = (pmask & ORBIT_ORBIT_FLAGS) | ORBIT_IS_SAT;
	else {
		mask = ORBIT_IS_WORLD;
		if(pmask & ORBIT_IS_PRIMARY)
		    mask |= ORBIT_ORBITS_PRIMARY;
		else if(pmask & ORBIT_ORBITS_PRIMARY)
		    mask |= ORBIT_ORBITS_COMP;
		else if(pmask & ORBIT_ORBITS_COMP)
		    mask |= ORBIT_ORBITS_TERT;
		else
		    return(FALSE);
	}

	so = (SimpleOrbit *)bo;

	p = new planet(NULL, u, 0, 0l, 0l, 0l, 0, "  ");
	// walk over bo (don''t need it anymore)
	if((bo = so->AppendToOrbitTable(mask, o, p)) == NULL) {
		delete p;
		return(FALSE);
	}

	return(TRUE);
}

// --------------------------------------------------------------------
//  map drawing
void
SystemFrame::DrawStar(wxDC& dc, int x, int y,
					  StarOrbit *so, bool large)
{
wxMemoryDC mem_dc;
wxBitmap *bm;
wxImageList *i_list;

	i_list = list->GetImageList();
	bm = (wxBitmap *)i_list->GetBitmap(so->GetImageID());
	mem_dc.SelectObject(*bm);
	dc.Blit(x - 6, y - 6, ICON_WIDTH, ICON_HEIGHT, &mem_dc, 0, 0, wxCOPY, TRUE);
	mem_dc.SelectObject(wxNullBitmap);
}

void
SystemFrame::DrawWorld(wxDC &dc, int x, int y,
							BaseOrbit *bo, bool large)
{
    if(bo->IsStar()) {
		DrawStar(dc, x, y, (StarOrbit *) bo, large);
	} else {
		wxMemoryDC mem_dc;
		wxBitmap *bm;
		wxImageList *i_list;

		int hab_flag;
		char *ptr;
		wxBrush *b;
		wxPen *p;

		hab_flag = bo->GetHabitible();
		if(hab_flag == HAB_HABIT)
		    ptr = "GREEN";
		else if(hab_flag < HAB_HABIT)
		    ptr = "RED";
		else
		    ptr = "BLUE";

		p = new wxPen(ptr, ORBIT_WIDTH, wxSOLID);
		b = new wxBrush(ptr, wxSOLID);
		dc.SetPen(*p);			
		dc.SetBrush(*b);

		i_list = list->GetImageList();
		bm = (wxBitmap *)i_list->GetBitmap(bo->GetImageID());
		mem_dc.SelectObject(*bm);
		dc.Blit(x - 6, y - 6, ICON_WIDTH, ICON_HEIGHT, &mem_dc, 0, 0, 
				wxCOPY, TRUE);
		mem_dc.SelectObject(wxNullBitmap);

		dc.SetPen(*wxBLACK_PEN);
		dc.SetBrush(*wxBLACK_BRUSH);
		delete p;
		delete b;
	}
}

void
SystemFrame::DrawMap(wxDC& dc)
{
    DrawMapAt(dc, map_center, map_first);
}

int
SystemFrame::DrawMapAt(wxDC& dc, int center, int first_orb,
					   int start_x, int start_y, bool draw_empty)
{
int count,ret=-1;
int last=-1;
int type;
char buffer[120],buffer2[120],buffer3[120];
float scale;
SimpleOrbit *so;
BaseOrbit *o;
BaseOrbit *first;
int x_center,y_center;
int x_corner,y_corner;

    count = 0;
	ret = 0;
#if 0
	dc.DrawLine(0, MAP_HEIGHT, MAP_WIDTH, MAP_HEIGHT);
	dc.DrawLine(MAP_WIDTH, 0, MAP_WIDTH, MAP_HEIGHT);
#endif
	// get the 'center'
	if((o = sys->GetIndexedOrbit(center)) == NULL)
		return(0);

	type = o->GetOrbType();
	so = (SimpleOrbit *) o;
	x_center = LARGEST_ORBIT + start_x + MAP_X_OFFSET;
	y_center = LARGEST_ORBIT + start_y + MAP_Y_OFFSET;
	x_corner = x_center - (MAP_WIDTH / 2);
	y_corner = y_center - (MAP_HEIGHT / 2);

	// first, figure out which orbits can be drawn
	if((first = so->GetNthOrbit(first_orb)) != NULL) {
		float min_scale;
		int x_world,y_world;

		min_scale = 0.0;
		
		// next, find the last one that can be drawn
		last = first_orb;
		while(TRUE) {
			BaseOrbit *temp;
			float temp_min,temp_max;

//			temp = sys->GetOrbit(st, pl1, sat);
			temp = so->GetNthOrbit(last);

			// exhaust them all?
			if(temp == NULL) {
				last--;
			    break;
			}

			// check for scale....
			if(temp->GetOrbit() >= 0.0) {
				if(type & ORBIT_IS_STAR)
				    temp_min = ((StarOrbit *)so)->GetAUOrbit(temp->GetOrbit());
				else
				    temp_min =((WorldOrbit *)so)->GetKmOrbit(temp->GetOrbit());

//			    temp_max /= (LARGEST_ORBIT / 2);
				temp_max = (LARGEST_ORBIT) / temp_min;

				if(min_scale == 0.0) {
					if(type & ORBIT_IS_STAR)
					    min_scale =
						  ((StarOrbit *)so)->GetAUOrbit(temp->GetOrbit());
					else
					    min_scale =
						  ((WorldOrbit *)so)->GetKmOrbit(temp->GetOrbit());
				} else {
					if((temp_max * min_scale) < (SMALLEST_ORBIT)) {
						last--;
						break;
					}
				}
			}

			scale = temp_max;
			// increments
			last++;
			count++;
		}

		// first pass, draw orbits
		while(last >= first_orb) {
			int hab_flag;
			char *ptr;
			int rad;
			float temp_rad;
		    BaseOrbit *temp;
			wxPen *p;

			temp = so->GetNthOrbit(last);
			if(type & ORBIT_IS_STAR)
			    temp_rad = ((StarOrbit *)so)->GetAUOrbit(temp->GetOrbit());
			else
			    temp_rad = ((WorldOrbit *)so)->GetKmOrbit(temp->GetOrbit());

			if(temp->GetOrbit() < 0.0) {
				x_world = x_center - (MAIN_STAR_RAD + (2 * STAR_SPIKE));
				// it might be nice to allow a second close companion
				y_world = y_center + (MAIN_STAR_RAD + (2 * STAR_SPIKE)) ;
//				DrawStar(dc, x_world, y_world);
			} else {
				rad = (int)(temp_rad * scale);

				if(rad <= MAP_HEIGHT) {
					hab_flag = temp->GetHabitible();
					if(hab_flag == HAB_HABIT)
					    ptr = "GREEN";
					else if(hab_flag < HAB_HABIT)
					    ptr = "RED";
					else
					    ptr = "BLUE";

					p = new wxPen(ptr, ORBIT_WIDTH, wxSOLID);
					if((temp->IsBelt()) || (temp->IsRing())) {
						wxBrush *brush;
						int temp_rad;
					    // windoze will need some work here:

					    temp_rad = rad + 5;
					    dc.SetPen(*wxWHITE_PEN);       // effectively invisible
						brush = wxTheBrushList->FindOrCreateBrush(ptr,
											wxCROSSDIAG_HATCH);
#if 0
					    brush = new wxBrush(ptr, wxSOLID);
					    brush->SetStipple(stipple_bm);
					    brush->SetStyle(wxSTIPPLE);
#endif
					    dc.SetBrush(*brush);

					    dc.DrawEllipse(x_center - temp_rad,
									   y_center - temp_rad,
									   2 * temp_rad,
									   2 * temp_rad);
				
						temp_rad -= 10;
						brush = wxTheBrushList->FindOrCreateBrush("WHITE",
											wxSOLID);
#if 0
						brush = dc.GetBrush();
						brush->SetStipple(NULL);
						brush->SetStyle(wxSOLID);
						brush->SetColour("WHITE");
#endif
						dc.SetBrush(*brush);
						dc.DrawEllipse(x_center - temp_rad,
									   y_center - temp_rad,
									   2 * temp_rad,
									   2 * temp_rad);

						dc.SetBrush(*wxTRANSPARENT_BRUSH);
					} else {
						dc.SetBrush(*wxTRANSPARENT_BRUSH);
						dc.SetPen(*p);
						dc.DrawEllipse(x_center - rad,
									   y_center - rad,
									   2 * rad,
									   2 * rad);
						dc.SetBrush(*wxTRANSPARENT_BRUSH);
					}

					x_world = x_center + (int)(x_table[ret] * rad);
					y_world = y_center + (int)(y_table[ret] * rad);

					dc.SetPen(*wxBLACK_PEN);
					dc.SetFont(*mapWindowFont);
					temp->FormatName(buffer, -1);
					dc.DrawText(buffer, MAP_TEXT_OFFSET_X1 + x_corner, 
								(ret * (2 * MAP_TEXT_OFFSET_Y) + y_corner));

					temp->FormatUWP(buffer2, -1);
					temp->FormatOrbit(buffer3);
					sprintf(buffer, "(%s) %s", buffer2, buffer3);
					dc.DrawText(buffer, MAP_TEXT_OFFSET_X1 + x_corner,
						ret * (2 * MAP_TEXT_OFFSET_Y) + MAP_TEXT_OFFSET_Y + y_corner);
	
					dc.DrawLine(MAP_TEXT_OFFSET_X + x_corner,
						   ret * (2 * MAP_TEXT_OFFSET_Y) + MAP_TEXT_OFFSET_Y +y_corner,
								x_world,
								y_world);

					if(!temp->IsBelt())
					    DrawWorld(dc, x_world, y_world, temp);
			
//					dc.SetPen(wxBLACK_PEN);
					if(temp->GetOrbit() >= 0.0)
					    delete p;
			
				}
			}
			count--;
			last--;
			ret++;
		}
	}

	// figure what to draw in the center 
	if((ret > 0) || (draw_empty)) {
		float temp_scale;
		
		dc.SetPen(*wxBLACK_PEN);  
		dc.SetFont(*mapBigFont);
		dc.DrawText(o->FormatName(buffer, -1),
					MAP_TEXT_OFFSET_X1 + x_corner,
					MAP_TEXT_OFFSET_Y1 + y_corner);
		dc.DrawLine(MAP_TEXT_OFFSET_X2 + x_corner,
					MAP_TEXT_OFFSET_Y + y_corner,
					x_center,
					y_center);
		
		if(first_orb < 1) {
			dc.SetFont(*mapHeaderFont);
			dc.DrawText("Inner System Profile",
						MAP_TEXT_OFFSET_X1 + x_corner,
						MAP_TEXT_OFFSET_Y2 + y_corner);
			dc.SetFont(*mapWindowFont);
			so->FormatName(buffer, -1);
			dc.DrawText(buffer,
						MAP_TEXT_OFFSET_X2 + x_corner,
						y_corner);

			so->FormatUWP(buffer2, -1);
			so->FormatOrbit(buffer3);
			sprintf(buffer, "(%s) %s", buffer2, buffer3);
			dc.DrawText(buffer,
						MAP_TEXT_OFFSET_X2 + x_corner,
						MAP_TEXT_OFFSET_Y + y_corner);

			DrawWorld(dc, x_center, y_center, so, TRUE);
		} else {
			dc.SetFont(*mapHeaderFont);
			dc.DrawText("Outer System Profile",
						MAP_TEXT_OFFSET_X1 + x_corner,
						MAP_TEXT_OFFSET_Y2 + y_corner);
			dc.SetFont(*mapWindowFont);
			dc.DrawText("Inner System",
						MAP_TEXT_OFFSET_X2 + x_corner,
						y_corner);
			dc.SetBrush(*wxTRANSPARENT_BRUSH);
			dc.DrawRectangle(x_center - (MAIN_STAR_RAD + STAR_SPIKE),
							 y_center - (MAIN_STAR_RAD + STAR_SPIKE),
							 2 * (MAIN_STAR_RAD + STAR_SPIKE),
							 2 * (MAIN_STAR_RAD + STAR_SPIKE));
			dc.SetBrush(*wxBLACK_BRUSH);
			dc.DrawEllipse(x_center - (MAIN_STAR_RAD),
						   y_center - (MAIN_STAR_RAD),
						   2 * (MAIN_STAR_RAD),
						   2 * (MAIN_STAR_RAD));
		}

		// draw scale bar
		dc.SetFont(*mapWindowFont);
		dc.SetBrush(*wxBLACK_BRUSH);
		dc.SetPen(*wxBLACK_PEN);
		sprintf(buffer, "0.0");
		dc.DrawText(buffer,
					MAP_TEXT_OFFSET_X2 - 10 + x_corner,
					MAP_TEXT_OFFSET_Y2 - SCALE_HEIGHT - 18 + y_corner);

		if(type & ORBIT_IS_STAR)
		    temp_scale = scale / 2;
		else
		    temp_scale = scale * 1000 / 2;
		sprintf(buffer, "%.1f",
				(((float) SCALE_LENGTH / 2.0) / temp_scale) + 0.05);
		dc.DrawText(buffer,
					MAP_TEXT_OFFSET_X2 - 10 + (SCALE_LENGTH / 2) + x_corner,
					MAP_TEXT_OFFSET_Y2 - SCALE_HEIGHT - 18 + y_corner);
		sprintf(buffer, "%.1f",
				(((float) SCALE_LENGTH) / temp_scale) + 0.05);
		dc.DrawText(buffer,
					MAP_TEXT_OFFSET_X2 - 10 + (SCALE_LENGTH) + x_corner,
					MAP_TEXT_OFFSET_Y2 - SCALE_HEIGHT - 18 + y_corner);
		dc.DrawLine(MAP_TEXT_OFFSET_X2 + x_corner,
					MAP_TEXT_OFFSET_Y2 - SCALE_HEIGHT - 4 + y_corner,
					MAP_TEXT_OFFSET_X2 + x_corner,
					MAP_TEXT_OFFSET_Y2 - 4 + y_corner);
		dc.DrawLine(MAP_TEXT_OFFSET_X2 + (SCALE_LENGTH / 2) + x_corner,
					MAP_TEXT_OFFSET_Y2 - SCALE_HEIGHT - 4 + y_corner,
					MAP_TEXT_OFFSET_X2 + (SCALE_LENGTH / 2) + x_corner,
					MAP_TEXT_OFFSET_Y2 - 4 + y_corner);
		dc.DrawLine(MAP_TEXT_OFFSET_X2 + SCALE_LENGTH + x_corner,
					MAP_TEXT_OFFSET_Y2 - SCALE_HEIGHT - 4 + y_corner,
					MAP_TEXT_OFFSET_X2 + SCALE_LENGTH + x_corner,
					MAP_TEXT_OFFSET_Y2 - 4 + y_corner);
		dc.DrawLine(MAP_TEXT_OFFSET_X2 + x_corner,
					MAP_TEXT_OFFSET_Y2 - (SCALE_HEIGHT / 2) - 4 + y_corner,
					MAP_TEXT_OFFSET_X2 + SCALE_LENGTH + x_corner,
					MAP_TEXT_OFFSET_Y2 - (SCALE_HEIGHT / 2) - 4 + y_corner);
		dc.DrawLine(MAP_TEXT_OFFSET_X2 + x_corner,
					MAP_TEXT_OFFSET_Y2 - 4 + y_corner,
					MAP_TEXT_OFFSET_X2 + SCALE_LENGTH + x_corner,
					MAP_TEXT_OFFSET_Y2 - 4 + y_corner);

		dc.DrawRectangle(MAP_TEXT_OFFSET_X2 + x_corner,
						 MAP_TEXT_OFFSET_Y2 - (SCALE_HEIGHT / 2) - 4 +y_corner,
						 BOX_LENGTH,
						 BOX_HEIGHT);
		dc.DrawRectangle(MAP_TEXT_OFFSET_X2 + (SCALE_LENGTH / 2) + x_corner,
						 MAP_TEXT_OFFSET_Y2 - (SCALE_HEIGHT / 2) - 4 +y_corner,
						 BOX_LENGTH,
						 BOX_HEIGHT);

		dc.SetFont(*mapHeaderFont);
		if(type & ORBIT_IS_STAR)
		    dc.DrawText("Scale in A.U.s",
						MAP_TEXT_OFFSET_X2 + x_corner,
						MAP_TEXT_OFFSET_Y2 + y_corner);
		else
		    dc.DrawText("Scale in 1000's Km",
						MAP_TEXT_OFFSET_X2 + x_corner,
						MAP_TEXT_OFFSET_Y2 + y_corner);
	}

	return(ret);
}


// --------------------------------------------------------------------
//  tree canvas stuff
void
SystemFrame::UpdateCurrentOrbit(long id)
{
	sys->SetCurrentOrbitV(id);
	detail->Refresh();
}

void
SystemFrame::UpdateTree(void)
{
    UpdateWithStarOrbit(0l, SI_PRIMARY);
//	list->dump_tree();
}

long
SystemFrame::UpdateWithStarOrbit(long p_id,
								 STAR_INDEX st)
{
BaseOrbit *bo;
int pl=0;
int orb_type,st1;
long id,ret,u_id;

    if(st == SI_PRIMARY)
	    st1 = (int) SI_1ST_COMP;
	else
		st1 = (int) st + 1;

	if((ret = id = UpdateTreeWithOrbit(p_id, st, -1, -1)) > -1) {
		while((bo = sys->GetOrbit(st, pl, -1)) != NULL) {
			orb_type = bo->GetOrbType();
			if(orb_type & ORBIT_IS_STAR) {
				u_id = UpdateWithStarOrbit(id, (STAR_INDEX) st1);
				st1 += 2;
			} else if(orb_type & ORBIT_IS_WORLD) {
				u_id = UpdateWithWorldOrbit(id, st, pl);
			}
			pl++;
		}
	}

	return(ret);
}

long
SystemFrame::UpdateWithWorldOrbit(long p_id,
								  STAR_INDEX st, int pl)
{
int sat=0;
long id,ret;

    if((ret = id = UpdateTreeWithOrbit(p_id, st, pl, -1)) > -1) {
		while(sys->GetOrbit(st, pl, sat) != NULL) {
			// woulda been just a wrapper....
			UpdateTreeWithOrbit(id, st, pl, sat);

			sat++;
		}
	}

	return(ret);
}

long
SystemFrame::UpdateTreeWithOrbit(long p_id, 
								 STAR_INDEX st, int pl, int sat)
{
long ret;
char buff[120];
BaseOrbit *bo;
static int star_ndx;

	if((bo = sys->GetOrbit(st, pl, sat)) != NULL) {
		int image_ndx=icon_dummy;

		sys->GetListing(st, pl, sat, buff);
		if(bo->IsStar()) {
			int r=0,g=0,b=0,star_class;
			StarOrbit *so = (StarOrbit *)bo;
			DetailStar *ds = so->GetStar();
			wxImageList *i_list;
			wxBitmap *bm;
			wxMemoryDC dc;
			wxBrush *brush;
			wxPen *pen;
			wxColour *clr;

			star_class = ds->GetStarClass();
			switch(ds->GetStarType()) {
				default:
				case 0:
					b = 127 + (int)(star_class * 12.7);
					break;
				case 1:
					r = g = 255 - (int)(star_class * 25.5);
					b = 255;
					break;
				case 2:
					r = g = 255;
					b = 255 - (int)(star_class * 12.7);
					break;
				case 3:
					r = g = 255;
					b = 128 - (int)(star_class * 12.7);
					break;
				case 4:
					r = 255;
					g = 255 - (int)(star_class * 25.5);
					break;
				case 5:
					r = 255 - (int)(star_class * 12.7);
					break;
			}
			if(r < 0) r = 0;
			if(g < 0) b = 0;
			if(b < 0) b = 0;
			if(r > 255) r = 255;
			if(g > 255) g = 255;
			if(b > 255) b = 255;
			clr = new wxColour(r, g, b);

			bm = new wxBitmap(ICON_HEIGHT, ICON_WIDTH);
			dc.SelectObject(*bm);
			// TRANSPARENT doesn't seem to work!!!
			dc.SetBrush(*wxWHITE_BRUSH);
			dc.SetPen(*wxWHITE_PEN);
			dc.DrawRectangle( 0, 0, ICON_HEIGHT, ICON_WIDTH);

			brush = new wxBrush(*clr, wxSOLID);
			if((r > 180) && (g > 180) && (b > 180))
				pen = new wxPen("Black", 1, wxSOLID);
			else
	    		pen = new wxPen(*clr, 1, wxSOLID);

			dc.SetPen(*pen);
			dc.SetBrush(*brush);

			dc.DrawEllipse(0, 0, ICON_HEIGHT-1, ICON_WIDTH-1);

			if(p_id == 0l)
				star_ndx = 0;
			image_ndx = icon_star0 + star_ndx;

			i_list = list->GetImageList();
			i_list->Replace(image_ndx, *bm);

			dc.SelectObject(wxNullBitmap);
			delete pen;
			delete brush;
			delete bm;
			delete clr;
			star_ndx++;
		} else if(bo->IsLargeGG()) {
			image_ndx = icon_large_gg;
		} else if(bo->IsSmallGG()) {
			image_ndx = icon_small_gg;
		} else if(bo->IsBelt() || bo->IsRing()){
			image_ndx = icon_size0_world;
		} else {
			planet *p=NULL;
			
			if(bo->GetOrbType() & ORBIT_IS_WORLD) {
				WorldOrbit *w = (WorldOrbit *)bo;
				p = w->GetWorld();
			} else if(bo->GetOrbType() & ORBIT_IS_SAT) {
				SatOrbit *s = (SatOrbit *)bo;
				p = s->GetSat();
			}
			if(p != NULL) {
				int at,hy;

				at = p->get_atmos_val();
				hy = p->get_hydro_val();
				if((at > 9) && (at < 13))
					image_ndx = icon_atmosa_world;
				else if(hy > 8)
					image_ndx = icon_hydroa_world;
				else if(hy > 4) {
					if(at > 4)
						image_ndx = icon_h4a4_world;
					else
						image_ndx = icon_h4a0_world;
				} else {
					if(at > 4)
						image_ndx = icon_h0a4_world;
					else
						image_ndx = icon_h0a0_world;
				}
			}
		}

		if(p_id == 0l)
			ret = list->AddRoot(buff, image_ndx);
		else
			ret = list->AppendItem(p_id, buff, image_ndx);
		bo->SetID(ret);
		bo->SetImageID(image_ndx);
		return(ret);
	}
	return(-1);
}

// ----------------------------------------------------------------------
int
SystemFrame::CalcPageCount(PRINT_TYPE t, PRINT_SOURCE s, int lines, bool msg)
{
int count=0,i=0,page_count=0,lines_per_page;
BaseOrbit *bo,*o;

    if((s == PS_ALL) || (s == PS_IGN))
	    bo = (BaseOrbit *)sys->GetPrimOrbit();
	else
	    bo = sys->GetCurrentOrbit();

    if(t == PT_MAP) {
		count = CalcMapCount(s, bo);
		lines_per_page = 6;
	} else {
		lines_per_page = lines - 2;
		if(msg)
			lines_per_page -= 2;

		while(TRUE) {
			if((s == PS_ALL) || (s == PS_IGN))
			    o = sys->GetIndexedOrbit(i++);
			else
			    o = bo->GetNthOrbit(i++);
			if(o == NULL)
			    break;

			if(!((o->GetOrbType() & ORBIT_IS_SAT) && (s == PS_IGN))) {
				if(t == PT_LIST)
				    count++;
				else {
					int temp_count;

					if(o->IsLargeGG() || o->IsSmallGG())
					    temp_count = (LINE_GG_RETURN + 1);
					else if(o->IsStar())
					    temp_count = (LINE_STAR_RETURN + 1);
					else
					    temp_count = (LINE_OTHER_RETURN + 1);

					if((temp_count + count) > lines_per_page) {
						page_count++;
						count = temp_count;
					} else
						count += temp_count;
				}
			}
	    }
	}

	if(t != PT_DETAIL) {
		page_count = count / lines_per_page;
		if((count % lines_per_page) != 0)
		    page_count++;
	} else {
		page_count++;
	    if(count != 0)
		    page_count++;
	}

	return(page_count);
}

int
SystemFrame::CalcMapCount(PRINT_SOURCE s, BaseOrbit *bo)
{
int i=0,map_count=0;
float min_scale,temp_min,temp_max;
BaseOrbit *o;

    min_scale = 0.0;

	if((s == PS_IGN) && ((bo->GetOrbType() & ORBIT_IS_STAR) == 0))
	    return(0);
	
	while(TRUE) {
		o = bo->GetNthOrbit(i++);
		if(o == NULL)
		    break;

		if((o->GetOrbType() & ORBIT_IS_SAT) && (s == PS_IGN))
		    continue;

		// check for scale....
		if(o->GetOrbit() >= 0.0) {
			if(bo->GetOrbType() & ORBIT_IS_STAR)
			    temp_min = ((StarOrbit *)bo)->GetAUOrbit(o->GetOrbit());
			else
			    temp_min =((WorldOrbit *)bo)->GetKmOrbit(o->GetOrbit());

			temp_max = (LARGEST_ORBIT) / temp_min;

			if(min_scale == 0.0) {
				if(bo->GetOrbType() & ORBIT_IS_STAR)
				    min_scale =
					  ((StarOrbit *)bo)->GetAUOrbit(o->GetOrbit());
				else
				    min_scale =
					  ((WorldOrbit *)bo)->GetKmOrbit(o->GetOrbit());
			} else {
				if((temp_max * min_scale) < (SMALLEST_ORBIT)) {
					map_count++;
					min_scale = 0.0;
				}
			}
		}
		
		if((o->GetOrbType() & ORBIT_IS_SAT) == 0)
		    map_count += CalcMapCount(s, o);
	}

	if(min_scale > 0.0)
	    map_count++;
	
	return(map_count);
}

bool
SystemFrame::CalcMapIndecies(PRINT_SOURCE s, int index, int *cent, int *first)
{
int page_index=0;
BaseOrbit *bo;

	*cent = *first = 0;
	if((s == PS_ALL) || (s == PS_IGN))
	    bo = sys->GetPrimOrbit();
	else
	    bo = sys->GetCurrentOrbit();
#if 0
fprintf(stderr, "\n>>>>> checking for %d\n", index);
#endif
	if(CalcOrbitIndecies(bo, s, index, &page_index, cent, first))
	    return(TRUE);

	return(FALSE);
}

void
SystemFrame::GetMapSize(int *h, int *w)
{
    *h = MAP_HEIGHT;
	*w = MAP_WIDTH;
}

bool
SystemFrame::CalcOrbitIndecies(BaseOrbit *o, PRINT_SOURCE s, int index,
							   int *cur_page, int *cent, int *first)
{
int page_index,i=0;
float min_scale,temp_min,temp_max;
BaseOrbit *bo,*prev=NULL;

    page_index = *cur_page;
    min_scale = 0.0;

	while((bo = o->GetNthOrbit(i)) != NULL) {
#if 0
fprintf(stderr, "p:%d  i:%d  orb:%f  ",
		page_index, i, bo->GetOrbit());
#endif
		// check for scale....
        prev = bo;
		if(bo->GetOrbit() >= 0.0) {
			if(o->GetOrbType() & ORBIT_IS_STAR)
			    temp_min = ((StarOrbit *)o)->GetAUOrbit(bo->GetOrbit());
			else
			    temp_min =((WorldOrbit *)o)->GetKmOrbit(bo->GetOrbit());

			temp_max = (LARGEST_ORBIT) / temp_min;

			if(min_scale == 0.0) {
#if 0
fprintf(stderr, "\n");
#endif
				*first = i;
				if(o->GetOrbType() & ORBIT_IS_STAR)
				    min_scale =
					  ((StarOrbit *)o)->GetAUOrbit(bo->GetOrbit());
				else
				    min_scale =
					  ((WorldOrbit *)o)->GetKmOrbit(bo->GetOrbit());
			} else {
#if 0
fprintf(stderr, "scale:%f  max:%f  min:%f\n",
	    min_scale, temp_max, temp_min);
#endif
				if((temp_max * min_scale) < (SMALLEST_ORBIT)) {
#if 0
fprintf(stderr, "new page  --------------\n");
#endif
					page_index++;
					i--;
					min_scale = 0.0;
					if(page_index > index) {
						*cent = sys->GetOrbitIndex(o);
#if 0
fprintf(stderr, ">>>>taking p:%d f:%d c:%d\n", page_index, *first, *cent);
#endif
						return(TRUE);
					}
				}
			}
		}
		i++;
	}
	
	*cur_page = page_index;
	// fell thru, check if anything was accumulated
	if((min_scale > 0.0) && (prev != NULL)) {
		page_index++;
		if(page_index > index) {
			*cent = sys->GetOrbitIndex(o);
#if 0
fprintf(stderr, ">>>>taking p:%d f:%d c:%d\n", page_index, *first, *cent);
#endif
            return(TRUE);
		}
	}
	// start looking through sub orbits
	//  only printing current orbit?
	if(s == PS_CUR)
	    // yes, so we are done
	    return(FALSE);

	i = 0;
    while((bo = o->GetNthOrbit(i++)) != NULL) {
		if((s != PS_IGN) || (bo->GetOrbType() & ORBIT_IS_STAR)) {
		    if(CalcOrbitIndecies(bo, s, index, &page_index, cent, first))
			    return(TRUE);
		}
	}

	*cur_page = page_index;
	
	return(FALSE);
}

int
SystemFrame::CalcWorldsOnPage(bool list, PRINT_SOURCE s, bool hdr,
							  int lines_per_page, int page)
{
int ndx=0;
int line_count=0,page_count=1,on_page_count=0;
BaseOrbit *bo;

// adjust for header and trailer
	lines_per_page -= 2;
	if(hdr)
		lines_per_page -= 2;

	if(list)
	    return(lines_per_page);

	while((bo = sys->GetIndexedOrbit(ndx++)) != NULL) {
		int temp_count;
		
		if(bo->IsLargeGG() || bo->IsSmallGG())
		    temp_count = (LINE_GG_RETURN + 1);
		else if(bo->IsStar())
		    temp_count = (LINE_STAR_RETURN + 1);
		else
		    temp_count = (LINE_OTHER_RETURN + 1);

		if((line_count + temp_count) > lines_per_page) {
			page_count++;
			line_count = temp_count;
		}
		else
		    line_count += temp_count;

		if(page_count > page)
		    return(on_page_count);

		if(page_count == page)
		    on_page_count++;
    }

	return(0);
}

// ======================================================================
SystemDetail::SystemDetail(wxWindow *frame) :
		wxScrolledWindow(frame, -1, wxPoint(0, 0), wxSize(500, 305))
{
}

void SystemDetail::OnDraw(wxDC& dc)
{
SystemDisplay *sd;
BaseOrbit *bo=NULL;
System *sys;

    dc.Clear();

    sd = new SystemDisplay(&dc);
    sd->Start();
	sys = frame->GetSystem();
	bo = sys->GetCurrentOrbit();
    sd->DrawDetail(sys, bo);

    delete sd;
}

// ======================================================================
MapCanvas::MapCanvas(wxWindow *frame) :
		wxScrolledWindow(frame, -1, wxPoint(0, 0), wxSize(MAP_WIDTH, MAP_HEIGHT))
{
    map_ready = FALSE;
	map_ready = TRUE;
}

void
MapCanvas::OnDraw(wxDC& dc)
{
    dc.Clear();
	if(map_ready)
	    frame->DrawMap(dc);
}

HIT_TEST_VAL
MapCanvas::HitText(int x, int y, int *r_x, int *r_y)
{
	// for now
	return(HT_NONE);
}

// ======================================================================
SystemDisplay::~SystemDisplay()
{
}

int
SystemDisplay::DrawDetails(PRINT_SOURCE ps, char *msg1, char *msg2,
						   int start, int lines)
{
int world_count=0,line_count=0;
System *sys;

	Start(msg1);

    sys = frame->GetSystem();
	if(sys != NULL) {
		BaseOrbit *bo;
		
		if((ps == PS_CUR) || (ps == PS_SUB))
		    bo = sys->GetCurrentOrbit();
		else
		    bo = sys->GetPrimOrbit();

		if(ps == PS_CUR) {
			DrawDetail(sys, bo);
			world_count = 1;
	    } else
		    DrawDetails(sys, bo, &world_count, &line_count, start, lines, ps);
	}
	Finish(msg2);

    return(world_count);
}

int
SystemDisplay::DrawDetails(System *sys, BaseOrbit *bo, int *world_count,
						   int *line_count, int start, int lines,
						   PRINT_SOURCE ps)
{
int wcount,lcount,temp_count,ret=0;

    wcount = *world_count;
	lcount = *line_count;
	if(wcount >= start) {
		if(bo->IsLargeGG() || bo->IsSmallGG())
		    temp_count = (LINE_GG_RETURN + 1);
		else if(bo->IsStar())
		    temp_count = (LINE_STAR_RETURN + 1);
		else
		    temp_count = (LINE_OTHER_RETURN + 1);

		if((temp_count + lcount) <= lines) {
		    DrawDetail(sys, bo);
			Line(" ");
			lcount += temp_count;
		    ret = 1;
		} else
		    ret = -1;
	}
	wcount++;

	if(ret > -1) {
		if((bo->GetOrbType() & ORBIT_IS_STAR) ||
		   ((ps != PS_IGN) && (bo->GetOrbType() & ORBIT_IS_WORLD))) {
			int ndx=0;
			BaseOrbit *bo1;

			while((bo1 = bo->GetNthOrbit(ndx++)) != NULL) {
				temp_count = DrawDetails(sys, bo1, &wcount, &lcount,
										 start, lines, ps);
				if(temp_count < 0) {
					*world_count = wcount;
					*line_count = lcount;
					return(-1);
				}
			}
		}
	}

	*world_count = wcount;
	*line_count = lcount;
	return(ret);
}

int
SystemDisplay::DrawDetail(System *sys, BaseOrbit *b_orb)
{
bool ok=FALSE;
char buff[120];
char buff1[120];
char buff2[120];
char *ptr1,*ptr2;
char type;
int ret;
planet *p;
DetailStar *st;

	if(b_orb != NULL) {
		b_orb->GetFormatedOrbit(buff);
		sprintf(buff1, "    Orbit: %s", buff);
		Line(buff1);
		sprintf(buff, "     Name: %s", b_orb->FormatName(buff2, -1));
		Line(buff);
		  
		type = b_orb->GetOrbType();

		if(type & ORBIT_IS_STAR) {
			StarOrbit *so = (StarOrbit *) b_orb;
			st = so->GetStar();

			sprintf(buff, "First Orb: %d", st->GetFirstOrbit());
			Line(buff);
			sprintf(buff, "Habitible: %d", st->GetHabZone());
			Line(buff);

			sprintf(buff, "    Class: %s", st->GetClassDesc(buff2));
			Line(buff);
			sprintf(buff, "     Size: %s", st->GetSizeDesc(buff2));
			Line(buff);
			sprintf(buff, "    Lumin: %f Sol", st->GetLumin());
			Line(buff);
			sprintf(buff, "     Temp: %f K", st->GetTemp());
			Line(buff);
			sprintf(buff, "   Radius: %f Sol", st->GetRad());
			Line(buff);
			sprintf(buff, "     Mass: %f Sol", st->GetMass());
			Line(buff);

			ret = LINE_GG_RETURN;
			ok = TRUE;
		} else if(b_orb->IsLargeGG() || b_orb->IsSmallGG()) {
			WorldOrbit *w = (WorldOrbit *)b_orb;
			p = w->GetWorld();
			sprintf(buff, "     Size: %s", p->GetSizeDesc(buff1));
			Line(buff);

			ret = LINE_GG_RETURN;
			ok = TRUE;
		} else {
			if(type & ORBIT_IS_WORLD) {
				WorldOrbit *w = (WorldOrbit *)b_orb;
				p = w->GetWorld();
			} else if(type & ORBIT_IS_SAT) {
				SatOrbit *s = (SatOrbit *)b_orb;
				p = s->GetSat();
			}

			ptr1 = p->GetPortDesc(buff1);
			ptr2 = SplitLine(ptr1);
			sprintf(buff, "     Port:  %s", ptr1);
			Line(buff);
			sprintf(buff, "            %s", ptr2);
			Line(buff);
			
			sprintf(buff, "     Size: %s", p->GetSizeDesc(buff1));
			Line(buff);
			sprintf(buff, "    Atmos: %s", p->GetAtmosDesc(buff1));
			Line(buff);
			sprintf(buff, "    Hydro: %s", p->GetHydroDesc(buff1));
			Line(buff);
			sprintf(buff, "      Pop: %s", p->GetPopDesc(buff1));
			Line(buff);
			sprintf(buff, "     Govt: %s", p->GetGovtDesc(buff1));
			Line(buff);
			sprintf(buff, "      Law: %s", p->GetLawDesc(buff1));
			Line(buff);
			sprintf(buff, "     Tech: %s", p->GetTechDesc(buff1));
			Line(buff);
			sprintf(buff, "    Bases: %s", sys->FormatBase(p->GetBase(),
															buff1));
			Line(buff);

			ptr1 = sys->FormatSystem(p->GetSysFlags(), buff1);
			ptr2 = SplitLine(ptr1);
			sprintf(buff, "   System: %s", ptr1);
			Line(buff);
			sprintf(buff, "           %s", ptr2);
			Line(buff);
			
			ptr1 = sys->FormatTrade(p->GetTradeCodes(), buff1);
			ptr2 = SplitLine(ptr1);
			sprintf(buff, "    Trade: %s", ptr1);
			Line(buff);
			sprintf(buff, "           %s", ptr2);
			Line(buff);
			
			ptr1 = sys->FormatOther(p->GetOtherCodes(), buff1);
			ptr2 = SplitLine(ptr1);
			sprintf(buff, "    Other: %s", ptr1);
			Line(buff);
			sprintf(buff, "           %s", ptr2);
			Line(buff);
			
			sprintf(buff, "     Zone: %s", sys->FormatZone(p->GetZone(),
															buff1));
			Line(buff);
			
			sprintf(buff, "    Alleg: %s", sys->FormatAlleg(p->GetAlleg(), buff1));
			Line(buff);
			
			ret = LINE_OTHER_RETURN;
			ok = TRUE;
		}
		
	}

	if(!ok) {
		Line(" Orbit:");
		Line("  Name:");
		Line("  Port:");
		Line(" ");
		Line("  Size:");
		Line(" Atmos:");
		Line(" Hydro:");
		Line("   Pop:");
		Line("  Govt:");
		Line("   Law:");
		Line("  Tech:");
		Line(" Bases:");
		Line("System:");
		Line(" ");
		Line(" Trade:");
		Line(" ");
		Line(" Other:");
		Line(" ");
		Line("  Zone:");
		Line(" Alleg:");
		ret = LINE_OTHER_RETURN;
	}

	return(ret);
}

int
SystemDisplay::DrawLines(char *msg1, char *msg2,
						 int start_count, int end_count)
{
int count=0;
System *sys;

	Start(msg1);

    sys = frame->GetSystem();
	if(sys != NULL)
	    count = DrawStarLine(sys, SI_PRIMARY,
							 &count, start_count, end_count);
	
	Finish(msg2);
	
	return(count);
}

	
// draw a sat
int
SystemDisplay::DrawSatLine(System *sys, STAR_INDEX st,int pl, int sat,
						   int *count, int start_count, int end_count)
{
int ret=0;

	ret = DrawLine(sys, st, pl, sat, count, start_count, end_count);
	return(ret);
}

// draw a world and all of _it's_ orbits
int
SystemDisplay::DrawWorldLine(System *sys, STAR_INDEX st, int pl,
							 int *count, int start_count, int end_count)
{
BaseOrbit *bo;
int sat=0,ret=-1;

    if((bo = sys->GetOrbit(st, pl, -1)) != NULL) {
		ret = DrawLine(sys, st, pl, -1, count, start_count, end_count);
		while(sys->GetOrbit(st, pl, sat) != NULL) {
			ret += DrawSatLine(sys, st, pl, sat, count,
							   start_count, end_count);
			sat++;
		}
	}
	return(ret);
}


// draw a star and all of _it's_ orbits
int
SystemDisplay::DrawStarLine(System *sys, STAR_INDEX st, int *count,
							int start_count, int end_count)
{
BaseOrbit *bo;
int pl=0,ret=-1;
int orb_type,st1;

    if(st == SI_PRIMARY)
	    st1 = (int) SI_1ST_COMP;
	else
		st1 = (int) st + 1;
		
	if((bo = sys->GetOrbit(st, -1, -1)) != NULL) {
		ret = DrawLine(sys, st, -1, -1, count,
						   start_count, end_count);  

		while((bo = sys->GetOrbit(st, pl, -1)) != NULL) {
			orb_type = bo->GetOrbType();
			if(orb_type & ORBIT_IS_STAR) {
				ret += DrawStarLine(sys, (STAR_INDEX) st1, count,
										start_count, end_count);
				st1 =+ 2;
			} else if(orb_type & ORBIT_IS_WORLD) {
				ret += DrawWorldLine(sys, st, pl, count, start_count,
										 end_count);
			}
			pl++;
		}
	}
	return(ret);
}

// just draw the indicated line
int
SystemDisplay::DrawLine(System *sys, STAR_INDEX st, int pl, int sat,
							int *count, int start_count, int end_count)
{
char buff[120];
int ret=0;

		if((sys->GetListing(st, pl, sat, buff) != NULL) && (buff[0] != 0)) {
		    if((*count >= start_count) && (*count <= end_count)) {
				strstrip(buff, FALSE);
				Line(buff);
				ret++;
			}
			*count++;
		}
	return(ret);
}

// ======================================================================
BEGIN_EVENT_TABLE(ListCanvas, wxTreeCtrl)
	EVT_TREE_SEL_CHANGED(TREE_CTRL, ListCanvas::OnSelChanged)
END_EVENT_TABLE()

ListCanvas::ListCanvas(wxWindow *p, wxFont *f, int d) :
    wxTreeCtrl(p, TREE_CTRL)
{
	image_list = new wxImageList(ICON_HEIGHT, ICON_WIDTH, TRUE);
	image_list->Add(wxICON(star_temp));		// this one's a dummy
	image_list->Add(wxICON(lgg));
	image_list->Add(wxICON(sgg));
	image_list->Add(wxICON(s0_world));
	image_list->Add(wxICON(aa_world));
	image_list->Add(wxICON(h8_world));
	image_list->Add(wxICON(h4a4_world));
	image_list->Add(wxICON(h4a0_world));
	image_list->Add(wxICON(h0a4_world));
	image_list->Add(wxICON(h0a0_world));
	image_list->Add(wxICON(star_temp));
	image_list->Add(wxICON(star_temp));
	image_list->Add(wxICON(star_temp));
	image_list->Add(wxICON(star_temp));
	image_list->Add(wxICON(star_temp));
	SetImageList(image_list);

}

ListCanvas::~ListCanvas()
{
	delete image_list;
}

void
ListCanvas::OnSelChanged(wxTreeEvent& event)
{
	frame->UpdateCurrentOrbit(event.GetItem());
}

// ======================================================================
enum {
	T_ID_CONTINUE=500,
	T_ID_CANCEL
};

BEGIN_EVENT_TABLE(TempFrame, wxFrame)
	EVT_BUTTON(T_ID_CONTINUE, TempFrame::Ok)
	EVT_BUTTON(T_ID_CANCEL, TempFrame::Cancel)
END_EVENT_TABLE()

TempFrame::TempFrame(char *filename) :
	wxFrame(NULL, -1, "System: File Not Present", 
		wxPoint(-1, -1), wxSize(-1,-1), wxCAPTION | wxTHICK_FRAME)
{
wxStaticText *msgs[2];
wxBoxSizer *bs[2];

	dirs = new MTUFile();
	msgs[0] = new wxStaticText(this, -1, filename);
	msgs[1] = new wxStaticText(this, -1, "does not exist.  Create?");
	cmds[0] = new wxButton(this, T_ID_CONTINUE, "Yes");
	cmds[1] = new wxButton(this, T_ID_CANCEL, "No");

	cmds[1]->SetDefault();

	bs[0] = new wxBoxSizer(wxVERTICAL);
	bs[1] = new wxBoxSizer(wxHORIZONTAL);

	bs[0]->Add(msgs[0], 0, wxALL, DLG_OFFSET);
	bs[0]->Add(msgs[1], 0, wxALL, DLG_OFFSET);
	bs[0]->Add(bs[1], 0, wxALL, DLG_OFFSET);
	bs[1]->Add(cmds[0], 0, wxALL, DLG_OFFSET);
	bs[1]->Add(cmds[1], 0, wxALL, DLG_OFFSET);

	SetAutoLayout(TRUE);
	SetSizer(bs[0]);
	bs[0]->Fit(this);
	bs[0]->SetSizeHints(this);
	Layout();
}

TempFrame::~TempFrame()
{
}

void 
TempFrame::Ok(wxCommandEvent& event)
{
bool q;
char lng[256];
SysgenDialog *dlg;
wxConfig *config;

	config = dirs->GetConfig();
	// users do the darndest things.....
	cmds[0]->Enable(FALSE);
	cmds[1]->Enable(FALSE);

	// XXX get lang and interaction from user
	dlg = new SysgenDialog(this);
	lng[0] = 0;
	q = FALSE;
	if(dlg->GetSysData(lng, &q)) {
		int i,cmd_len=0;
		char *buff,str[MAX_FILE_LENGTH];
		wxConfig *config;
		wxString dir,prg;

		for(i = 1;i < (app->argc);i++) {
			cmd_len += (strlen(app->argv[i]) + 1);
		}

		if(lng[0] != 0)
			cmd_len += (strlen(lng) + 4);
		if(q)
			cmd_len += 3;

		cmd_len += 4;		// for " -E\0"

		config = new wxConfig((const wxString) RESOURCE_FILE);
		wxConfigBase::Set(config);
		wxConfigBase::Get();

		sprintf(str, "/%s/%s", SECTION_EXECUTE, ENTRY_SYSGEN);
		if(config->Read((const wxString) str, &dir))
			cmd_len += (strlen(dir.GetData()) + 1);
		sprintf(str, "/%s/%s", SECTION_EXECUTE, ENTRY_DIR);
		if(config->Read((const wxString) str, &prg))
			cmd_len += (strlen(prg.GetData()) + 1);

		buff = new char[cmd_len];
		strcpy(buff, prg.GetData());
		strcat(buff, "/");
		strcat(buff, dir.GetData());

		for(i = 1;i < (app->argc);i++) {
			strcat(buff, " ");
			strcat(buff, app->argv[i]);
		}
		if(lng[0] != 0) {
			strcat(buff, " -L ");
			strcat(buff, lng);
		}
		if(!q)
			strcat(buff, " -q");


		strcat(buff, " -E");		// re-exec ourself

		wxExecute(buff, FALSE);
	}

	Close(TRUE);
}

void 
TempFrame::Cancel(wxCommandEvent& event)
{
	Close(TRUE);
}

// ======================================================================

// ======================================================================

