/*
 * File:		syst_dlg.cpp
 * Purpose:		get system params from user
 * Author:	
 * Created:	
 * Updated:	
 * Copyright: LGPL.
 *            Traveller is a registered trademark of Far Future Enterprises.	
 */

/* rcsid[] = "$RCSfile: syst_dlg.cpp,v $ $Revision: 1.13 $ $Author: man $ $Date: 2003/05/04 15:12:39 $" */

// For compilers that support precompilation, includes "wx.h".
#include "wx/wxprec.h"

#ifndef WX_PRECOMP
#include "wx/wx.h"
#endif

#include "syst_dlg.h"
#include "system.h"

#include "str_util.h"
#include "wx/printdlg.h"
#include "wx/image.h"

#define DUMMY_BUFFER_SIZE			120

#define DUMMY_INIT_STR      " XXXXXXXXXXXXXX "

enum {
	ID_CODES_BTN=700,
	ID_SETUP_BTN,
	ID_LANG_BTN,
	ID_WHICH_CB,
	ID_VFY_BTN
};

#define NUM_PORT_STR        10
#define NUM_SIZE_STR        12
#define NUM_ATMOS_STR       16
#define NUM_HYDRO_STR       11
#define NUM_POP_STR         12
#define NUM_GOVT_STR        17
#define NUM_LAW_STR         22

#define STAR_STRING         "Star"
#define GG_STRING           "Gas Giant"
#define BELT_STRING         "Belt"
#define RING_STRING         "Ring"
#define WORLD_STRING        "World"
#define SAT_STRING          "Satillite"

extern SystemFrame *frame;

// ---------------------------------------------------------------------------
// common strings used
#if 0
static char *uwp_desc[8] = {
  "Port",
  "Size",
  "Atmos",
  "Hydro",
  "Pop",
  "Govt",
  "Law",
  "Tech" };
#endif
static wxString gg_size_str[2] = {
    "Small",
	"Large" };

static wxString dummy_star_str[5] = {
  "         ",
  "         ",
  "         ",
  "         ",
  "         " };

// ===============================================================
SystemDialog::SystemDialog(wxWindow *p) :
  StdDialog(p, "Edit System Parameters")
{
wxStaticText *msg[4];
wxBoxSizer *bs;
wxFlexGridSizer *gs;

	msg[0] = new wxStaticText(this, -1, "Name:");
	msg[1] = new wxStaticText(this, -1, "Sector:");
	msg[2] = new wxStaticText(this, -1, "SubSector:");
	msg[3] = new wxStaticText(this, -1, "Location:");

	name = new wxTextCtrl(this, -1, DUMMY_INIT_STR, 
			wxPoint(-1, -1),
			wxSize(COL_OFFSET * 4, -1));
	name->SetValue("");
	sect = new wxTextCtrl(this, -1, DUMMY_INIT_STR, 
			wxPoint(-1, -1),
			wxSize(COL_OFFSET * 4, -1));
	sect->SetValue("");
	ss = new wxTextCtrl(this, -1, DUMMY_INIT_STR, 
			wxPoint(-1, -1),
			wxSize(COL_OFFSET * 4, -1));
	ss->SetValue("");
	loc = new wxTextCtrl(this, -1, DUMMY_INIT_STR, 
			wxPoint(-1, -1),
			wxSize(COL_OFFSET * 4, -1));
	loc->SetValue("");

	bs = new wxBoxSizer(wxVERTICAL);
	gs = new wxFlexGridSizer(4, 2, 0, 0);
	gs->Add(msg[0], 0, wxALL | wxALIGN_BOTTOM, DLG_OFFSET/2);
	gs->Add(name, 0, wxALL, DLG_OFFSET/2);
	gs->Add(msg[1], 0, wxALL | wxALIGN_BOTTOM, DLG_OFFSET/2);
	gs->Add(sect, 0, wxALL, DLG_OFFSET/2);
	gs->Add(msg[2], 0, wxALL | wxALIGN_BOTTOM, DLG_OFFSET/2);
	gs->Add(ss, 0, wxALL, DLG_OFFSET/2);
	gs->Add(msg[3], 0, wxALL | wxALIGN_BOTTOM, DLG_OFFSET/2);
	gs->Add(loc, 0, wxALL, DLG_OFFSET/2);
	bs->Add(gs, 0, wxALL, DLG_OFFSET);
	DoLayout(wxALL, DLG_OFFSET, bs);
}

bool
SystemDialog::GetSystemParams(char *n, char *st, char *sst, char *l)
{
    name->SetValue(n);
	loc->SetValue(l);
	sect->SetValue(st);
	ss->SetValue(sst);
	
	if(wxID_CANCEL == ShowModal())
	    return(FALSE);

	strcpy(n, name->GetValue().GetData());
	strcpy(l, loc->GetValue().GetData());
	strcpy(st, sect->GetValue().GetData());
	strcpy(sst, ss->GetValue().GetData());

	return(TRUE);
}

// ===============================================================
EditStarDialog::EditStarDialog(wxWindow *p) :
    StdDialog(p, "Edit Star")
{
wxStaticText *msg;

	msg = new wxStaticText(this, -1, "Name:");

	name = new wxTextCtrl(this, -1, DUMMY_INIT_STR, 
			wxPoint(-1, -1),
			wxSize((int) (4 * COL_OFFSET), -1));
	name->SetValue("");

	stars = new StarBox(this);

	wxBoxSizer* sizer_16 = new wxBoxSizer(wxVERTICAL);
    wxStaticBoxSizer* sizer_18 = stars->GetSizer();
    wxBoxSizer* sizer_17 = new wxBoxSizer(wxHORIZONTAL);
    sizer_17->Add(msg, 0, wxALL|wxALIGN_BOTTOM, DLG_OFFSET);
    sizer_17->Add(name, 0, wxALL, DLG_OFFSET);
    sizer_16->Add(sizer_17, 0, 0, 0);
    sizer_16->Add(sizer_18, 0, wxALL, DLG_OFFSET);
    DoLayout(wxALL, DLG_OFFSET, sizer_16);
}

bool
EditStarDialog::GetStar(char *n, int *ty, int *cl, int *sz)
{
	name->SetValue(n);
	stars->SetStar(ty, cl, sz);

	if(wxID_CANCEL == ShowModal())
	    return(FALSE);

	strcpy(n, name->GetValue());
	stars->GetStar(ty, cl, sz);
	return(TRUE);
}

// ===============================================================
EditGGDialog::EditGGDialog(wxWindow *p) :
    StdDialog(p, "Edit Gas Giant")
{    
wxStaticText *msg[2];
wxBoxSizer *bs;
wxFlexGridSizer *gs;

	msg[0] = new wxStaticText(this, -1, "Name:");
	msg[1] = new wxStaticText(this, -1, "Size:");

 	name = new wxTextCtrl(this, -1, DUMMY_INIT_STR, 
			wxPoint(-1, -1),
			wxSize(3 * COL_OFFSET, -1));
	name->SetValue("");
	gg = new wxChoice(this, -1,
			wxPoint(-1, -1),
			wxSize((int)(COL_OFFSET * 2.5), -1),
			2, gg_size_str);

	bs = new wxBoxSizer(wxVERTICAL);
	gs = new wxFlexGridSizer(2, 2, 0, 0);
	gs->Add(msg[0], 0, wxALL | wxALIGN_BOTTOM, DLG_OFFSET/2);
	gs->Add(name, 0, wxALL, DLG_OFFSET/2);
	gs->Add(msg[1], 0, wxALL | wxALIGN_BOTTOM, DLG_OFFSET/2);
	gs->Add(gg, 0, wxALL, DLG_OFFSET/2);
	bs->Add(gs, 0, wxALL, DLG_OFFSET);
	DoLayout(wxALL, DLG_OFFSET, bs);
}

bool
EditGGDialog::GetGG(char *n, int *g)
{
    name->SetValue(n);
    gg->SetSelection(*g);

	if(wxID_CANCEL == ShowModal())
	    return(FALSE);

	strcpy(n, name->GetValue());
	*g = gg->GetSelection();
	return(TRUE);
}

// ===============================================================
BEGIN_EVENT_TABLE(EditWorldDialog, wxDialog)
	EVT_BUTTON(ID_CODES_BTN, EditWorldDialog::DoCodes)
	EVT_BUTTON(ID_VFY_BTN, EditWorldDialog::DoVerify)
END_EVENT_TABLE()

// ---------------------------------------------------------------
EditWorldDialog::EditWorldDialog(wxWindow *p, CodeTable *ct) :
    StdDialog(p, "Edit World")
{
wxStaticText *msg[4];
wxButton *btn[2];

	uwp_box = new UWPBox(this, "UWP");

	msg[0] = new wxStaticText(this, -1, "Name:");
	name = new wxTextCtrl(this, -1, DUMMY_INIT_STR,
			wxPoint(-1, -1),
			wxSize(4 * COL_OFFSET, -1));
	name->SetValue("");

	btn[0] = new wxButton(this, ID_VFY_BTN, "Verify UWP");
	btn[0]->Enable(FALSE);
	btn[1] = new wxButton(this, ID_CODES_BTN, "Codes");

	msg[1] = new wxStaticText(this, -1, "Bases");
	bases = new CodeList(this, TRUE);

	msg[2] = new wxStaticText(this, -1, "Allegance");
	alleg = new CodeList(this, TRUE);

	msg[3] = new wxStaticText(this, -1, "Zone:");
	zone = new ZoneChoice(this, TRUE);

	wxBoxSizer* sizer_8 = new wxBoxSizer(wxVERTICAL);
    wxBoxSizer* sizer_9 = new wxBoxSizer(wxHORIZONTAL);
    wxBoxSizer* sizer_13 = new wxBoxSizer(wxVERTICAL);
    wxBoxSizer* sizer_15 = new wxBoxSizer(wxHORIZONTAL);
    wxFlexGridSizer* grid_sizer_3 = new wxFlexGridSizer(2, 2, 0, 0);
    wxBoxSizer* sizer_14 = new wxBoxSizer(wxHORIZONTAL);
    wxBoxSizer* sizer_10 = new wxBoxSizer(wxVERTICAL);
    wxStaticBoxSizer* sizer_11 = uwp_box->GetSizer();
    sizer_10->Add(sizer_11, 0, wxALL, DLG_OFFSET);
    sizer_10->Add(btn[0], 0, wxALL|wxEXPAND, DLG_OFFSET);
    sizer_9->Add(sizer_10, 0, 0, 0);
    sizer_14->Add(msg[0], 0, wxALL|wxALIGN_BOTTOM, DLG_OFFSET);
    sizer_14->Add(name, 0, wxALL, DLG_OFFSET);
    sizer_14->Add(btn[1], 0, wxALL, DLG_OFFSET);
    sizer_13->Add(sizer_14, 0, 0, 0);
    grid_sizer_3->Add(msg[1], 0, wxALL|wxALIGN_BOTTOM, DLG_OFFSET/2);
    grid_sizer_3->Add(msg[2], 0, wxALL|wxALIGN_BOTTOM, DLG_OFFSET/2);
    grid_sizer_3->Add(bases, 0, wxALL, DLG_OFFSET/2);
    grid_sizer_3->Add(alleg, 0, wxALL, DLG_OFFSET/2);
    sizer_13->Add(grid_sizer_3, 0, wxEXPAND, 0);
    sizer_15->Add(msg[3], 0, wxALL|wxALIGN_BOTTOM, DLG_OFFSET);
    sizer_15->Add(zone, 0, wxALL, DLG_OFFSET);
    sizer_13->Add(sizer_15, 0, 0, 0);
    sizer_9->Add(sizer_13, 0, 0, 0);
    sizer_8->Add(sizer_9, 0, 0, 0);
    DoLayout(wxALL, DLG_OFFSET, sizer_8);

	code_dlg = new EditCodeDialog(this, ct);
//	err_dlg = new ErrorDialog(this);

	bo = NULL;
}

bool
EditWorldDialog::GetWorld(BaseOrbit *b_orb, BaseTable *bt, CodeTable *at)
{
planet *p;
unsigned long d1,d2;
char b,z,*a,buff[120],*ptr;

	// save a local/global copy
    bo = b_orb;
    if(bo->GetOrbType() & ORBIT_IS_SAT) {
		SatOrbit *so;
		
		so = (SatOrbit *)bo;
		p = so->GetSat();
	} else {
		WorldOrbit *wo;
		
		wo = (WorldOrbit *)bo;
		p = wo->GetWorld();
	}

	// make the uwp local
	luwp[0] = p->get_port();
	luwp[1] = p->get_size();
	luwp[2] = p->get_atmos();
	luwp[3] = p->get_hydro();
	luwp[4] = p->get_pop();
	luwp[5] = p->get_govt();
	luwp[6] = p->get_law();
	luwp[7] = p->get_tech();
	luwp[8] = 0;

    if(bo->GetOrbType() & ORBIT_IS_SAT) {
		uwp_box->SetUWP(luwp, UI_SYS_SAT);
	}
	else {
		uwp_box->SetUWP(luwp, UI_SYS_PLANET);
	}

	// grab the name
	p->GetName(buff);
	name->SetValue(buff);
				   
	// point to the right base (if any)
	b = p->GetBase();
	sprintf(buff, "%c", b);
	bases->SetCode(buff, bt);

	// point to the right alleg (if any)
	a = p->GetAlleg();
	alleg->SetCode(a, at);

	// point to the right zone (if any)
	zone->SetZone(z);

	// grab the local codes for the codes dialog box
	bo->GetCodes(&d1, &code_val, &d2);

	// get the input
	if(wxID_CANCEL == ShowModal())
	    return(FALSE);

	// fell thru, copy stuff back
	// adjust the uwp
	uwp_box->GetUWP(p);
	// grab the name
	p->SetName((char *) name->GetValue().GetData());
	
	// grab bases
	ptr = bases->GetCode();
	p->SetBase(ptr[0]);

	// grab alleg
	p->SetAlleg(alleg->GetCode());

	// grab zone
	p->SetZone(zone->GetZone());

	// grab codes
	p->SetSysFlags(code_val);
	
	return(TRUE);
}

#if 0
// ---------------------------------------------------------------------
// this gets a bit hokey.....
//	This retrofit of main worlds (worlds from a sector file) onto a
//	Book 6 system causes some real headaches here.  The best that
//  can be done is to trap the port type and assume that if a 
//	starport, this was a main world.
short
EditWorldDialog::ErrorCheck(char *user_uwp, char *adj_uwp)
{
unsigned short ret;
int i;
int uwp_vals[MAX_UWP_CHOICES];

	// convert to ints (makes life simpler)
	uwp_vals[0] = user_uwp[0];
	if((user_uwp[1] >= '0') || (user_uwp[1] <= '9'))
		uwp_vals[1] = user_uwp[1] - '0';
	else if('A' == user_uwp[1])
		uwp_vals[1] = user_uwp[1] - 'A' + 10;
	// deviate heavily here.....
	else if('S' == user_uwp[1])
		uwp_vals[1] = -1;
	else if('R' == user_uwp[1])
		uwp_vals[1] = -2;
	else
		uwp_vals[1] = -10;

	for(i = 2;i < MAX_UWP_CHOICES;i++) {
		if(' ' == user_uwp[i])
			uwp_vals[i] = -1;
		else if(user_uwp[i] < 'A')
			uwp_vals[i] = user_uwp[i] - '0';
		else
			uwp_vals[i] = user_uwp[i] - 'A' + 10;
	}

	if(strchr("ABCDEX", user_uwp[0]) != NULL)
		ret = CheckWorld(uwp_vals);
	else
	    ret = CheckPlanet(uwp_vals);

	if(ret != 0) {
		adj_uwp[0] = (char) uwp_vals[0];
		if(uwp_vals[1] < -1) adj_uwp[1] = 'R';
		else if(uwp_vals[1] < 0) adj_uwp[1] = 'S';
		else if(uwp_vals[1] < 10) adj_uwp[1] = (char) uwp_vals[1] + '0';
		else adj_uwp[1] = (char) uwp_vals[1] + 'A' - 10;
		for(i = 2;i < MAX_UWP_CHOICES;i++) {
			if(uwp_vals[i] < 0) adj_uwp[i] = ' ';
			else if(uwp_vals[i] < 10) adj_uwp[i] = uwp_vals[i] + '0';
			else adj_uwp[i] = uwp_vals[i] + 'A' - 10;
		}
	}

	return(ret);
}
#endif

// check an array of 'main world' uwp's
short
EditWorldDialog::CheckWorld(int *u)
{
short ret=0;
int min,max;

	// the infamous shouldn't happen case:
	if(u[1] < -1) {
		ret = CheckRing(u);
	} else {
		// atmos
		if(0 == u[1]) {				// astroid belt
			if(u[2] != -1) {
				u[2] = -1;
				ret |= UWP_MASK_ATMOS;
			}
			if(u[3] != 0) {
				u[3] = 0;
				ret |= UWP_MASK_HYDRO;
			}
		} else {
			if(-1 == u[1]) {		// size small
				min = 0;
				max = 5;
			} else {
				min = 2 - 7 + 10;
				max = 12 - 7 + 10;
			}
			if(!CheckUWP(&u[2], min, max))
				ret |= UWP_MASK_ATMOS;

			// hydro
			min = 2 - 7 + u[2];
			max = 12 - 7 + u[2];
			if((u[2] < 2) || (u[2] > 9)) {
				min -= 4;
				max -= 4;
			}
			if(!CheckUWP(&u[3], min, max))
				ret |= UWP_MASK_HYDRO;
		}

		// general pop check
		ret |= CheckNoPop(u);
		if(u[4] > -1) {
			// check govt
			if(!CheckUWP(&u[5], 2 - 7 + u[4], 12 - 7 + u[4]))
				ret |= UWP_MASK_GOVT;

			// check law
			if(!CheckUWP(&u[6], 2 - 7 + u[5], 12 - 7 + u[5]))
				ret |= UWP_MASK_LAW;

			// check tech
			min = 1;
			max = 6;
			if(u[0] == 'A') {
				min += 6;
				max += 6;
			}
			if(u[0] == 'B') {
				min += 4;
				max += 4;
			}
			if(u[0] == 'C') {
				min += 2;
				max += 2;
			}
			if(u[0] == 'X') {
				min -= 4;
				max -= 4;
			}
			if(u[1] < 2) {
				min += 2;
				max += 2;
			} else if(u[1] < 5) {
				min += 1;
				max += 1;
			}
			if((u[2] < 4) || (u[2] > 9)) {
				min += 1;
				max += 1;
			}
			if(u[3] == 9) {
				min += 1;
				max += 1;
			}
			if(u[3] == 10) {
				min += 2;
				max += 2;
			}
			if(u[4] < 6) {
				min += 1;
				max += 1;
			}
			if(u[4] == 9) {
				min += 2;
				max += 2;
			}
			if(u[4] == 10) {
				min += 4;
				max += 4;
			}
			if((u[5] == 0) || (u[5] == 5)) {
				min += 1;
				max += 1;
			}
			if(u[5] == 13) {
				min -= 2;
				max -= 2;
			}
			if(!CheckUWP(&u[7], min, max))
				ret |= UWP_MASK_TECH;
		}
	}
   
	return(ret);
}

// check an array of 'planet' uwp's
// returns TRUE if nothing was changed
// NOTE:
//    size 0 (asteroid belt) is never checked and
//    size R and S are temporarly 0 and -1
// NOTE2:
//    I've departed from the books a little bit
//    Aside from the use of ' ' for _no_ population
//    (10^0 = 1 => 1-9 people NOT 0 people),
//    I've tried to make satillites and worlds
//    more consistant.
short
EditWorldDialog::CheckPlanet(int *u)
{
short min,max;
int zone,star_type;
float orb;
short ret = 0;
BaseOrbit *par,*star;

	if(u[1] < -1)
		return(CheckRing(u));

	orb = bo->GetOrbit();
	zone = bo->GetHabitible();
	par = bo->GetParent();
	star = par;
	while(!star->IsStar())
		star = star->GetParent();
	star_type = ((StarOrbit *)star)->GetStar()->GetStarType();

	if(!bo->IsBelt()) {
		min = 0;
		max = 10;
		if(bo->GetOrbType() & ORBIT_IS_SAT) {
			int parent_size;

			parent_size = ((WorldOrbit *)par)->GetWorld()->get_size();
			if(parent_size == SIZE_LARGE_GG) {
				min = -2;
				max = 8;
			} else if(parent_size == SIZE_SMALL_GG) {
				min = -4;
				max = 6;
			} else if(parent_size == SIZE_SMALL) {
				min = -1;
				max = 0;
			} else {
				min = parent_size - 6;
				max = parent_size - 1;
			}
		} else {
			if(star_type == 6)
				max -= 2;
			if(orb <= 0.5)
				max -= 5;
			else if(orb <= 1.5)
				max -= 4;
			else if(orb <= 2.5)
				max -= 2;
		}
		if(!CheckUWP(&u[1], min, max)) {
			if(u[1] < 1) u[1] = -1;
			ret |= UWP_MASK_SIZE;
		}
	} 

	if(0 == u[1]) {			// astroid belt
		if(u[2] != -1) {
			u[2] = -1;
			ret |= UWP_MASK_ATMOS;
		}
		if(u[3] != 0) {
			u[3] = 0;
			ret |= UWP_MASK_HYDRO;
		}
	} else {
		// check atmos
		if(u[1] > 10)
			min = max = 0;
		else {
			min = 2 - 7 + u[1];
			max = 12 - 7 + u[1];
			if(zone < HAB_HABIT) {
				min -= 2;
				max -= 2;
			}
			if(zone > HAB_HABIT) {
				min -= 4;
				max -= 4;
			}
		}
	   // &#^$&*^$ special case!
	   if(!((zone > HAB_OUTER) && (u[2] == 0) && (u[2] == 10))) {
		   if(!CheckUWP(&u[2], min, max))
			   ret |= UWP_MASK_ATMOS;
	   }

	   // check hydro
	   if((u[1] > 10) || (zone < HAB_HABIT))
		   min = max = 0;
	   else {
		   min = 2 - 7 + u[1];
		   max = 12 - 7 + u[1];
		   if(zone > HAB_HABIT) {
			   min -= 2;
			   max -= 2;
			   }
		   if(u[2] > 9) {
			   min -= 4;
			   max -= 4;
			   }
		   }
	   if(min < 0)
		   min = 0;
	   if(max > 0)
		   max = 0;
	   
	   if(!CheckUWP(&u[3], min, max))
		   ret |= UWP_MASK_HYDRO;	   
	}

	ret |= CheckNoPop(u);

	if(u[4] > -1) {
		// check pop
		min = 0;
		max = 10;
		if(zone < HAB_HABIT) {
			min -= 5;
			max -= 5;
		}
		if(zone > HAB_HABIT) {
			min -= 3;
			max -= 3;
		}
		if((u[2] != 0) && (u[2] != 5) && (u[2] != 6) && 
				(u[2] != 8)) {
			min -= 2;
			max -= 2;
		}
		if(!CheckUWP(&u[4], min, max))
			ret |= UWP_MASK_POP;

		// check govt
		if((u[5] > 6) || (u[5] == 4) || (u[5] == 5)) {
			u[5] = 6;
			ret |= UWP_MASK_GOVT;
		}
		// by Book 6, there is no real way to check law and tl
		//  (altough ridiculas values less than 0 are checked
	}

   // check port
   min = 1;
   max = 6;
   if(u[4] > 5) {
	   min += 2;
	   max += 2;
   }
   if(u[4] < 0) {
	   min -= 3;
	   max -= 3;
   }
   else if(u[4] < 2) {
	   min -= 2;
	   max -= 2;
   }
   switch(min) {
	   case 3:
		   if(u[0] > 'H') {
			   u[0] = 'H';
			   ret |= UWP_MASK_PORT;
		   }
		   break;
	   case 4:
	   case 5:
		   if(u[0] > 'G') {
			   u[0] = 'G';
			   ret |= UWP_MASK_PORT;
		   }
		   break;
	   case 6:
	   case 7:
	   case 8:
		   if(u[0] != 'F') {
			   u[0] = 'F';
			   ret |= UWP_MASK_PORT;
		   }
		   break;
	   default:
		   break;
	}
    switch(max) {
	   case -2:
	   case -1:
	   case 0:
	   case 1:
	   case 2:
		   if(u[0] != 'Y') {
				u[0] = 'Y';
				ret |= UWP_MASK_PORT;
		    }
		   break;
	   case 3:
		   if(u[0] < 'X') {
			   u[0] = 'X';
			   ret |= UWP_MASK_PORT;
		   }
		   break;
	   case 4:
	   case 5:
		   if(u[0] < 'H') {
			   u[0] = 'H';
			   ret |= UWP_MASK_PORT;
		   }
		   break;
	   default:
		   break;
	}
	 
	return(ret);
}

// nice simple case: rings
short
EditWorldDialog::CheckRing(int *u)
{
int ret = 0;

	if(u[2] != 0) {
		u[2] = 0;
		ret |= UWP_MASK_ATMOS;
	}
	if(u[3] != 0) {
		u[3] = 0;
		ret |= UWP_MASK_HYDRO;
	}
	if(u[4] != -1) {
		u[4] = -1;
		ret |= UWP_MASK_POP;
	}
	if(u[5] != -1) {
		u[5] = -1;
		ret |= UWP_MASK_GOVT;
	}
	if(u[6] != -1) {
		u[6] = -1;
		ret |= UWP_MASK_LAW;
	}
	if(u[7] != -1) {
		u[7] = -1;
		ret |= UWP_MASK_TECH;
	}

	return(ret);
}

// just makes sure the pop, govt, law and tech are consistantly
//  all -1 or not -1
short
EditWorldDialog::CheckNoPop(int *u)
{
short ret=0;
short mask=UWP_MASK_GOVT;
int i;

    if(u[4] < 0) {
		for(i = 5; i < 8;i++) {
	  		if(u[i] > -1) {
				u[i] = -1;
				ret |= mask;
			}
			mask *= 2;
		}
	} else {
		for(i = 5; i < 8;i++) {
	  		if(u[i] < 0) {
				u[i] = 0;
				ret |= mask;
			}
			mask *= 2;
		}
	}

	return(ret);
}

// generic check of val against min and max values
bool
EditWorldDialog::CheckUWP(int *val, int min, int max)
{
bool ret=TRUE;

    if(*val < min) {
		*val = min;
		ret = FALSE;
	}
	if(*val > max) {
		*val = max;
		ret = FALSE;
	}

	return(ret);
}

void
EditWorldDialog::DoCodes(wxCommandEvent& event)
{
    code_dlg->GetCodes(&code_val);
}

void
EditWorldDialog::DoVerify(wxCommandEvent& event)
{
	uwp_box->CheckUWP();
#if 0
char user_uwp[MAX_UWP_CHOICES+1],adj_uwp[MAX_UWP_CHOICES+1];
unsigned short vfy;

	uwp_box->GetUWP(user_uwp);
	uwp_box->GetUWP(adj_uwp);

	if((vfy = ErrorCheck(user_uwp, adj_uwp)) != 0) {
		ERROR_RESP er;
		char buff[DUMMY_BUFFER_SIZE];
		unsigned short mask=UWP_MASK_PORT;
		int i;
		bool comma=FALSE;

		buff[0] = 0;
		for(i = 0; i < MAX_UWP_CHOICES;i++) {
			if(mask & vfy) {
				if(comma)
				    strcat(buff, ", ");
				strcat(buff, uwp_desc[i]);
				comma = TRUE;
			}
			mask *= 2;
		}
		err_dlg->info->SetLabel(buff);

		er = err_dlg->GetResp();
		if(ER_OK == er) {
			// set controls to adj_uwp
			uwp_box->StrToChoice(adj_uwp);
		} else if(ER_RESET == er) {
			// set controls to original
			uwp_box->StrToChoice(luwp);
		} else {
			// leave controls as user_uwp ie. do nothing
		}
	} else {
		wxMessageBox("UWP has been verified.", "UWP verify.",
					 wxOK | wxCENTRE | wxICON_INFORMATION, this);
	}
#endif
}

// ===============================================================
MoveDialog::MoveDialog(wxWindow *p) :
  StdDialog(p, "Move Orbit To:")
{
wxStaticText *msg[2];
wxBoxSizer *bs;
wxFlexGridSizer *gs;

	msg [0] = new wxStaticText(this, -1, "Star:");
	msg [1] = new wxStaticText(this, -1, "Orbit:");

	star = new wxChoice(this, -1,
			wxPoint(-1, -1),
			wxSize(COL_OFFSET * 2, -1),
			5, dummy_star_str);

	orb = new wxTextCtrl(this, -1, " XXXXX ",
			wxPoint(-1, -1),
			wxSize(COL_OFFSET * 2, -1));
	orb->SetValue("");

	bs = new wxBoxSizer(wxVERTICAL);
	gs = new wxFlexGridSizer(2, 2, 0, 0);
	gs->Add(msg[0], 0, wxALL | wxALIGN_BOTTOM, DLG_OFFSET/2);
	gs->Add(star, 0, wxALL, DLG_OFFSET/2);
	gs->Add(msg[1], 0, wxALL | wxALIGN_BOTTOM, DLG_OFFSET/2);
	gs->Add(orb, 0, wxALL, DLG_OFFSET/2);
	bs->Add(gs, 0, wxALL, DLG_OFFSET);
	DoLayout(wxALL, DLG_OFFSET, bs);
}

bool
MoveDialog::GetMove(System *s)
{
int i,st_count=0,st_index=0;
BaseOrbit *bo,*bo1,*bo2;
StarOrbit *so=NULL;
char buff1[DUMMY_BUFFER_SIZE];

    orb->SetValue("");
	star->Clear();

	// get the current world
	bo2 = bo = s->GetCurrentOrbit();
	if(bo->GetOrbType() & ORBIT_IS_PRIMARY) {  
		wxMessageBox(" Primary orbit can not be moved!", "Move Error",
					 wxOK | wxCENTRE | wxICON_EXCLAMATION);
		return(FALSE);
	}
	
	// find it''s star
	while((bo1 = bo2->GetParent()) != NULL) {
		if(bo1->GetOrbType() & ORBIT_IS_STAR) {
			so = (StarOrbit *)bo1;
			break;
		}
		bo2 = bo1;
	}
	
	// find all the stars
	for(i = 0;i < MAX_STARS;i++) {
		StarOrbit *so1;
		
		if((so1 = (StarOrbit *)s->GetOrbit((STAR_INDEX) i, -1, -1)) != NULL) {
			char buff2[DUMMY_BUFFER_SIZE];

			if((((BaseOrbit *)so1)->GetOrbit() < 0.0) &&
			   (!(so1->GetOrbType() & ORBIT_IS_PRIMARY)))
			    continue;
			sprintf(buff1, " %d:%s ", st_count, so1->FormatUWP(buff2, -1));
			star->Append(buff1);

			if(so1 == so)
			    st_index = st_count;

			st_count++;
		}
	}

	while(TRUE) {
		STAR_INDEX s_ndx;
		char *ptr;
		float val;
		ORBIT_CHECK_RESP ocr;
		int resp;
		
		// only one valid star?
		if(st_count < 2)
		    star->Show(FALSE);
		else {
			star->Show(TRUE);
			star->SetSelection(st_index);
		}
		
		// get the orbit
		sprintf(buff1, "%.1f", bo->GetOrbit());
		orb->SetValue(buff1);

		// do it!
		if(wxID_CANCEL == ShowModal())
		    break;

		// get input
		if(st_count < 1)
		    s_ndx = SI_PRIMARY;
		else {
		    ptr = (char *)star->GetString(star->GetSelection()).GetData();
			s_ndx = (STAR_INDEX) ((ptr[1] - '0') + 1);
		}
		ptr = (char *)orb->GetValue().GetData();
		if(strcasecmp(ptr, "close") == 0)
		    val = -1.0;
		else {
			if(sscanf(ptr, "%f", &val) != 1) {
				wxMessageBox(" Bad orbit value!", "Move Error",
							 wxOK | wxCENTRE | wxICON_EXCLAMATION);
				continue;
			}
		}

		ocr = s->CheckOrbit(s_ndx, val, bo->GetEnumType(), FALSE);
		if(ocr == OC_LOOSE_SAT) {
			resp = wxMessageBox(
					   "This will delete this orbits satillites.  Continue?",
					   "Move Conflict",
					   wxYES_NO | wxCANCEL | wxCENTRE | wxICON_QUESTION);
			if(resp == wxOK)
			    ocr = s->CheckOrbit(s_ndx, val, bo->GetEnumType());
			else if(resp == wxNO)
			    continue;
			else
			    break;
		}

		switch(ocr) {
			case OC_TOO_MANY_STARS:
			    ptr = "Too many star orbits.";
				break;
			case OC_NO_ORB:
			    ptr = "Orbit not available.";
				break;
			case OC_CONFLICTS:
			    ptr = "Star confliction.";
				break;
			case OC_STAR_CONFLICTS:
			    ptr = "Orbit conflicts with existing star.";
				break;
			case OC_NO_SAT:
			    ptr = "Can not be a satillite.";
				break;
			case OC_NOT_A_WORLD:
			    ptr = "Can not add that type of orbit.";
				break;
			default:
			    ptr = NULL;
				break;
		}
				
		if(ptr != NULL)
		    wxMessageBox(ptr, "Move Error",
						 wxOK | wxCENTRE | wxICON_EXCLAMATION);
		else {
			if(s->MoveCurrentOrbit(s_ndx, val))
			    return(TRUE);

			wxMessageBox("Unable to move orbit!", "Move Error",
						 wxOK | wxCENTRE | wxICON_EXCLAMATION);
			break;
		}
	}

	return(FALSE);
}

// ===============================================================
AddDialog::AddDialog(wxWindow *p) :
    StdDialog(p, "Add Orbit At:")
{
wxStaticText *msg[2];
wxBoxSizer *bs;
wxFlexGridSizer *gs;

	msg[0] = new wxStaticText(this, -1, "Orbit:");
	msg[1] = new wxStaticText(this, -1, "Type:");

	orb = new wxTextCtrl(this, -1, " XXXXX ",
			wxPoint(-1, -1),
			wxSize(COL_OFFSET * 2, -1));
	orb->SetValue("");

	typ = new wxChoice(this, -1,
			wxPoint(-1, -1),
			wxSize(COL_OFFSET * 2, -1),
			5, dummy_star_str);

	bs = new wxBoxSizer(wxVERTICAL);
	gs = new wxFlexGridSizer(2, 2, 0, 0);
	gs->Add(msg[0], 0, wxALL | wxALIGN_BOTTOM, DLG_OFFSET/2);
	gs->Add(orb, 0, wxALL, DLG_OFFSET/2);
	gs->Add(msg[1], 0, wxALL | wxALIGN_BOTTOM, DLG_OFFSET/2);
	gs->Add(typ, 0, wxALL, DLG_OFFSET/2);
	bs->Add(gs, 0, wxALL, DLG_OFFSET);
	DoLayout(wxALL, DLG_OFFSET, bs);
}

bool
AddDialog::GetAdd(BaseOrbit *o, STAR_INDEX *si, ORBIT_TYPE *ot, float *orbit)
{
float val;
char *ptr;

    if(o->GetOrbType() & ORBIT_IS_SAT) {
		wxMessageBox("Can not add orbit to Satillite!", "Add Error",
					 wxOK | wxCENTRE | wxICON_EXCLAMATION);
		return(FALSE);
	}
    if(o->IsBelt()) {
		wxMessageBox("Can not add orbit to a Belt!", "Add Error",
					 wxOK | wxCENTRE | wxICON_EXCLAMATION);
		return(FALSE);
	}

	if((!(o->GetOrbType() & ORBIT_IS_PRIMARY)) &&o->IsStar() &&
	   (o->GetOrbit() < 0.5)) {
		wxMessageBox("Can not add orbit to a close companion!", "Add Error",
					 wxOK | wxCENTRE | wxICON_EXCLAMATION);
		return(FALSE);
	}

	typ->Clear();
	if(o->IsStar()) {
		bool star_ok=TRUE;
		
		if(o->GetOrbOrbitType() & ORBIT_ORBITS_COMP)
		    star_ok = FALSE;
		else if(o->GetOrbOrbitType() & ORBIT_ORBITS_PRIMARY) {
			CompOrbit *co = (CompOrbit *)o;

			if(co->GetTert() != NULL)
			    star_ok = FALSE;
		} else {
			PrimOrbit *po = (PrimOrbit *)o;

			if((po->GetComp(0) != NULL) && (po->GetComp(1) != NULL))
			    star_ok = FALSE;
		}
		if(star_ok)							 
		    typ->Append(STAR_STRING);
		typ->Append(GG_STRING);
		typ->Append(BELT_STRING);
		typ->Append(WORLD_STRING);
	} else {
	    typ->Append(SAT_STRING);
	    typ->Append(RING_STRING);
	}
	orb->SetValue("");

	while(TRUE) {
		if(wxID_CANCEL == ShowModal())
		    break;

		ptr = (char *)orb->GetValue().GetData();
		if((o->IsStar()) &&
		   (strcmp(typ->GetStringSelection(), STAR_STRING) == 0) &&
		   (strcasecmp(ptr, "close") == 0))
		    val = -1.0;
		else {
			if((sscanf(ptr, "%f", &val) != 1) && (val < 0.0)) {
				wxMessageBox(" Bad orbit value!", "Move Error",
							 wxOK | wxCENTRE | wxICON_EXCLAMATION);
				continue;
			}
		}

		*si = o->GetStarIndex();
		
		*orbit = val;
		if(strcmp(typ->GetStringSelection(), STAR_STRING) ==0)
		    *ot = OT_STAR;
		else if(strcmp(typ->GetStringSelection(), GG_STRING) ==0)
		    *ot = OT_GG;
		else if(strcmp(typ->GetStringSelection(), BELT_STRING) ==0)
		    *ot = OT_BELT;
		else if(strcmp(typ->GetStringSelection(), WORLD_STRING) ==0)
		    *ot = OT_WORLD;
		else if(strcmp(typ->GetStringSelection(), RING_STRING) ==0)
		    *ot = OT_RING;
		else if(strcmp(typ->GetStringSelection(), SAT_STRING) ==0)
		    *ot = OT_SATILLITE;

		switch(*ot) {
			case OT_STAR:
			case OT_GG:
			case OT_BELT:
			  int resp;
			  
			  resp = wxMessageBox(
							"This may conflict with existing sector data.",
							"Move Warning",
							wxOK | wxCANCEL | wxCENTRE | wxICON_INFORMATION);
			  if(resp == wxOK)
				  return(TRUE);
			  break;
			case OT_SATILLITE:
			case OT_WORLD:
			case OT_RING:
			    return(TRUE);
			default:
			    wxMessageBox("Something's not right!", "Move Error",
							 wxOK | wxCENTRE | wxICON_EXCLAMATION);
				break;
		}
	}

	return(FALSE);
}

// ===========================================================================
BEGIN_EVENT_TABLE(SysgenDialog, wxDialog)
	EVT_BUTTON(ID_LANG_BTN, SysgenDialog::LangFcn)
END_EVENT_TABLE()

// ---------------------------------------------------------------------------
SysgenDialog::SysgenDialog(wxWindow *p) :
	StdDialog(p, "System Configuration")
{
	temp_lang = NULL;
	quiet = new wxCheckBox(this, -1, "Interactive");
	lang_sel = new wxButton(this, ID_LANG_BTN, "Change",
			wxPoint(-1, -1),
			wxSize(-1, -1));
	lang = new wxStaticText(this, -1, "Language: XXXXXXXXXXXXXXXXXXXXXXXXXXXX",
			wxPoint(-1, -1),
			wxSize(DLG_OFFSET * 20, -1));

	sizer = new wxBoxSizer(wxVERTICAL);
	sizer->Add(quiet, 0, wxALL, DLG_OFFSET);
	sizer->Add(lang_sel, 0, wxALL, DLG_OFFSET);
	sizer->Add(lang, 0, wxALL | wxEXPAND, DLG_OFFSET);
	DoLayout(wxALL, DLG_OFFSET, sizer);
}

SysgenDialog::~SysgenDialog()
{
}

void
SysgenDialog::LangFcn(wxCommandEvent& event)
{
    if(temp_lang == NULL) {
		wxString str;
		char *ptr;

		str = wxFileSelector("Language File", NULL, NULL, NULL, "*.lng",
				 0, this);
		ptr = (char *)str.GetData();
		if(ptr[0] != 0) {
			char buff[120];

			temp_lang = new char[strlen(ptr) + 1];
			strcpy(temp_lang, ptr);

			sprintf(buff, "Language: %s", temp_lang);
			lang->SetLabel(buff);
			lang->SetSize(-1, -1, 300, -1, wxSIZE_USE_EXISTING);
			lang_sel->SetLabel("Clear");
			Fit();
		}
	} else {
		delete temp_lang;
		temp_lang = NULL;
		lang->SetLabel("Language: <None>");
		lang_sel->SetLabel("Change");
	}
	sizer->Fit(this);
	sizer->SetSizeHints(this);
	Layout();
}

bool
SysgenDialog::GetSysData(char *lng, bool *q)
{
char buff[DUMMY_BUFFER_SIZE];

	temp_lang = NULL;
	if((lng != NULL) && (lng[0] != 0)) {
		temp_lang = new char[strlen(lng) + 1];
		strcpy(temp_lang, lng);
		sprintf(buff, "Language: %s", lng);
	} else
		sprintf(buff, "Language: <None>");
	lang->SetLabel(buff);
	quiet->SetValue(*q);

	if(ShowModal() == wxID_CANCEL) {
		if(NULL == temp_lang)
			delete temp_lang;
		return(FALSE);
	}

	if((temp_lang != NULL) && (temp_lang[0] != 0))
		strcpy(lng, temp_lang);
	else
		lng[0] = 0;
	if(NULL == temp_lang)
		delete temp_lang;
	*q = quiet->GetValue();

	return(TRUE);
}

// ============================================================
#define MAPS_PER_PAGE        6
#define MAP_BORDER_SIZE      5
#define MAP_BORDER_OFFSET    2
#define MAP_TEXT_OFFSET     20

#define MAP_X_OFFSET1       90
#define MAP_X_OFFSET2      890
#define MAP_Y_OFFSET1      200
#define MAP_Y_OFFSET2      800
#define MAP_Y_OFFSET3     1400
#define MAP_Y_TEXT1         20
#define MAP_Y_TEXT2       2010

SystemPrintOut::SystemPrintOut(PRINT_TYPE t, PRINT_SOURCE s, char *msg) :
	wxPrintout()
{
	if(msg == NULL)
		header_msg = NULL;
	else {
		header_msg = new char[strlen(msg) + 1];
		strcpy(header_msg, msg);
	}
	pt = t;
	ps = s;
#ifdef __WXGTK__
	first_pass = TRUE;
#endif
	page_max = frame->CalcPageCount(pt, ps, LINES_PER_PAGE,header_msg != NULL);
}

SystemPrintOut::~SystemPrintOut()
{
	if(header_msg != NULL)
		delete header_msg;
}


#define DEFAULT_MARGIN_X		20
#define DEFAULT_MARGIN_Y		DEFAULT_MARGIN_X

bool
SystemPrintOut::HasPage(int page)
{
	if(page <= page_max) 
		return(TRUE);
	return(FALSE);
}

void
SystemPrintOut::GetPageInfo(int *minPage, int *maxPage, int *selPageFrom, int *selPageTo)
{
	*minPage = *selPageFrom = 1;
	*maxPage = *selPageTo = page_max;
}

bool
SystemPrintOut::OnPrintPage(int page)
{
wxDC *dc;
int w,h;
float scale;
bool ret=FALSE;
char buff[120];

	dc = GetDC();
	if(dc) {
		int a_h,a_w;
		
		// figure out how much there is to work with
		dc->GetSize(&w,&h);
		a_h = h - 55;
		a_w = w - 35;
		
		if(pt == PT_MAP) {
			bool drawn=FALSE;
			int i;
			int map_h,map_w,max_x,max_y;
			float scale_x, scale_y;
			int pos_x,pos_y;
			int x_cen[MAPS_PER_PAGE] = { MAP_X_OFFSET1, MAP_X_OFFSET2,
										   MAP_X_OFFSET1, MAP_X_OFFSET2,
										   MAP_X_OFFSET1, MAP_X_OFFSET2 };
			int y_cen[MAPS_PER_PAGE] = { MAP_Y_OFFSET1, MAP_Y_OFFSET1,
										   MAP_Y_OFFSET2, MAP_Y_OFFSET2,
										   MAP_Y_OFFSET3, MAP_Y_OFFSET3 };

			// figure out how much we need
			frame->GetMapSize(&map_h, &map_w);
			max_x = (2 * map_w) + (MAP_BORDER_SIZE * 3);
			max_y = (3 * map_h) + (MAP_BORDER_SIZE * 4);
			max_y += (5 * MAP_TEXT_OFFSET); // pretty abatrary (for text)....
			// set the map positions(change this if MAP_PER_PAGE changes)
			x_cen[0] = x_cen[2] = x_cen[4] = MAP_BORDER_SIZE;
			x_cen[1] = x_cen[3] = x_cen[5] = (2 * MAP_BORDER_SIZE) + map_w;
			y_cen[0] = y_cen[1] = MAP_BORDER_SIZE + (4 * MAP_TEXT_OFFSET);
			y_cen[2] = y_cen[3] = (2 * MAP_BORDER_SIZE) + map_h +
			  (4 * MAP_TEXT_OFFSET);
			y_cen[4] = y_cen[5] = (3 * MAP_BORDER_SIZE) + (2 * map_h) +
			  (4 * MAP_TEXT_OFFSET);

			// figure the scale
			scale_x = (float)a_w / (float)max_x;
			scale_y = (float)a_h / (float)max_y;
			scale = wxMin(scale_x, scale_y);
			dc->SetUserScale(scale, scale);
			// set it in the middle
			pos_x = (int)(((w - (max_x * scale))/ 2.0) / scale);
			pos_y = (int)(((h - (max_y * scale))/ 2.0) / scale);
#if 0
fprintf(stderr, "\npos:%d,%d\nscale:%f,%f  max:%d,%d\n",
		pos_x, pos_y, scale_x, scale_y, max_x, max_y);
fprintf(stderr, "map_w: %d  map_h: %d\n", map_w, map_h);
fprintf(stderr, "    w: %d      h: %d\n", w, h);
fprintf(stderr, "  a_w: %d    a_h: %d\n", a_w, a_h);
for(i = 0;i < MAPS_PER_PAGE;i++)
    fprintf(stderr, "%i: %d,%d\n", i, x_cen[i], y_cen[i]);
#endif
			for(i = 0;i < MAPS_PER_PAGE;i++) {
				int cent_ndx, first_ndx;

				if(frame->CalcMapIndecies(ps, ((page - 1) * MAPS_PER_PAGE) + i,
										  &cent_ndx, &first_ndx)) {
					wxBrush *brush;
					
					drawn = TRUE;
					frame->DrawMapAt(*dc, cent_ndx, first_ndx,
									 x_cen[i] + pos_x, y_cen[i] + pos_y, TRUE);

					brush = new wxBrush("BLACK", wxTRANSPARENT);
					dc->SetBrush(*brush);
					dc->DrawRectangle(x_cen[i] + pos_x - MAP_BORDER_OFFSET,
									  y_cen[i] + pos_y - MAP_BORDER_OFFSET,
									  map_w + (2 * MAP_BORDER_OFFSET),
									  map_h + (2 * MAP_BORDER_OFFSET));
					dc->SetBrush(*wxTRANSPARENT_BRUSH);
					delete brush;
				}
			}
			ret = FALSE; // for now....
		} else if(page > 0) {
			SystemDisplay *sd;
			int line_count,i,cur_count;

			sd = new SystemDisplay(dc, 60, 20);
			sprintf(buff, "Page %d of %d", page, page_max);

			// fairly arbitrary, i would like about 100 lines......
			scale = (float)a_h / (float)(LINES_PER_PAGE * TEXT_Y_INCR);
			line_count = LINES_PER_PAGE - 2;
			if(header_msg != NULL)
				line_count -= 2;
			dc->SetUserScale(scale, scale);

			cur_count = 0;
			for(i = 1;i < page;i++)
			    cur_count += frame->CalcWorldsOnPage(pt == PT_LIST,
													 ps, header_msg != NULL,
													 line_count, i);

			if(pt == PT_LIST)
			    sd->DrawLines(header_msg, buff, cur_count, line_count);
			else
			    sd->DrawDetails(ps, header_msg, buff, cur_count, line_count);

			delete sd;
			ret = TRUE;
		}
	}

	return(ret);
}

// ===============================================================
BEGIN_EVENT_TABLE(PrintDialog, wxDialog)
	EVT_BUTTON(ID_SETUP_BTN, PrintDialog::OnSetup)
	EVT_RADIOBOX(ID_WHICH_CB, PrintDialog::OnWhich)
END_EVENT_TABLE()

// ---------------------------------------------------------------
static wxString printer_choices[7] = {
  "Listing",
  "Detail",
  "Map",
  "All Orbits",
  "Current Orbit",
  "Current and Suborbits",
  "Ignore Satillites" };

// ---------------------------------------------------------------
PrintDialog::PrintDialog(wxWindow *p) :
	StdDialog(p, "Print Options")
{
wxButton *btn;

#if wxUSE_LIBPNG
	wxImage::AddHandler( new wxPNGHandler );
#endif

#if wxUSE_LIBJPEG
	wxImage::AddHandler( new wxJPEGHandler );
#endif
	wxImage::AddHandler( new wxGIFHandler );

	g_dlg = NULL;

    which = new wxRadioBox(this, ID_WHICH_CB, "",
			wxPoint(-1, -1),
			wxSize(-1, -1),
			3, printer_choices, 1, wxRA_SPECIFY_COLS);

	opts = new wxRadioBox(this, -1, "", 
			wxPoint(-1, -1),
			wxSize(-1, -1),
			4, &printer_choices[3], 1, wxRA_SPECIFY_COLS);

	file = new wxCheckBox(this, -1, "Print to File");

	btn = new wxButton(this, ID_SETUP_BTN, "Setup");
	
	wxBoxSizer* sizer_1 = new wxBoxSizer(wxVERTICAL);
    wxBoxSizer* sizer_2 = new wxBoxSizer(wxHORIZONTAL);
    wxBoxSizer* sizer_3 = new wxBoxSizer(wxVERTICAL);
    sizer_2->Add(which, 0, wxALL, DLG_OFFSET);
    sizer_2->Add(opts, 0, wxALL, DLG_OFFSET);
    sizer_3->Add(file, 0, wxALL, DLG_OFFSET);
    sizer_3->Add(btn, 0, wxALL, DLG_OFFSET);
    sizer_2->Add(sizer_3, 0, 0, 0);
    sizer_1->Add(sizer_2, 0, 0, 0);
	DoLayout(wxALL, DLG_OFFSET, sizer_1);

	setup_dialog = NULL;
	p_data = new wxPrintData;
	ps_data = new wxPageSetupDialogData;
#if defined(__WXGTK__) || defined(__WXMOTIF__)
    (*p_data) = * wxThePrintSetupData;
#endif
}

void
PrintDialog::OnWhich(wxCommandEvent& event)
{
	switch(which->GetSelection()) {
		case 0:
			opts->Enable(0, FALSE);
			opts->Enable(1, FALSE);
			opts->Enable(2, FALSE);
			opts->Enable(3, FALSE);
//			file->Enable(TRUE);
			break;
		case 1:
			opts->Enable(0, TRUE);
			opts->Enable(1, TRUE);
			opts->Enable(2, TRUE);
			opts->Enable(3, TRUE);
//			file->Enable(TRUE);
			break;
		case 2:
			opts->Enable(0, TRUE);
			opts->Enable(1, TRUE);
			opts->Enable(2, TRUE);
			opts->Enable(3, TRUE);
//			file->Enable(FALSE);
			break;
	}
}

void
PrintDialog::OnSetup(wxCommandEvent& event)
{
wxPrintDialogData printDialogData(* p_data);

	if(setup_dialog == NULL)
		setup_dialog = new wxPrintDialog(this, & printDialogData);
	setup_dialog->GetPrintDialogData().SetSetupDialog(TRUE);
	setup_dialog->ShowModal();
}

void
PrintDialog::DoPrint(System *sys)
{
PRINT_TYPE pt;
PRINT_SOURCE ps;
char *ptr,*ptr1=NULL,buff[120],n_buff[256];
FILE *fpx=NULL;
SystemDisplay *sd=NULL;
SystemPrintOut *print_out;
wxString str;

    Reset();

	if(wxID_CANCEL == ShowModal())
	    return;

	buff[0] = 0;
	if((ptr = sys->GetName()) != NULL)
	    strcat(buff, ptr);
	if((ptr = sys->GetLoc()) != NULL) {
		if(buff[0] != 0)
		    strcat(buff, "; ");
		strcat(buff, ptr);
	}
	if((ptr = sys->GetSSName()) != NULL) {
		if(buff[0] != 0)
		    strcat(buff, "; ");
		strcat(buff, ptr);
	}
	if((ptr = sys->GetSectName()) != NULL) {
		if(buff[0] != 0)
		    strcat(buff, "; ");
		strcat(buff, ptr);
	}
	if(ptr[0] != 0)
	    ptr1 = buff;
		
	pt = (PRINT_TYPE) (which->GetSelection() + 1);
	ps = (PRINT_SOURCE) (opts->GetSelection());
	// XXX needs some work (should bring in the Dirs junk):
	if(sys->GetLoc() != NULL)
		sprintf(n_buff, "%s", sys->GetLoc());
	else
		n_buff[0] = 0;

	if(file->GetValue()) {
		if(pt == PT_MAP) {
			bool use_scale=TRUE;
			char *file,*desc;
			int width,height,o_width,o_height;
			int g_type;
			int img_count=0;
			int cent_ndx, first_ndx;
			float scale=1;
			wxBitmap *bm;
			wxMemoryDC dc;
			wxImage *img;
		
			g_type = wxBITMAP_TYPE_JPEG;

			frame->GetMapSize(&height, &width);
			o_width = width;
			o_height = height;

			// Find out what the user wants...
			if(NULL == g_dlg)
				g_dlg = new GraphicDialog(this);

			if(!g_dlg->GetGraphicInfo(&g_type, &use_scale, &width, &height, 
						&scale, &desc))
				return;

			if(use_scale) {
				width = (int) (scale * (float)width);
				height = (int) (scale * (float)height);
			} else {
				float s_w,s_h;

				s_w = (float) width / (float) o_width;
				s_h = (float) height / (float) o_height;
				if(s_w > s_h)		// XXX shouldn't matter
					scale = s_w;
				else
					scale = s_h;
			}

			// ...and where to put it
			while(frame->CalcMapIndecies(ps, img_count,
							  &cent_ndx, &first_ndx)) {
				if(n_buff[0] != 0) {
					switch(g_type) {
						case wxBITMAP_TYPE_XPM:
							strcat(n_buff, ".xpm");
							break;
						case wxBITMAP_TYPE_TIF:
							strcat(n_buff, ".tif");
							break;
						case wxBITMAP_TYPE_GIF:
							strcat(n_buff, ".gif");
							break;
						case wxBITMAP_TYPE_PNG:
							strcat(n_buff, ".png");
							break;
						case wxBITMAP_TYPE_JPEG:
							strcat(n_buff, ".jpg");
							break;
					}
				}
				str = wxFileSelector("Print To", NULL, n_buff, NULL, 
					desc,
					wxHIDE_READONLY | wxSAVE | wxOVERWRITE_PROMPT, 
					this);
				ptr = (char *)str.GetData();
				if((NULL == ptr) || (0 == ptr[0]))
					break;

				wxBeginBusyCursor();
				file = new char[strlen(ptr) + 1];
				strcpy(file, ptr);

				bm = new wxBitmap(width, height);
				dc.SelectObject(*bm);
				dc.SetUserScale(scale, scale);

				dc.SelectObject(*bm);
				dc.SetBrush(*wxWHITE_BRUSH);
				dc.SetPen(*wxWHITE_PEN);
				dc.DrawRectangle( 0, 0, o_width, o_height);

				frame->DrawMapAt(dc, cent_ndx, first_ndx, 0, 0, TRUE);

				dc.SelectObject(wxNullBitmap);

				img = new wxImage(*bm);
				img->SaveFile(file, g_type);

				delete img;
				delete bm;
				delete file;
				wxEndBusyCursor();
				img_count++;
			}
		} else {
			if(n_buff[0] != 0)
				strcat(n_buff, ".txt");
			str = wxFileSelector("Print To", NULL, n_buff, NULL, "*.txt",
							 wxHIDE_READONLY | wxSAVE | wxOVERWRITE_PROMPT,
							 this);
			ptr = (char *)str.GetData();
			if((ptr == NULL) || (0 == ptr[0]) || 
					((fpx = fopen(ptr, "a+")) == NULL))
		  		return;

			sd = new SystemDisplay(fpx);

			if(pt == PT_DETAIL)
				sd->DrawDetails(ps, ptr1);
			else
		    	sd->DrawLines(ptr1);

			delete sd;
		}
	} else {
		wxPrinter *printer;
		print_out = new SystemPrintOut(pt, ps, ptr1);

		printer = new wxPrinter();
		printer->Print(this, print_out, TRUE);
		
		delete printer;
		delete print_out;
	}
}

void
PrintDialog::Reset(void)
{
    which->SetSelection(0);
	opts->SetSelection(0);
	file->SetValue(FALSE);
}

