/*
 * File:      sysgen.cpp
 * Purpose:	  generate Traveller systems (1-3)
 * Author:	
 * Created:	
 * Updated:	
 * Copyright: LGPL.
 *            Traveller is a registered trademark of Far Future Enterprises.	
 */

/* rcsid[] = "$RCSfile: sysgen.cpp,v $ $Revision: 1.19 $ $Author: man $ $Date: 2003/04/27 19:32:31 $" */

// For compilers that support precompilation, includes "wx.h".
#include "wx/wxprec.h"

#ifndef WX_PRECOMP
#include "wx/wx.h"
#endif

#include <stdlib.h>
#include <math.h>
#include <string.h>
#include "sysgen.h"
#include "t_res.h"
#include "t_help.h"
//#include "t_dice.h"

// button defines
enum {
	TB_F_REV = 501,
	TB_REV,
	TB_FWD,
	TB_F_FWD,
	TB_CONFIG,
	TB_STAR1,
	TB_STAR2,
	TB_STAR3,
	TB_STAR4,
	TB_STAR5,
	TB_GG,
	TB_BELT,
	TB_MW,
	TB_SAVE,
	TB_HELP,
	TB_EXIT,
	SERVER_ID,
	SOCKET_ID
};

#ifdef __WXGTK__
#include "bitmaps/belt.xpm"
#include "bitmaps/config.xpm"
#include "bitmaps/exit.xpm"
#include "bitmaps/f_fwd.xpm"
#include "bitmaps/f_rev.xpm"
#include "bitmaps/fwd.xpm"
#include "bitmaps/gg.xpm"
#include "bitmaps/help.xpm"
#include "bitmaps/rev.xpm"
#include "bitmaps/save.xpm"
#include "bitmaps/star1.xpm"
#include "bitmaps/star2.xpm"
#include "bitmaps/star3.xpm"
#include "bitmaps/star4.xpm"
#include "bitmaps/star5.xpm"
#include "bitmaps/sysgen.xpm"
#include "bitmaps/world.xpm"
#endif

IMPLEMENT_APP(SysgenApp)

// handy references
SysgenApp *app=NULL;
BuildFrame *frame = NULL;
wxFont *NormalFont = NULL;

static bool l_init = FALSE;

// ===============================================================
// `Main program' equivalent, creating windows and returning main app frame
bool
SysgenApp::OnInit(void)
{
struct sys_conf cfg;

	app = this;

    // initialize cfg
    cfg.name = NULL;
	cfg.sect = NULL;
	cfg.ssect = NULL;
	cfg.loc = NULL;
	cfg.x = cfg.y = cfg.z = 0;
	cfg.use_xyz = FALSE;
	cfg.belts = -1;
	cfg.gg = -1;
	cfg.port = 0;
	memcpy(cfg.uwp, "\0\0\0\0\0\0\0\0", sizeof(cfg.uwp));
	memcpy(cfg.stars[0], "R\0\0\0\0\0\0", sizeof(cfg.stars[0]));
	memcpy(cfg.stars[1], "R\0\0\0\0\0\0", sizeof(cfg.stars[1]));
	memcpy(cfg.stars[2], "R\0\0\0\0\0\0", sizeof(cfg.stars[2]));
	memcpy(cfg.bases, "\0", sizeof(cfg.bases));
	memcpy(cfg.alleg, "\0\0", sizeof(cfg.alleg));
	memcpy(cfg.zone, "\0", sizeof(cfg.zone));
	cfg.lang = NULL;
	cfg.dirs = new MTUFile();
	cfg.out_file = NULL;
	cfg.quiet_mode = FALSE;
	cfg.no_sleep = FALSE;
	cfg.run_editor = FALSE;
	cfg.interactive = FALSE;
	cfg.no_main = TRUE;
	cfg.full_rand = FALSE;
	cfg.tech_method = TG_STANDARD;
	cfg.tech_min = -1;
	cfg.tech_max = -1;

	codes = new TCodes(cfg.dirs->GetMTUDataDir());

	ParseCommandLine(&cfg);

	if((cfg.use_xyz) && (NULL == cfg.loc)) {
		int x,y,z;

		x = abs(cfg.x % 10);
		y = abs(cfg.y % 10);
		z = abs(cfg.z % 10);
		cfg.loc = new char[30];
		sprintf(cfg.loc, "x:%d.%d y:%d.%d z:%d.%d",
			cfg.x / 10, x,
			cfg.y / 10, y,
			cfg.z / 10, z);
	}
		
	// create a font
    NormalFont = new wxFont(11, wxMODERN, wxNORMAL, wxNORMAL); 

    frame = new BuildFrame(&cfg);
	// Give it an icon
#ifdef __WINDOWS__
	frame->SetIcon(wxIcon("sysgen_icn"));
#endif
#ifdef __WXGTK__
	frame->SetIcon(wxIcon(sysgen_xpm));
#endif

	if(cfg.interactive)
		frame->Show(FALSE);
	else
		frame->Show(TRUE);
	
	l_init = TRUE;
	return(TRUE);
}

void
SysgenApp::CleanUp(void)
{
	if(codes != NULL) {
		delete codes;
		codes = NULL;
	}
}

bool
SysgenApp::ParseString(char **dest, char *src)
{
    if(*dest != NULL)
	    return(FALSE);
	if(src == NULL)
	    return(TRUE);
	*dest = new char[strlen(src) + 1];
	strcpy(*dest, src);
	return(TRUE);
}

bool
SysgenApp::CopyString(char *dest, char *src, unsigned int sz)
{
    if(strlen(src) > sz)
	    return(FALSE);
	strcpy(dest, src);
	return(TRUE);
}

void
SysgenApp::ParseCommandLine(struct sys_conf *cfg)
{
bool tl_error=FALSE;
int i=1,star_count=0;
char buff[120], last_arg=0;
char *sect_file=NULL;
DetailSector *sect=NULL;

    buff[0] = 0;
    while(i < argc) {
		if('-' == argv[i][0]) {
			if(strchr("abeEFgHilLnNopqrsSTtuXYzZ", argv[i][1]) == NULL) {
				fprintf(stderr, "BadOption: %c\n", argv[i][1]);
				fprintf(stderr, "Options:\n"
						" -n <system name>\n"
						" -S <Sector Name>\n"
						" -s <Subsector name>\n"
						" -l <location>\n"
						" -g <# of gas giants; R for random>\n"
						" -b <# of asteroid belts; R for random>\n"
						" -u <world UWP>\n"
						" -e <base code>\n"
						" -a <alleg. code>\n"
						" -z <TAS zone>\n"
						" -r <star (1-3)>\n"
						" -F <sector file>\n"
						" -L <language file>\n"
						" -o <output file>\n"
						" -X <x location>\n"
						" -Y <y location>\n"
						" -Z <z location>\n"
						" -E\n"
						" -p <port>\n"
						" -H <tl enum>\n"
						" -t <lower tl>\n"
						" -T <upper tl>\n"
						// "-N undocumented: no sleep\n"
						" -i\n"
						" -q\n");
				exit(-1);
			}

			if(last_arg != 0) {
				ProcessArgs(cfg, last_arg, buff, &star_count, &sect_file);
			}

			last_arg = argv[i][1];
			buff[0] = 0;
			if(argv[i][2] != 0)
			    strcat(buff, &argv[i][2]);
		} else
		    strcat(buff, argv[i]);
		i++;
	}

	if(last_arg != 0)
	    ProcessArgs(cfg, last_arg, buff, &star_count, &sect_file);
	
	// catch some obvious errors
	if((cfg->quiet_mode) && (NULL == cfg->out_file )) {
		fprintf(stderr, "-o <output file> must be specified with -q\n");
		exit(-6);
	}
	switch(cfg->tech_method) {
		case TG_STANDARD:
		case TG_DM:
		case TG_NO_DM:
			break;
		case TG_MIN_MAX:
		case TG_FLAT_RANGE:
		case TG_RANGE:
			if((cfg->tech_min < 0) || (cfg->tech_min > cfg->tech_max))
				tl_error = TRUE;
			break;
		case TG_VALUE:
			if(cfg->tech_min < 0)
				tl_error = TRUE;
			break;
		default:
			tl_error = TRUE;
	}
	if(tl_error) {
		fprintf(stderr, "Tech Method (%d) won't work with (%d, %d)\n",
				cfg->tech_method, cfg->tech_min, cfg->tech_max);
		exit(-7);
	}
//fprintf(stderr, "-->sf:%s l:%s\n", sect_file, cfg->loc);
	if(sect_file != NULL) {
		if(cfg->loc == NULL) {
			fprintf(stderr, "No location.\n");
			exit(-4);
		}
		
		sect = new DetailSector(codes);

		if(sect->LoadFile(sect_file) > 0) {
			int i,x,y;
			HexData *hex;

			i = atol(cfg->loc);
			x = i / 100;
			y = i % 100;
			hex = sect->GetHex(x - 1,y - 1);
			if(hex == NULL) {
				fprintf(stderr, "Problem with location %s.\n", cfg->loc);
				exit(-5);
			}
			ParseString(&cfg->name, hex->world->GetName(buff));
			ParseString(&cfg->sect, sect->GetName());
			i = (((x - 1) / 8)) + (((y - 1) / 10) * 4);
			ParseString(&cfg->ssect, sect->GetSSName(i));
			cfg->belts = hex->world->GetNumBelts();
			cfg->gg = hex->world->GetNumGG();
			cfg->uwp[0] = hex->world->get_port();
			cfg->uwp[1] = hex->world->get_size();
			cfg->uwp[2] = hex->world->get_atmos();
			cfg->uwp[3] = hex->world->get_hydro();
			cfg->uwp[4] = hex->world->get_pop();
			cfg->uwp[5] = hex->world->get_govt();
			cfg->uwp[6] = hex->world->get_law();
			cfg->uwp[7] = hex->world->get_tech();
			for(i = 0;i < 3;i++) {
				buff[0] = 0;
				hex->world->GetStarStr(i, buff);
				if(buff[0] == 0)
				    buff[0] = 'N';
				CopyString(cfg->stars[i], buff, 7);
				strstrip(cfg->stars[i], TRUE);
				strstrip(cfg->stars[i], FALSE);
			}
			cfg->bases[0] = hex->world->GetBase();
			cfg->zone[0] = hex->world->GetZone();
			CopyString(cfg->alleg, hex->world->GetAlleg(), 2);
			cfg->no_main = FALSE;
//fprintf(stderr, "-->n:%s s:%s ss:%s b:%d g:%d u:%s s1:%s s2:%s s3:%s b:%c z:%s a:%c%c\n", cfg->name, cfg->sect, cfg->ssect, cfg->belts, cfg->gg, cfg->uwp, cfg->stars[0], cfg->stars[1], cfg->stars[2], cfg->bases, cfg->zone, cfg->alleg[0], cfg->alleg[1]);
		} else {
			fprintf(stderr, "Problem with sector file %s\n", sect_file);
			exit(-3);
		}

		delete sect;
		delete sect_file;
	}
}

void
SysgenApp::ProcessArgs(struct sys_conf *cfg, char last_arg, char *buff,
					   int *star_count, char **sect_file)
{
bool no_err=TRUE;			

//fprintf(stderr, "%c %s\n", last_arg, buff);
    switch(last_arg) {
		case 'n':
		    no_err = ParseString(&cfg->name, buff);
			break;
		case 'S':
		    no_err = ParseString(&cfg->sect, buff);
			break;
		case 's':
		    no_err = ParseString(&cfg->ssect, buff);
			break;
		case 'l':
		    no_err = ParseString(&cfg->loc, buff);
			break;
		case 'g':
		    if(('R' == buff[0]) || ('r' == buff[0]))
			    cfg->gg = -1;
			else if((buff[0] >= '0') &&
					(buff[0] <= (MAX_GG + '0')))
			    cfg->gg = buff[0] - '0';
			else
			    no_err = FALSE;
			break;
		case 'b':
		    if(('R' == buff[0]) || ('r' == buff[0]))
			    cfg->belts = -1;
			else if((buff[0] >= '0') &&
					(buff[0] <= (MAX_BELTS + '0')))
			    cfg->belts = buff[0] - '0';
			else
			    no_err = FALSE;
			break;
		case 'u':
		    no_err = CopyString(cfg->uwp, buff, 8);
			break;
		case 'e':
		    cfg->bases[0] = buff[0];
			break;
		case 'a':
		    no_err = CopyString(cfg->alleg, buff, 2);
			break;
		case 'z':
		    cfg->zone[0] = buff[0];
			break;
		case 'r':
		    {
			no_err = CopyString(cfg->stars[*star_count], buff, 7);
			*star_count = *star_count + 1;
			if(*star_count > 3) no_err = FALSE;
			break;
			}
		case 'F':
		    no_err = ParseString(sect_file, buff);
			break;
		case 'L':
		    cfg->lang = new WordGen(buff);
			no_err = cfg->lang->Valid();
			break;
		case 'o':
		    no_err = ParseString(&cfg->out_file, buff);
			break;
		case 'q':
		    cfg->quiet_mode = TRUE;
		    break;
		case 'N':
		    cfg->no_sleep = TRUE;
		    break;
		case 'E':
		    cfg->run_editor = TRUE;
		    break;
		case 'X':
			cfg->x = atoi(buff);
			cfg->use_xyz = TRUE;
			break;
		case 'Y':
			cfg->y = atoi(buff);
			cfg->use_xyz = TRUE;
			break;
		case 'Z':
			cfg->z = atoi(buff);
			cfg->use_xyz = TRUE;
		    break;
		case 'p':
			cfg->port = buff[0];
			break;
		case 'i':
			cfg->interactive = TRUE;
			break;
		case 'H':
			cfg->tech_method = (TECH_GEN) atoi(buff);
			break;
		case 't':
			cfg->tech_min = atoi(buff);
			break;
		case 'T':
			cfg->tech_max = atoi(buff);
			break;
		default:
			no_err = FALSE;
			break;
	}
			
	if(!no_err) {
		fprintf(stderr, "Problem with parameter: %c\n", last_arg);
		exit(-2);
	}
}

// ===============================================================
InitTimer::InitTimer() : wxTimer() 
{
}

InitTimer::~InitTimer() 
{
}

void
InitTimer::Notify(void)
{
	if(l_init != FALSE) {
		Stop();
		frame->Proceed();
//		frame->Close(TRUE);
	}
	wxYield();
}

// ===============================================================
void
BuildFrame::CleanUp(void)
{
	delete gen;
	gen = NULL;

	if(cfg.name != NULL)
	    delete cfg.name;
	if(cfg.sect != NULL)
	    delete cfg.sect;
	if(cfg.ssect != NULL)
	    delete cfg.ssect;
	if(cfg.loc != NULL)
	    delete cfg.loc;
	if(cfg.lang != NULL)
	    delete cfg.lang;
	//	if(cfg.lang_file != NULL)
	//	    delete cfg.lang_file;
	if(cfg.out_file != NULL)
	    delete cfg.out_file;
	//	if(cfg.sect_file != NULL)
	//	    delete cfg.sect_file;
	cfg.name = cfg.sect = cfg.ssect = cfg.loc = cfg.out_file = NULL;
	cfg.lang = NULL;

	if(cfg.dirs != NULL)
		delete cfg.dirs;
	cfg.dirs = NULL;
	app->CleanUp();
}

// ===============================================================
#define QUIET_FRAME_WIDTH        300
#define QUIET_FRAME_HEIGHT       75

static char *state_msgs[] = {
  "Determining star presences and system features.",
  "Editing stars.",
  "Placing known components.",
  "Selecting main world.",
  "System finished." };

void
BuildFrame::Proceed(void)
{
	Process();
	wxYield();
	if(!cfg.no_sleep)
		wxSleep(1);
	gen->Save(cfg.out_file, &cfg);
	if(cfg.run_editor) {
		char buff[1024];

		sprintf(buff, "%s %s", cfg.dirs->GetSystem(), cfg.out_file);
		wxExecute(buff, FALSE);
	}
	wxYield();
	Close(TRUE);
}

void 
BuildFrame::Process(void)
{
int i=0;

	while(TRUE) {
		wxYield();
		msg->SetLabel(state_msgs[i]);
		g->SetValue((i + 1) * 20);
		i++;
		if(!gen->NextState(&cfg))
			break;
		if(!cfg.no_sleep)
			wxSleep(1);
	}
}

// ===============================================================
// ---------------------------------------------------------------
#define BUILD_FRAME_WIDTH         500
#define BUILD_FRAME_HEIGHT        400
#define BUILD_PANEL_HEIGHT        80

// ---------------------------------------------------------------
BEGIN_EVENT_TABLE(BuildFrame, wxFrame)
	EVT_SIZE(BuildFrame::OnSize)
	EVT_BUTTON(TB_F_REV, BuildFrame::Cancel)
	EVT_BUTTON(TB_REV, BuildFrame::Prev)
	EVT_BUTTON(TB_FWD, BuildFrame::Next)
	EVT_BUTTON(TB_F_FWD, BuildFrame::End)
	EVT_BUTTON(TB_CONFIG, BuildFrame::Configure)
	EVT_BUTTON(TB_STAR1, BuildFrame::Star1)
	EVT_BUTTON(TB_STAR2, BuildFrame::Star2)
	EVT_BUTTON(TB_STAR3, BuildFrame::Star3)
	EVT_BUTTON(TB_STAR4, BuildFrame::Star4)
	EVT_BUTTON(TB_STAR5, BuildFrame::Star5)
	EVT_BUTTON(TB_GG, BuildFrame::Gg)
	EVT_BUTTON(TB_HELP, BuildFrame::OnAbout)
	EVT_BUTTON(TB_BELT, BuildFrame::Belt)
	EVT_BUTTON(TB_MW, BuildFrame::Mw)
	EVT_BUTTON(TB_SAVE, BuildFrame::Save)
	EVT_BUTTON(TB_EXIT, BuildFrame::Exit)
END_EVENT_TABLE()

#define MAX_BITMAPS		16
BuildFrame::BuildFrame(struct sys_conf *c) :
	wxFrame(NULL, -1, "Generate System", wxPoint(0, 0), 
			wxSize(BUILD_FRAME_WIDTH, BUILD_FRAME_HEIGHT))
{
int i;

	done_flag = FALSE;
    // copy the cfg struct 
    cfg.name = c->name;
	cfg.sect = c->sect;
	cfg.ssect = c->ssect;
	cfg.loc = c->loc;
	cfg.belts = c->belts;
	cfg.gg = c->gg;
	cfg.port = c->port;
	memcpy(cfg.uwp, "\0\0\0\0\0\0\0\0", sizeof(cfg.uwp));
	strcpy(cfg.uwp, c->uwp);
	for(i = 0;i < 3;i++) {
		memcpy(cfg.stars[i], "\0\0\0\0\0\0\0", sizeof(cfg.stars[i]));
		strcpy(cfg.stars[i], c->stars[i]);
	}
	cfg.bases[0] = c->bases[0];
	cfg.bases[1] = 0;
	cfg.alleg[0] = c->alleg[0];
	cfg.alleg[1] = c->alleg[1];
	cfg.alleg[2] = c->alleg[2];
	cfg.zone[0] = c->zone[0];
	cfg.zone[1] = 0;
	cfg.lang = NULL;
	//	cfg.lang_file = c->lang_file;
	cfg.out_file = c->out_file;
	//	cfg.sect_file = c->sect_file;
	cfg.quiet_mode = c->quiet_mode;
	cfg.no_sleep = c->no_sleep;
	cfg.run_editor = c->run_editor;
	cfg.interactive = c->interactive;
	cfg.no_main = c->no_main;
	cfg.full_rand = c->full_rand;
	cfg.tech_method = c->tech_method;
	cfg.tech_min = c->tech_min;
	cfg.tech_max = c->tech_max;

//	cfg.sup_warnings = FALSE;
//	cfg.sup_star_adjust = FALSE;
	cfg.dirs = c->dirs;

    // create the generator
    gen = new SystemGenerate(cfg.dirs->GetMTUDataDir(),
					cfg.tech_method, cfg.tech_min, cfg.tech_max);

	if((cfg.quiet_mode) || (cfg.interactive))
		InitQuiet();
	else
		InitBuild();
}

void
BuildFrame::InitBuild(void)
{
int i;
wxBitmap *bitmaps[MAX_BITMAPS];
int status_widths[2] = { -1, 128 };

	// null out the dialogs (create on the fly)
	config = NULL;
	edit_star = NULL;
	move = NULL;
	select = NULL;
	mw_sel = NULL;

	// now do the window stuff
	CreateStatusBar(2);
	SetStatusWidths(2, status_widths);

	canvas = new SysGenCanvas(this);
    canvas->SetScrollbars(4, 4, 200, 500, 0, 0);
	canvas->SetBackgroundColour(*wxWHITE);

	// Set up toolbar
#ifdef __WXMSW__
	bitmaps[0] = new wxBitmap("icon1");
	bitmaps[1] = new wxBitmap("icon2");
	bitmaps[2] = new wxBitmap("icon3");
	bitmaps[3] = new wxBitmap("icon4");
	bitmaps[4] = new wxBitmap("icon5");
	bitmaps[5] = new wxBitmap("icon6");
	bitmaps[6] = new wxBitmap("icon7");
	bitmaps[7] = new wxBitmap("icon8");
	bitmaps[8] = new wxBitmap("icon9");
	bitmaps[9] = new wxBitmap("icon10");
	bitmaps[10] = new wxBitmap("icon11");
	bitmaps[11] = new wxBitmap("icon12");
	bitmaps[12] = new wxBitmap("icon13");
	bitmaps[13] = new wxBitmap("icon14");
	bitmaps[14] = new wxBitmap("icon15");
	bitmaps[15] = new wxBitmap("icon16");
#else
	bitmaps[0] = new wxBitmap(f_rev_xpm);
	bitmaps[1] = new wxBitmap(rev_xpm);
	bitmaps[2] = new wxBitmap(fwd_xpm);
	bitmaps[3] = new wxBitmap(f_fwd_xpm);
	bitmaps[4] = new wxBitmap(config_xpm);
	bitmaps[5] = new wxBitmap(star1_xpm);
	bitmaps[6] = new wxBitmap(star2_xpm);
	bitmaps[7] = new wxBitmap(star3_xpm);
	bitmaps[8] = new wxBitmap(star4_xpm);
	bitmaps[9] = new wxBitmap(star5_xpm);
	bitmaps[10] = new wxBitmap(gg_xpm);
	bitmaps[11] = new wxBitmap(belt_xpm);
	bitmaps[12] = new wxBitmap(world_xpm);
	bitmaps[13] = new wxBitmap(save_xpm);
	bitmaps[14] = new wxBitmap(exit_xpm);
	bitmaps[15] = new wxBitmap(help_xpm);
#endif

	// 'main' toolbar/panel
	tb_init = new wxPanel(this);

	new wxBitmapButton(tb_init, TB_F_REV, *bitmaps[0], 
			wxPoint((DLG_OFFSET * 1) + (SML_BTN_SIZE * 0), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));
	new wxBitmapButton(tb_init, TB_REV, *bitmaps[1], 
			wxPoint((DLG_OFFSET * 1) + (SML_BTN_SIZE * 1), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));
	new wxBitmapButton(tb_init, TB_FWD, *bitmaps[2], 
			wxPoint((DLG_OFFSET * 1) + (SML_BTN_SIZE * 2), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));
	new wxBitmapButton(tb_init, TB_F_FWD, *bitmaps[3], 
			wxPoint((DLG_OFFSET * 1) + (SML_BTN_SIZE * 3), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));

	new wxBitmapButton(tb_init, TB_CONFIG, *bitmaps[4], 
			wxPoint((DLG_OFFSET * 2) + (SML_BTN_SIZE * 4), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));
	new wxBitmapButton(tb_init, TB_HELP, *bitmaps[15], 
			wxPoint((DLG_OFFSET * 3) + (SML_BTN_SIZE * 5), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));
	new wxBitmapButton(tb_init, TB_EXIT, *bitmaps[14], 
			wxPoint((DLG_OFFSET * 4) + (SML_BTN_SIZE * 6), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));

	// stars toolbar/panel
	tb_stars = new wxPanel(this);

	new wxBitmapButton(tb_stars, TB_F_REV, *bitmaps[0], 
			wxPoint((DLG_OFFSET * 1) + (SML_BTN_SIZE * 0), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));
	new wxBitmapButton(tb_stars, TB_REV, *bitmaps[1], 
			wxPoint((DLG_OFFSET * 1) + (SML_BTN_SIZE * 1), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));
	new wxBitmapButton(tb_stars, TB_FWD, *bitmaps[2], 
			wxPoint((DLG_OFFSET * 1) + (SML_BTN_SIZE * 2), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));
	new wxBitmapButton(tb_stars, TB_F_FWD, *bitmaps[3], 
			wxPoint((DLG_OFFSET * 1) + (SML_BTN_SIZE * 3), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));

	new wxBitmapButton(tb_stars, TB_STAR1, *bitmaps[5], 
			wxPoint((DLG_OFFSET * 2) + (SML_BTN_SIZE * 4), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));
	new wxBitmapButton(tb_stars, TB_STAR2, *bitmaps[6], 
			wxPoint((DLG_OFFSET * 2) + (SML_BTN_SIZE * 5), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));
	new wxBitmapButton(tb_stars, TB_STAR3, *bitmaps[7], 
			wxPoint((DLG_OFFSET * 2) + (SML_BTN_SIZE * 6), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));
	new wxBitmapButton(tb_stars, TB_STAR4, *bitmaps[8], 
			wxPoint((DLG_OFFSET * 2) + (SML_BTN_SIZE * 7), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));
	new wxBitmapButton(tb_stars, TB_STAR5, *bitmaps[9], 
			wxPoint((DLG_OFFSET * 2) + (SML_BTN_SIZE * 8), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));

	new wxBitmapButton(tb_stars, TB_HELP, *bitmaps[15], 
			wxPoint((DLG_OFFSET * 3) + (SML_BTN_SIZE * 9), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));
	new wxBitmapButton(tb_stars, TB_EXIT, *bitmaps[14], 
			wxPoint((DLG_OFFSET * 4) + (SML_BTN_SIZE * 10), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));

	// known toolbar/panel
	tb_known = new wxPanel(this);

	new wxBitmapButton(tb_known, TB_F_REV, *bitmaps[0], 
			wxPoint((DLG_OFFSET * 1) + (SML_BTN_SIZE * 0), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));
	new wxBitmapButton(tb_known, TB_REV, *bitmaps[1], 
			wxPoint((DLG_OFFSET * 1) + (SML_BTN_SIZE * 1), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));
	new wxBitmapButton(tb_known, TB_FWD, *bitmaps[2], 
			wxPoint((DLG_OFFSET * 1) + (SML_BTN_SIZE * 2), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));
	new wxBitmapButton(tb_known, TB_F_FWD, *bitmaps[3], 
			wxPoint((DLG_OFFSET * 1) + (SML_BTN_SIZE * 3), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));

	new wxBitmapButton(tb_known, TB_GG, *bitmaps[10], 
			wxPoint((DLG_OFFSET * 2) + (SML_BTN_SIZE * 4), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));
	new wxBitmapButton(tb_known, TB_BELT, *bitmaps[11], 
			wxPoint((DLG_OFFSET * 2) + (SML_BTN_SIZE * 5), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));
	new wxBitmapButton(tb_known, TB_MW, *bitmaps[12], 
			wxPoint((DLG_OFFSET * 2) + (SML_BTN_SIZE * 6), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));

	new wxBitmapButton(tb_known, TB_HELP, *bitmaps[15], 
			wxPoint((DLG_OFFSET * 3) + (SML_BTN_SIZE * 7), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));

	new wxBitmapButton(tb_known, TB_EXIT, *bitmaps[14], 
			wxPoint((DLG_OFFSET * 4) + (SML_BTN_SIZE * 8), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));

	// mw toolbar/panel
	tb_mw = new wxPanel(this);

	new wxBitmapButton(tb_mw, TB_F_REV, *bitmaps[0], 
			wxPoint((DLG_OFFSET * 1) + (SML_BTN_SIZE * 0), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));
	new wxBitmapButton(tb_mw, TB_REV, *bitmaps[1], 
			wxPoint((DLG_OFFSET * 1) + (SML_BTN_SIZE * 1), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));
	new wxBitmapButton(tb_mw, TB_FWD, *bitmaps[2], 
			wxPoint((DLG_OFFSET * 1) + (SML_BTN_SIZE * 2), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));
	new wxBitmapButton(tb_mw, TB_F_FWD, *bitmaps[3], 
			wxPoint((DLG_OFFSET * 1) + (SML_BTN_SIZE * 3), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));

	new wxBitmapButton(tb_mw, TB_MW, *bitmaps[12], 
			wxPoint((DLG_OFFSET * 2) + (SML_BTN_SIZE * 4), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));
	new wxBitmapButton(tb_mw, TB_HELP, *bitmaps[15], 
			wxPoint((DLG_OFFSET * 3) + (SML_BTN_SIZE * 5), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));

	new wxBitmapButton(tb_mw, TB_EXIT, *bitmaps[14], 
			wxPoint((DLG_OFFSET * 4) + (SML_BTN_SIZE * 6), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));

	// finish toolbar/panel
	tb_finish = new wxPanel(this);

	new wxBitmapButton(tb_finish, TB_F_REV, *bitmaps[0], 
			wxPoint((DLG_OFFSET * 1) + (SML_BTN_SIZE * 0), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));
	new wxBitmapButton(tb_finish, TB_REV, *bitmaps[1], 
			wxPoint((DLG_OFFSET * 1) + (SML_BTN_SIZE * 1), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));
	new wxBitmapButton(tb_finish, TB_FWD, *bitmaps[2], 
			wxPoint((DLG_OFFSET * 1) + (SML_BTN_SIZE * 2), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));
	new wxBitmapButton(tb_finish, TB_F_FWD, *bitmaps[3], 
			wxPoint((DLG_OFFSET * 1) + (SML_BTN_SIZE * 3), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));

	new wxBitmapButton(tb_finish, TB_SAVE, *bitmaps[13], 
			wxPoint((DLG_OFFSET * 2) + (SML_BTN_SIZE * 4), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));

	new wxBitmapButton(tb_finish, TB_HELP, *bitmaps[15], 
			wxPoint((DLG_OFFSET * 3) + (SML_BTN_SIZE * 5), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));

	new wxBitmapButton(tb_finish, TB_EXIT, *bitmaps[14], 
			wxPoint((DLG_OFFSET * 4) + (SML_BTN_SIZE * 6), DLG_OFFSET),
			wxSize(SML_BTN_SIZE, SML_BTN_SIZE));

	for(i = 0;i < MAX_BITMAPS;i++)
		delete bitmaps[i];
	ReflectState();
}

void
BuildFrame::InitQuiet(void)
{
wxPanel *panel;

	timer = new InitTimer();

	wxColour col1("red");
	wxColour col2("cyan");

	panel = new wxPanel(this);
	g = new wxGauge(panel, -1, 100, 
		wxPoint(DLG_OFFSET, DLG_OFFSET),
		wxSize(QUIET_FRAME_WIDTH - (3 * DLG_OFFSET), QUIET_FRAME_HEIGHT - 50),
		wxGA_HORIZONTAL);
	g->SetShadowWidth(3);
	g->SetValue(0);

	msg = new wxStaticText(panel, -1, "Initializing....",
		wxPoint(DLG_OFFSET, QUIET_FRAME_HEIGHT - 30),
		wxSize(QUIET_FRAME_WIDTH - (3 * DLG_OFFSET), -1));

	Fit();
	SetSize(QUIET_FRAME_WIDTH, QUIET_FRAME_HEIGHT);

	timer->Start(500, FALSE);
}

BuildFrame::~BuildFrame()
{
	CleanUp();
}

// -------------------------------------------------------------------------
// button functions
void
BuildFrame::Next(wxCommandEvent& event)
{
    if(gen->NextState(&cfg))
	    ReflectState();
}

void
BuildFrame::Prev(wxCommandEvent& event)
{
    if(gen->PrevState(&cfg))
	    ReflectState();
}

void
BuildFrame::Cancel(wxCommandEvent& event)
{
    if(gen->Cancel(&cfg))
	    ReflectState();
}

void
BuildFrame::End(wxCommandEvent& event)
{
    if(gen->End(&cfg))
	    ReflectState();
}

void
BuildFrame::Configure(wxCommandEvent& event)
{
	if(config == NULL)
	    config = new ConfigDialog(this, app->codes->GetBaseTable(),
								  app->codes->GetAllegTable());
	if(config->GetConfig(&cfg)) {
	}
}

void
BuildFrame::Star1(wxCommandEvent& event)
{
	EditStars(0);
}

void
BuildFrame::Star2(wxCommandEvent& event)
{
	EditStars(1);
}

void
BuildFrame::Star3(wxCommandEvent& event)
{
	EditStars(2);
}

void
BuildFrame::Star4(wxCommandEvent& event)
{
	EditStars(3);
}

void
BuildFrame::Star5(wxCommandEvent& event)
{
	EditStars(4);
}

void
BuildFrame::Gg(wxCommandEvent& event)
{
	EditOrbit(TRUE);
}

void
BuildFrame::Mw(wxCommandEvent& event)
{
	EditMW();
}

void
BuildFrame::Belt(wxCommandEvent& event)
{
	EditOrbit(FALSE);
}

void
BuildFrame::Save(wxCommandEvent& event)
{
char *name;

	if(cfg.out_file == NULL) {
		wxString ptr;
		ptr = wxFileSelector("Save System File",
							  NULL,
							  NULL,
							  "*.sys",
							  "*.sys",
							  wxSAVE | wxOVERWRITE_PROMPT |
							  wxHIDE_READONLY,
							  this);
		name = (char *)ptr.GetData();
	} else
	    name = cfg.out_file;
	if(name != NULL)
		gen->Save(name, &cfg);
}

void
BuildFrame::Exit(wxCommandEvent& event)
{
	Close(TRUE);
}

void
BuildFrame::OnAbout(wxCommandEvent& event)
{
AboutDialog *dlg;

	dlg = new AboutDialog(this);
	dlg->ShowAbout("System Generator");
	delete dlg;
}

// -------------------------------------------------------------------------
void
BuildFrame::ReflectState()
{
BuildState curr;
wxPanel *old_tb;

    old_tb = panel;
    curr = gen->GetState();
    SetStatusText(state_msgs[curr]);

	switch(curr) {
		case BS_INIT:
		    panel = tb_init;
			tb_init->Show(TRUE);
			tb_stars->Show(FALSE);
			tb_known->Show(FALSE);
			tb_mw->Show(FALSE);
			tb_finish->Show(FALSE);
		    break;
		case BS_STAR:
		    panel = tb_stars;
			tb_init->Show(FALSE);
			tb_stars->Show(TRUE);
			tb_known->Show(FALSE);
			tb_mw->Show(FALSE);
			tb_finish->Show(FALSE);
		    break;
		case BS_KNOWN:
		    panel = tb_known;
			tb_init->Show(FALSE);
			tb_stars->Show(FALSE);
			tb_known->Show(TRUE);
			tb_mw->Show(FALSE);
			tb_finish->Show(FALSE);
		    break;
		case BS_MW_SELECT:
		    panel = tb_mw;
			tb_init->Show(FALSE);
			tb_stars->Show(FALSE);
			tb_known->Show(FALSE);
			tb_mw->Show(TRUE);
			tb_finish->Show(FALSE);
		    break;
		case BS_FINISH:
		    panel = tb_finish;
			tb_init->Show(FALSE);
			tb_stars->Show(FALSE);
			tb_known->Show(FALSE);
			tb_mw->Show(FALSE);
			tb_finish->Show(TRUE);
		    break;
	}

	if(old_tb != panel) {
		wxSizeEvent event(wxSize(-1, -1), -1);
		OnSize(event);
	}

	canvas->Refresh();
}

// ---------------------------------------------------------------
void
BuildFrame::EditOrbit(bool gg)
{
bool flag;

    if(select == NULL)
	    select = new SelectOrbDialog(this);

	if(gg)
	    flag = select->DoOrbSelect("Edit Gas Giant Orbit", &cfg, gen, TRUE);
	else
	    flag = select->DoOrbSelect("Edit Belt Orbit", &cfg, gen, FALSE);

	if(flag)
	    canvas->Refresh();
}

// ---------------------------------------------------------------
void
BuildFrame::EditMW(void)
{
int st=0,o=0,s=-1;

    if(gen->GetState() == BS_KNOWN) {
		if(!cfg.no_main) {
			int i;
			BuildStar *bs;

			for(i = 0;i < MAX_STARS;i++) {
				if((bs = gen->GetStar(i)) != NULL) {
					if(bs->GetMWOrbit() >= 0) {
						st = i;
						o = bs->GetMWOrbit();
					}
				}
			}

			if(move == NULL)
			    move = new MoveOrbDialog(this);

			if(move->GetOrbit("Main World Orbit", gen, &st, &o, TRUE)) {
				bs->SetMWOrbit(ORBIT_UNDEFINED);
				bs = gen->GetStar(st);
				bs->SetMWOrbit(o);
				canvas->Refresh();
			}
		}
	} else {
		gen->GetMWndx(&st, &o, &s);
		
		if(mw_sel == NULL)
		    mw_sel = new SelectMWDialog(this);

		if(mw_sel->SelectMW(gen, &st, &o, &s)) {
			gen->ClearMW();
			gen->GenMW(st, o, s);
		    canvas->Refresh();
		}
	}
}

// ---------------------------------------------------------------
void
BuildFrame::EditStars(int ndx)
{
BuildStar *bs;
bool test=TRUE;

	bs = gen->GetStar(ndx);
	if(ndx > 2) {
		if(gen->GetStar(ndx - 2) == NULL)
		    test = FALSE;
		else if(gen->GetStar(ndx - 2)->GetOrbit() <= ORBIT_FAR)
		    test = FALSE;
	} else if(cfg.stars[ndx][0] == 'N')
		test = FALSE;
	
	if(!test) {
		char buff[120];

		sprintf(buff, "Star %d is not editable!", ndx + 1);
		wxMessageBox(buff, "Error", wxOK | wxCENTRE | wxICON_EXCLAMATION,
					 this);
		return;
	}

    if(edit_star == NULL)
	    edit_star = new EditStarDialog(this);

	if(edit_star->GetStar(ndx, gen))
		canvas->Refresh();
}

// -------------------------------------------------------------------------
// Reposition the toolbar and child subwindow
void
BuildFrame::OnSize(wxSizeEvent& event)
{
int x, y, tb_height;

    if (NULL == panel) {
		wxFrame::OnSize(event);
		return;
	}

	GetClientSize(&x, &y);

	tb_height = (2 * DLG_OFFSET) + SML_BTN_SIZE;

	panel->SetSize(0, 0, x, tb_height);

	canvas->SetSize(0, tb_height, x, y - tb_height);
}

// -------------------------------------------------------------------------
#define BASE_OFFSET      5
#define GEN_TEXT_OFFSET  2
#define ORBIT_OFFSET     25
#define TEXT_INCR        11
#define BOX_OFFSET       5
#define STAR_BOX_X       215
#define STAR_BOX_Y1      72
#define STAR_BOX_Y2      108
#define STAR_BOX_Y3      46

void
BuildFrame::Draw(wxDC *dc)
{
int y=BASE_OFFSET,x=BASE_OFFSET;

    dc->SetFont(*NormalFont);

	DrawStar(0, dc, x, y);
}

int
BuildFrame::DrawStar(int ndx, wxDC *dc, int x, int y)
{
int i,j,k;
int ndx1=-1,ndx2=-1,orb1=-1,orb2=-1;
BuildStar *bs,*b1=NULL,*b2=NULL;
BuildOrbit *bo;
char buff1[120],buff2[120];

    if((bs = gen->GetStar(ndx)) != NULL) {
		// draw the star info
		if(ndx == 0) {
			if((b1 = gen->GetStar(1)) != NULL) {
				ndx1 = 1;
				orb1 = b1->GetOrbit();
			}
			if((b2 = gen->GetStar(2)) != NULL) {
				ndx2 = 2;
				orb2 = b1->GetOrbit();
			}
		} else if(ndx < 3) {
			if((b1 = gen->GetStar(ndx + 2)) != NULL) {
				ndx1 = ndx + 2;
				orb1 = b1->GetOrbit();
			}
		}

		switch(gen->GetState()) {
			case BS_KNOWN:
			    dc->DrawRectangle(x, y, STAR_BOX_X, STAR_BOX_Y2);
				break;
			case BS_STAR:
			    dc->DrawRectangle(x, y, STAR_BOX_X, STAR_BOX_Y1);	
				break;
			default:
			    dc->DrawRectangle(x, y, STAR_BOX_X, STAR_BOX_Y3);
				break;
		}
		
		if(ndx == 0)
		    sprintf(buff1, "Orbit: Primary  ");
		else {
			if(bs->GetOrbit() < 0)
			    sprintf(buff1, "Orbit: Close");
			else
			    sprintf(buff1, "Orbit: %d  ", bs->GetOrbit());
		}
		strcat(buff1, bs->GetStarStr(buff2));
		dc->DrawText(buff1,
					 x + GEN_TEXT_OFFSET,
					 y + GEN_TEXT_OFFSET);

		if(bs->GetNumOrbits() > -1)
		    sprintf(buff1, "First Orbit: %d  Last Orbit: %d",
					bs->GetFirstOrbit(), bs->GetNumOrbits());
		else
		    sprintf(buff1, "First Orbit: %d  Last Orbit: ",
					bs->GetFirstOrbit());
		dc->DrawText(buff1,
					 x + GEN_TEXT_OFFSET,
					 y + (GEN_TEXT_OFFSET * 2) + TEXT_INCR);

		if(bs->GetHabZone() > -1)
		    sprintf(buff1, "Habitible Zone: %d",
					bs->GetHabZone());
		else
		    sprintf(buff1, "Habitible Zone: ");
		dc->DrawText(buff1,
					 x + GEN_TEXT_OFFSET,
					 y + (GEN_TEXT_OFFSET * 3) + (TEXT_INCR * 2));
			
		if((gen->GetState() == BS_KNOWN) || (gen->GetState() == BS_STAR)) {
			sprintf(buff1, "Captured Orbits: ");
			for(j = 0;j < 3;j++) {
				if((k = bs->GetCapt(j)) > -1) {
					sprintf(buff2, "%d ", k);
					strcat(buff1, buff2);
				}
			}
			dc->DrawText(buff1,
						 x + GEN_TEXT_OFFSET,
						 y + (GEN_TEXT_OFFSET * 4) + (TEXT_INCR * 3));

			sprintf(buff1, "Empty Orbits: ");
			for(j = 0;j < 3;j++) {
				if((k = bs->GetEmpty(j)) > -1) {
					sprintf(buff2, "%d ", k);
					strcat(buff1, buff2);
				}
			}
			dc->DrawText(buff1,
						 x + GEN_TEXT_OFFSET,
						 y + (GEN_TEXT_OFFSET * 5) + (TEXT_INCR * 4));
		}
		
		// add orbit info
		if(gen->GetState() == BS_KNOWN) {
			sprintf(buff1, "GG: ");
			for(j = 0;j < MAX_GG;j++) {
				if((k = bs->GetGGOrbit(j)) > - 1) {
					sprintf(buff2, "%d ", k);
					strcat(buff1, buff2);
				}
			}
			dc->DrawText(buff1,
						 x + GEN_TEXT_OFFSET,
						 y + (GEN_TEXT_OFFSET * 6) + (TEXT_INCR * 5));

			sprintf(buff1, "Belts: ");
			for(j = 0;j < MAX_BELTS;j++) {
				if((k = bs->GetBeltOrbit(j)) > - 1) {
					sprintf(buff2, "%d ", k);
					strcat(buff1, buff2);
				}
			}
			dc->DrawText(buff1,
						 x + GEN_TEXT_OFFSET,
						 y + (GEN_TEXT_OFFSET * 7) + (TEXT_INCR * 6));

			sprintf(buff1, "Main World: ");
			if((k = bs->GetMWOrbit()) > - 1) {
				sprintf(buff2, "%d ", k);
				strcat(buff1, buff2);
			}
			dc->DrawText(buff1,
					     x + GEN_TEXT_OFFSET,
						 y + (GEN_TEXT_OFFSET * 8) + (TEXT_INCR * 7));
		}

		switch(gen->GetState()) {
			case BS_KNOWN:
			    y += STAR_BOX_Y2 + 10;
				break;
			case BS_STAR:
			    y += STAR_BOX_Y1 + 10;
				break;
			default:
			    y += STAR_BOX_Y3 + 10;
				break;
		}
		
		x += ORBIT_OFFSET;
		i = 0;
		while((bo = bs->GetWorldOrbit(i)) != NULL) {
			if(ndx1 > -1) {
				if(orb1 < bo->GetOrbit()) {
					y = DrawStar(ndx1, dc, x, y);
					ndx1 = -1;
				}
			}
			if(ndx2 > -1) {
				if(orb2 < bo->GetOrbit()) {
					y = DrawStar(ndx2, dc, x, y);
					ndx2 = -1;
				}
			}
			
			y = DrawOrbit(bo, dc, x, y);
			i++;
		}

		if(ndx1 > -1)
			y = DrawStar(ndx1, dc, x, y);

		if(ndx2 > -1)
		    y = DrawStar(ndx2, dc, x, y);
	}
	
	return(y);
}

int
BuildFrame::DrawOrbit(BuildOrbit *bo, wxDC *dc, int x, int y)
{
char buff1[120],buff2[40],buff3[40],*ptr;
int i=0;
BuildOrbit *sat;

    if((ptr = bo->GetName()) != NULL)
	  sprintf(buff1, "%s  %s %s  %s", bo->FormatOrbit(buff2), ptr,
			  bo->GetData(),
			  bo->GetFac(buff3));
	else
	  sprintf(buff1, "%s  %s  %s", bo->FormatOrbit(buff2), bo->GetData(),
			  bo->GetFac(buff3));
	dc->DrawText(buff1,
				 x,
				 y - (TEXT_INCR - GEN_TEXT_OFFSET));
	//fprintf(stderr, "%d %s\n", y, buff);
	x += ORBIT_OFFSET;
	y += TEXT_INCR;
	while((sat = bo->GetSat(i)) != NULL) {
		y = DrawOrbit(sat, dc, x, y);
		i++;
	}

	return(y);
}

// =================================================================
// ===============================================================
// ===============================================================
SysGenCanvas::SysGenCanvas(wxWindow *frame,
						   int x, int y, int w, int h,
						   long style) :
    wxScrolledWindow(frame, -1, wxPoint(x, y), wxSize(w, h), style)
{
}

SysGenCanvas::~SysGenCanvas()
{
}

void
SysGenCanvas::OnDraw(wxDC& dc)
{
	if(frame != NULL) {
		dc.Clear();
		frame->Draw(&dc);
	}
	else
	    dc.Clear();
}

// =================================================================

