// File:        secgen.cpp
// Purpose:     sector generator
// Author:
// Created:
// Updated:
// Copyright:   LGPL
// Traveller is a registered trademark of Far Future Enterprises.
// Portions based upon material Copyright 1977-2002 Far Future Enterprises.
// 
// rcsid[] = "$RCSfile: secgen.cpp,v $ $Revision: 1.3 $ $Author: man $ $Date: 2002/06/05 04:51:07 $"

// For compilers that support precompilation, includes "wx.h".
#include "wx/wxprec.h"

#ifndef WX_PRECOMP
#include "wx/wx.h"
#endif

//#include "wx/txtstrm.h"
#include "wx/socket.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "secgen.h"
#include "t_dlg.h"
#include "t_res.h"
#include "parse.h"

#ifdef __WXGTK__
#include "bitmaps/sector.xpm"
#endif

IMPLEMENT_APP(App)

// handy references (AKA stupid globals)
App *app=NULL;
Frame *frame=NULL;

static bool l_init = FALSE;

// ===============================================================
// `Main program' equivalent, creating windows and returning main app frame
bool
App::OnInit(void)
{
struct secgen_conf *cfg;
char config_file[MAX_FILE_LENGTH];
char *ptr1;
//int c_port=-1;
wxConfig *config;
wxString ptr;

	app = this;
    // initialize cfg with defaults
	cfg = new secgen_conf;			// XXX deleted in frame dtor
    cfg->name = NULL;
	cfg->file_name = NULL;
	cfg->no_sleep = FALSE;
	cfg->gen_stars = TRUE;
	cfg->gen_sys = FALSE;
	cfg->gen_routes = FALSE;
	cfg->gen_zone = FALSE;
	cfg->import = IM_NONE;
	cfg->density[0] = cfg->density[1] = cfg->density[2] = cfg->density[3] = 
		cfg->density[4] = cfg->density[5] = cfg->density[6] = cfg->density[3] = 
		SPARSE_DENSITY;
//	cfg->density[0] = DENSE_DENSITY;
//	cfg->density[1] = RIFT_DENSITY;
//	cfg->density[2] = SPARSE_DENSITY;
//	cfg->density[3] = RIFT_DENSITY;
//	cfg->density[4] = DENSE_DENSITY;
//	cfg->density[5] = RIFT_DENSITY;
//	cfg->density[6] = SPARSE_DENSITY;
//	cfg->density[7] = RIFT_DENSITY;
//	cfg->maturity[0] = BACKWATER_MATURE;
//	cfg->maturity[1] = CLUSTER_MATURE;
//	cfg->maturity[2] = STANDARD_MATURE;
//	cfg->maturity[3] = STANDARD_MATURE;
	cfg->maturity[0] = cfg->maturity[1] = cfg->maturity[2] = cfg->maturity[3] = 
		cfg->maturity[4] = cfg->maturity[5] = cfg->maturity[6] = 
		cfg->maturity[7] = STANDARD_MATURE;
	cfg->d3 = FALSE;
	cfg->min_x = -20;
	cfg->min_y = -20;
	cfg->min_z = -20;
	cfg->max_x = 20;
	cfg->max_y = 20;
	cfg->max_z = 20;
	// XXX sysgen stuff....
	cfg->sys.name = NULL;
	cfg->sys.sect = NULL;
	cfg->sys.ssect = NULL;
	cfg->sys.loc = NULL;
	cfg->sys.x = cfg->sys.y = cfg->sys.z = 0;
	cfg->sys.use_xyz = FALSE;
	cfg->sys.belts = -1;
	cfg->sys.gg = -1;
	cfg->sys.port = 0;
	memcpy(cfg->sys.uwp, "\0\0\0\0\0\0\0\0", sizeof(cfg->sys.uwp));
	memcpy(cfg->sys.stars[0], "R\0\0\0\0\0\0", sizeof(cfg->sys.stars[0]));
	memcpy(cfg->sys.stars[1], "R\0\0\0\0\0\0", sizeof(cfg->sys.stars[1]));
	memcpy(cfg->sys.stars[2], "R\0\0\0\0\0\0", sizeof(cfg->sys.stars[2]));
	memcpy(cfg->sys.bases, "\0", sizeof(cfg->sys.bases));
	memcpy(cfg->sys.alleg, "\0\0", sizeof(cfg->sys.alleg));
	memcpy(cfg->sys.zone, "\0", sizeof(cfg->sys.zone));
	cfg->sys.lang = NULL;
	cfg->sys.dirs = new MTUFile();
	cfg->sys.out_file = NULL;
	cfg->sys.quiet_mode = FALSE;
	cfg->sys.no_sleep = FALSE;
	cfg->sys.run_editor = FALSE;
	cfg->sys.interactive = FALSE;
	cfg->sys.no_main = TRUE;
	cfg->sys.full_rand = FALSE;
	cfg->sys.tech_method = TG_STANDARD;
	cfg->sys.tech_min = -1;
	cfg->sys.tech_max = -1;

	config = cfg->sys.dirs->GetConfig();
	ptr1 = cfg->sys.dirs->GetMTUDataDir();
	cfg->codes = new TCodes(ptr1);

	sprintf(config_file, "%s/%s", ptr1, JUMP_FILE);
	cfg->jumps = new JumpRoute(config_file);
	sprintf(config_file, "%s/%s", ptr1, PORT_FILE);
	cfg->ports = new PortGen(config_file);

	// fill in the rest
	ParseCommandLine(cfg);

    frame = new Frame(cfg);
	// Give it an icon
#ifdef __WINDOWS__
	frame->SetIcon(wxIcon("sector_icn"));
#endif
#ifdef __WXGTK__
	frame->SetIcon(wxIcon(sector_xpm));
#endif

	frame->Show(TRUE);
	
	l_init = TRUE;
//	delete config;

	return(TRUE);
}

void 
App::Usage(void)
{
	fprintf(stderr, "Valid Options:\n"
				" -n <sector name>\n"
				" -F <sector file name (without extension>\n"
				" -D <sector wide density (0-4)>\n"
				" -M <sector wide maturity (0-3)>\n"
				" -d <corner (0-3;0-7 for 3D):density (0-4)>\n"
				" -m <corner (0-3;0-7 for 3D):maturity (0-3)>\n"
				" -L <language file>\n"
				" -x <min x value for 3D>\n"
				" -y <min y value for 3D>\n"
				" -z <min z value for 3D>\n"
				" -X <min x value for 3D>\n"
				" -Y <min y value for 3D>\n"
				" -Z <min z value for 3D>\n"
				"flags:\n"
				" -S do not generate stellar info\n"
				" -s generate system info for each world\n"
				" -R generate routes\n"
				" -o generate (red/amber) zones\n"
				" -l do not sleep\n"
				" -3 generate a 3D sector\n"
				" -q\n");
}

void
App::ParseCommandLine(struct secgen_conf *cfg)
{
int i=1;
char buff[120], last_arg=0;

    buff[0] = 0;
    while(i < argc) {
		if('-' == argv[i][0]) {
			if(strchr("nFDMdmLxyzXYZSsRol3q", argv[i][1]) == NULL) {
				fprintf(stderr, "BadOption: %c\n", argv[i][1]);
				Usage();
				exit(-1);
			}

			if(last_arg != 0)
				ProcessArgs(cfg, last_arg, buff);

			last_arg = argv[i][1];
			buff[0] = 0;
			if(argv[i][2] != 0)
			    strcat(buff, &argv[i][2]);
		}
		else
		    strcat(buff, argv[i]);
		i++;
	}

	if(last_arg != 0)
	    ProcessArgs(cfg, last_arg, buff);

	// last sanity checks
	if(NULL == cfg->file_name) {
		fprintf(stderr, "No output file name given.  Exiting.\n");
		Usage();
		exit(-1);
	}
	if(NULL == cfg->name) {
		if(NULL == cfg->sys.lang) {
			fprintf(stderr, 
					"No sector name nor valid language given.  Exiting.\n");
			Usage();
			exit(-1);
		} else {
			cfg->sys.lang->GenerateWord(buff);
			cfg->name = new char[strlen(buff) + 1];
			strcpy(cfg->name, buff);
		}
	}
	// XXX yukky
	if(cfg->d3) {
		char *tmp;

		tmp = new char[strlen(cfg->file_name) + strlen(".sec") + 1];
		sprintf(tmp, "%s%s", cfg->file_name, ".sec");
		delete cfg->file_name;
		cfg->file_name = tmp;
	}
}

void
App::ProcessArgs(struct secgen_conf *cfg, char last_arg, char *buff)
{
bool no_err=TRUE;
char *ptr=NULL;

    switch(last_arg) {
		case 'n':
			no_err = ParseString(&cfg->name, buff);
			break;
		case 'F':
			no_err = ParseString(&cfg->file_name, buff);
			break;
		case 'D':
			cfg->density[0] = cfg->density[1] = cfg->density[2] = 
				cfg->density[3] = cfg->density[4] = cfg->density[5] = 
				cfg->density[6] = cfg->density[7] = (SYSTEM_DENSITY)atoi(buff);
			break;
		case 'M':
			cfg->maturity[0] = cfg->maturity[1] = cfg->maturity[2] = 
				cfg->maturity[3] = cfg->maturity[4] = cfg->maturity[5] = 
				cfg->maturity[6] = cfg->maturity[7] = (TRAVEL_MATURE)atoi(buff);
			break;
		case 'd':
			if(buff[1] != ':')
				no_err = FALSE;
			else {
				// XXX
				cfg->density[buff[0]-'0'] = (SYSTEM_DENSITY)(buff[2] - '0');
			}
			break;
		case 'm':
			if(buff[1] != ':')
				no_err = FALSE;
			else {
				// XXX
				cfg->maturity[buff[0]-'0'] = (TRAVEL_MATURE)(buff[2] - '0');
			}
			break;
		case 'L':
			if(no_err = ParseString(&ptr, buff)) {
				cfg->sys.lang = new WordGen(ptr);
				no_err = cfg->sys.lang->Valid();
			}
			delete ptr;
			break;
		case 'x':
			cfg->min_x = atoi(buff);
			break;
		case 'y':
			cfg->min_y = atoi(buff);
			break;
		case 'z':
			cfg->min_z = atoi(buff);
			break;
		case 'X':
			cfg->max_x = atoi(buff);
			break;
		case 'Y':
			cfg->max_y = atoi(buff);
			break;
		case 'Z':
			cfg->max_z = atoi(buff);
			break;
		case 'S':
			cfg->gen_stars = FALSE;
			break;
		case 's':
			cfg->gen_sys = TRUE;
			break;
		case 'R':
			cfg->gen_routes = TRUE;
			break;
		case 'o':
			cfg->gen_zone = TRUE;
			break;
		case 'l':
			cfg->no_sleep = TRUE;
			break;
		case '3':
			cfg->d3 = TRUE;
			break;
		case 'q':
			// ???
			break;
	}
			
	if(!no_err) {
		fprintf(stderr, "Problem with parameter: %c\n", last_arg);
		exit(-2);
	}
}


bool
App::ParseString(char **dest, char *src)
{
    if(*dest != NULL)
	    return(FALSE);
	if(src == NULL)
	    return(TRUE);
	*dest = new char[strlen(src) + 1];
	strcpy(*dest, src);
	return(TRUE);
}

bool
App::CopyString(char *dest, char *src, unsigned int sz)
{
    if(strlen(src) > sz)
	    return(FALSE);
	strcpy(dest, src);
	return(TRUE);
}

// ===============================================================
Timer::Timer() : 
	wxTimer() 
{
}

Timer::~Timer() 
{
}

void
Timer::Notify(void)
{
	if(l_init != FALSE) {
		Stop();
		frame->Proceed();
		frame->Close(TRUE);
	}
	wxYield();
}


// ===============================================================
#define BUILD_FRAME_WIDTH		500
#define BUILD_FRAME_HEIGHT		400
#define BUILD_PANEL_HEIGHT		 80
#define QUIET_FRAME_WIDTH		300
#define QUIET_FRAME_HEIGHT		150

#define SOCKET_ID				600

// ---------------------------------------------------------------
BEGIN_EVENT_TABLE(Frame, wxFrame)
//	EVT_IDLE(Frame::OnIdle)
//	EVT_SOCKET(SOCKET_ID, Frame::OnSocketEvent)
END_EVENT_TABLE()

//Frame::Frame(struct secgen_conf *config, int p) :
Frame::Frame(struct secgen_conf *config) :
	wxFrame(NULL, -1, "Generate Sector", wxPoint(0, 0), 
			wxSize(BUILD_FRAME_WIDTH, BUILD_FRAME_HEIGHT))
{
wxPanel *panel;

	timer = new Timer();
//	process = NULL;
	cfg = config;
//	c_port = p;
	if(cfg->d3) {
		cur_x = cfg->min_x;
		cur_y = cfg->min_y;
		cur_z = cfg->min_z;
		loc_count = 0;
		loc_tot = (cfg->max_x - cfg->min_x + 1) *
			(cfg->max_y - cfg->min_y + 1) * (cfg->max_z - cfg->min_z + 1);
	}
	sect = new DetailSector(cfg->codes);
	sect3d = new Sector3D(cfg->codes, cfg->name);
	sys_gen = new SystemGenerate(cfg->sys.dirs->GetMTUDataDir(),
					cfg->sys.tech_method, cfg->sys.tech_min, cfg->sys.tech_max);
	if(cfg->file_name != NULL)
		sect3d->SetName(cfg->file_name);

	panel = new wxPanel(this);
	msg1 = new wxStaticText(panel, -1, "Initializing....",
		wxPoint(DLG_OFFSET, DLG_OFFSET),
		wxSize(QUIET_FRAME_WIDTH - (3 * DLG_OFFSET), -1));
	g1 = new wxGauge(panel, -1, 100, 
		wxPoint(DLG_OFFSET, (0 * DLG_OFFSET) + (1 * ROW_OFFSET)),
		wxSize(QUIET_FRAME_WIDTH - (3 * DLG_OFFSET), BUILD_PANEL_HEIGHT - 50),
		wxGA_HORIZONTAL);
	g1->SetShadowWidth(3);
	g1->SetValue(0);
//	g1->SetBackgroundColour(*wxRED);
//	g1->SetForegroundColour(*wxCYAN);
	msg2 = new wxStaticText(panel, -1, "Initializing....",
		wxPoint(DLG_OFFSET, (4 * DLG_OFFSET) + (2 * ROW_OFFSET)),
		wxSize(QUIET_FRAME_WIDTH - (3 * DLG_OFFSET), -1));
	g2 = new wxGauge(panel, -1, 100, 
		wxPoint(DLG_OFFSET, (3 * DLG_OFFSET) + (3 * ROW_OFFSET)),
		wxSize(QUIET_FRAME_WIDTH - (3 * DLG_OFFSET), BUILD_PANEL_HEIGHT - 50),
		wxGA_HORIZONTAL);
	g2->SetShadowWidth(3);
	g2->SetValue(0);
//	g2->SetBackgroundColour(*wxRED);
//	g2->SetForegroundColour(*wxCYAN);

	Fit();
	SetSize(QUIET_FRAME_WIDTH, QUIET_FRAME_HEIGHT);

	timer->Start(500, FALSE);

}

Frame::~Frame()
{
	if(sect != NULL)
		delete sect;
	if(sect3d != NULL)
		delete sect3d;
	if(sys_gen != NULL)
		delete sys_gen;

	if(cfg->name != NULL)
		delete cfg->name;
	if(cfg->file_name != NULL)
		delete cfg->file_name;
	if(cfg->codes != NULL)
		delete cfg->codes;
	if(cfg->ports != NULL)
		delete cfg->ports;
	if(cfg->jumps != NULL)
		delete cfg->jumps;

	if(cfg->sys.name != NULL)
	    delete cfg->sys.name;
	if(cfg->sys.sect != NULL)
	    delete cfg->sys.sect;
	if(cfg->sys.ssect != NULL)
	    delete cfg->sys.ssect;
	if(cfg->sys.loc != NULL)
	    delete cfg->sys.loc;
	if(cfg->sys.lang != NULL)
	    delete cfg->sys.lang;
	if(cfg->sys.out_file != NULL)
	    delete cfg->sys.out_file;
	if(cfg->sys.dirs != NULL)
		delete cfg->sys.dirs;

	delete cfg;						// XXX
}

// ---------------------------------------------------------------------
static char *state_msgs[] = {
	"Initializing....",
	"Setting Names",
	"Determining System Presenses",
	"Determining UWPs",
//	"Determining Bases",
	"Determining Trade Classifications",
	"Determining PGB's",
	"Determining Zone",
	"Determining Stars",
	"Determining Travel Zones",
	"Checking Routes",
	"Writing File(s)",
	"Quiting"
};

void
Frame::Proceed(void)
{
int i=0;

	while(TRUE) {
		if(cfg->d3) {
			int ret;

			FeedBack3D(cur_x, cur_y, cur_z, loc_count, loc_tot, 0);
			ret = Gen3D();
			if(ret < 0)
				// error
				break;
			else if(0 == ret) {
				FeedBack3D(cur_x, cur_y, cur_z, loc_count, loc_tot, 99);
				sect3d->Write();
				break;
			}
		} else {
			msg1->SetLabel(state_msgs[i]);
			g1->SetValue((i + 1) * (100 / GS_WRITE));
			wxYield();
			i++;
			if(!NextState())
				break;
		}
		if(!cfg->no_sleep)
			wxSleep(1);
	}
	wxYield();
	if(!cfg->no_sleep)
		wxSleep(1);
	wxYield();
}

bool
Frame::NextState(void)
{
bool ret = TRUE;

//fprintf(stderr, "NextState(): %d\n", state);
	switch(state) {
		case GS_INIT:
//			if(StartSysgen(TRUE))
				state = GS_NAMING;
//			else			
//				ret = FALSE;
			break;
		case GS_NAMING:
			SetNaming();
			state = GS_PRES;
			break;
		case GS_PRES:
			if(CheckPresence())
				state = GS_UWP;
			else
				ret = FALSE;
			break;
		case GS_UWP:
			if(SetUWP())
				state = GS_TRADE;
			else
				ret = FALSE;
			break;
		case GS_TRADE:
			CheckTrade();
			state = GS_PGB;
			break;
		case GS_PGB:
			SetPGB();
			if(cfg->gen_sys)
				state = GS_ZONE;
			else
				state = GS_STARS;
			break;
		case GS_STARS:
			if(SetStars())
				state = GS_ZONE;
			else
				ret = FALSE;
			break;
		case GS_ZONE:
			SetZone();
			state = GS_ROUTE;
			break;
		case GS_ROUTE:
			SetRoute();
			state = GS_WRITE;
			break;
		case GS_WRITE:
			Write();
		default:
			ret = FALSE;
			break;
	}

	return(ret);
}

// ---------------------------------------------------------------
// 3D code
int
Frame::Gen3D(void)
{
	FeedBack3D(cur_x, cur_y, cur_z, loc_count, loc_tot, 33);
	if(CheckPresence3D()) {
		FeedBack3D(cur_x, cur_y, cur_z, loc_count, loc_tot, 67);
		if(!GenStar3D())
			return(-1);
	}

	cur_x++;
	if(cur_x > cfg->max_x) {
		cur_x = cfg->min_x;
		cur_y++;
		if(cur_y > cfg->max_y) {
			cur_y = cfg->min_y;
			cur_z++;
			if(cur_z > cfg->max_z) {
				return(0);
			}
		}
	}

	FeedBack3D(cur_x, cur_y, cur_z, loc_count, loc_tot, 99);
	return(1);
}

//#define DEBUG_ROLL
bool
Frame::CheckPresence3D(void)
{
int edge[12],face[6];
SYSTEM_DENSITY val;

	edge[0] = (cfg->density[0] + cfg->density[1]) / 2;
	edge[1] = (cfg->density[0] + cfg->density[2]) / 2;
	edge[2] = (cfg->density[2] + cfg->density[3]) / 2;
	edge[3] = (cfg->density[1] + cfg->density[3]) / 2;
	edge[4] = (cfg->density[4] + cfg->density[5]) / 2;
	edge[5] = (cfg->density[4] + cfg->density[6]) / 2;
	edge[6] = (cfg->density[6] + cfg->density[7]) / 2;
	edge[7] = (cfg->density[5] + cfg->density[7]) / 2;
	edge[8] = (cfg->density[0] + cfg->density[4]) / 2;
	edge[9] = (cfg->density[2] + cfg->density[6]) / 2;
	edge[10] = (cfg->density[3] + cfg->density[7]) / 2;
	edge[11] = (cfg->density[1] + cfg->density[5]) / 2;

	face[0] = (edge[0] + edge[1] + edge[2] + edge[3]) / 4;
	face[1] = (edge[1] + edge[5] + edge[8] + edge[9]) / 4;
	face[2] = (edge[2] + edge[6] + edge[9] + edge[10]) / 4;
	face[3] = (edge[3] + edge[7] + edge[10] + edge[11]) / 4;
	face[4] = (edge[0] + edge[4] + edge[8] + edge[11]) / 4;
	face[5] = (edge[4] + edge[5] + edge[6] + edge[7]) / 4;

	val = (SYSTEM_DENSITY) ((face[0] + face[1] + face[2] + face[3] + face[4] + face[5]) / 6);

#ifdef DEBUG_ROLL
	fprintf(stderr, "D: (%d %d %d): ", cur_x, cur_y, cur_z);
#endif
	if(RollPresence(val))
		return(TRUE);

	return(FALSE);
}

bool
Frame::GenStar3D(void)
{
char name_buff[80],star_buff[3][20];
int x,y,z;
Star3D *st;

	// nice d6's
	x = (cur_x * 10) + dice.Roll(6, 2) - 7;
	y = (cur_y * 10) + dice.Roll(6, 2) - 7;
	z = (cur_z * 10) + dice.Roll(6, 2) - 7;

	star_buff[0][0] = star_buff[1][0] = star_buff[2][0] = 0;
	if(cfg->sys.lang != NULL) 
		cfg->sys.lang->GenerateWord(name_buff);
	else 
		sprintf(name_buff, "%d %d %d", x, y, z);

	if(cfg->gen_sys) {
		char star_buff[3][20],loc_buff[8],file_buff[MAX_FILE_LENGTH];
		int i;
		BuildStar *bs;

		cfg->sys.sect = cfg->name;
		cfg->sys.ssect = NULL;
		sprintf(loc_buff, "%02d%02d", x+1, y+1);
		cfg->sys.loc = loc_buff;
		cfg->sys.belts = cfg->sys.gg = -1;
		cfg->sys.port = 0;
		memcpy(cfg->sys.uwp, "\0\0\0\0\0\0\0\0", sizeof(cfg->sys.uwp));
		memcpy(cfg->sys.stars[0], "R\0\0\0\0\0\0", sizeof(cfg->sys.stars[0]));
		memcpy(cfg->sys.stars[1], "R\0\0\0\0\0\0", sizeof(cfg->sys.stars[1]));
		memcpy(cfg->sys.stars[2], "R\0\0\0\0\0\0", sizeof(cfg->sys.stars[2]));
		memcpy(cfg->sys.bases, "\0", sizeof(cfg->sys.bases));
		memcpy(cfg->sys.alleg, "\0\0", sizeof(cfg->sys.alleg));
		memcpy(cfg->sys.zone, "\0", sizeof(cfg->sys.zone));

		if(cfg->sys.dirs->GetSectSysDir() == NULL)
			sprintf(file_buff, "x%dy%dz%d.sys", x, y, z);
		else
			sprintf(file_buff, "%s/x%dy%dz%d.sys", 
				cfg->sys.dirs->GetSectSysDir(), x, y, z);
fprintf(stderr, "<%s>\n", file_buff);

		sys_gen->Cancel(&cfg->sys);
		sys_gen->End(&cfg->sys);

		sys_gen->Save(file_buff, &cfg->sys);

		for(i = 0;i < 3;i++) {
			if((bs = sys_gen->GetStar(i)) != NULL) {
				bs->GetStarStr(star_buff[i]);
			}
		}
		cfg->sys.sect = NULL;
		cfg->sys.loc = NULL;
	} else {
		int max,t_roll,s_roll,st_ndx;
		GenStar *gs;

		gs = sys_gen->GetGenStar();
		max = gs->GenSystemNature();
		t_roll = s_roll = 0;
		for(st_ndx = 0;st_ndx < max;st_ndx++)
			gs->GenerateStar(st_ndx, &t_roll, &s_roll, NULL, star_buff[st_ndx]);
	}

	st = new Star3D(x, y, z, name_buff, star_buff[0]);
	if(star_buff[1][0] != 0)
		st->AddStar(name_buff, star_buff[1]);
	if(star_buff[2][0] != 0)
		st->AddStar(name_buff, star_buff[2]);
	sect3d->Append(st);
	world_count++;

	return(TRUE);
}

// ---------------------------------------------------------------
/*
	Naming -- the sector's name could be on the command line
	if there's a valid word file, generate a name for the
	sector and each subsector
*/
void 
Frame::SetNaming(void)
{
char buff[MAX_FILE_LENGTH];
int i;

	if(cfg->name != NULL)
		sect->SetName(cfg->name);
	if(cfg->file_name != NULL) {
		sprintf(buff, "%s.sec", cfg->file_name);
		sect->SetSecName(buff);
		sprintf(buff, "%s.dat", cfg->file_name);
		sect->SetDatName(buff);
	}
	if(cfg->sys.lang != NULL) {
		if(NULL == cfg->name)
			sect->SetName(cfg->sys.lang->GenerateWord(buff));
		for(i = 0;i < MAX_SS;i++) {
			sect->SetSSName(i, cfg->sys.lang->GenerateWord(buff));
		}
	}
	cfg->sys.dirs->CrackNames(sect->GetSecName());
}

//#define DEBUG_DENS
//#define DEBUG_PRES
bool 
Frame::CheckPresence(void)
{
int x,y;

	world_count = 0;
	if(IM_PRES == cfg->import) {
		// don't need to do anything yet....
		return(TRUE);
	} else if(IM_PRES == cfg->import) {
		if(!ImportFile())
			return(FALSE);
	} else if(IM_NONE == cfg->import) {
#ifdef DEBUG_DENS
		fprintf(stderr, "%d %d\n%d %d\n", cfg->density[0], cfg->density[1],
			cfg->density[2], cfg->density[3]);
#endif
		BuildMap(cfg->density[0], cfg->density[1], 
			cfg->density[2], cfg->density[3]);
#ifdef DEBUG_DENS
		for(y = 0;y < MAX_Y_HEX;y++) {
			for(x = 0;x < MAX_X_HEX;x++) {
				fprintf(stderr, "%d ", sec_map[x][y]);
			}
			fprintf(stderr, "\n");
		}
#endif

		for(x = 0;x < MAX_X_HEX;x++) {
			for(y = 0;y < MAX_Y_HEX;y++) {
#ifdef DEBUG_ROLL
				fprintf(stderr, "D: %02d%02d: ", x, y);
#endif
				if(RollPresence((SYSTEM_DENSITY) sec_map[x][y])) {
					sec_map[x][y] = 1;
					world_count++;
					sect->AddOrGetHex(x, y);
				} else {
					sec_map[x][y] = 0;
				}
				FeedBack(x, y, (x * MAX_Y_HEX) + y, MAX_X_HEX * MAX_Y_HEX);
			}
		}
	}

#ifdef DEBUG_PRES
	for(y = 0;y < MAX_Y_HEX;y++) {
		for(x = 0;x < MAX_X_HEX;x++) {
			fprintf(stderr, "%d ", sec_map[x][y]);
		}
		fprintf(stderr, "\n");
	}
#endif
	FeedBack(-1, -1, 0, 0);
	return(TRUE);
}

/*
	SetUWP
	Note that this (and the subsequent star generation) use the lib fcns.
*/
//#define DEBUG_MAT
//#define DEBUG_PORT
bool 
Frame::SetUWP(void)
{
bool done;
// XXX these buffer size should be looked at
char buff[MAX_FILE_LENGTH];
char uwp_buff[20];
char *ptr;
int x,y,count=0,i;
TRAVEL_MATURE mat;
HexData *hd;
UWPGen *ug;

	if(IM_PORT == cfg->import) {
		if(!ImportFile())
			return(FALSE);
	} else {
#ifdef DEBUG_MAT
		fprintf(stderr, "%d %d\n%d %d\n", cfg->maturity[0], cfg->maturity[1],
			cfg->maturity[2], cfg->maturity[3]);
#endif
		BuildMap(cfg->maturity[0], cfg->maturity[1], cfg->maturity[2],
			cfg->maturity[3]);

#ifdef DEBUG_MAT
		for(y = 0;y < MAX_Y_HEX;y++) {
			for(x = 0;x < MAX_X_HEX;x++) {
				fprintf(stderr, "%d ", sec_map[x][y]);
			}
			fprintf(stderr, "\n");
		}
#endif

	// generate ports
		for(x = 0;x < MAX_X_HEX;x++) {
			for(y = 0;y < MAX_Y_HEX;y++) {
				if(sect->GetHex(x, y) != NULL) {
					mat = (TRAVEL_MATURE) sec_map[x][y];
					sec_map[x][y] = cfg->ports->GenPort(mat);
				} else
					sec_map[x][y] = ' ';
			}
		}
#ifdef DEBUG_PORT
		for(y = 0;y < MAX_Y_HEX;y++) {
			for(x = 0;x < MAX_X_HEX;x++) {
				fprintf(stderr, "%c ", sec_map[x][y]);
			}
			fprintf(stderr, "\n");
		}
#endif
	}

	ug = sys_gen->GetGenUWP();
	ptr = cfg->sys.dirs->GetSectSysDir();

	// for each hex generate a uwp
	count = 0;
	for(x = 0;x < MAX_X_HEX;x++) {
		for(y = 0;y < MAX_Y_HEX;y++) {
			if(sec_map[x][y] != ' ') {
				hd = sect->GetHex(x, y);
				done = FALSE;
				if(cfg->sys.lang != NULL) 
					cfg->sys.lang->GenerateWord(buff);
				else 
					sprintf(buff, "%02d%02d", x+1, y+1);

				if(!cfg->gen_sys) {
					ug->GenUWP(sec_map[x][y], uwp_buff);
					hd->world = new main_world(buff, x, y, uwp_buff);
				} else {
					char star_buff[3][20],loc_buff[8];
					char file_buff[MAX_FILE_LENGTH];
					BuildOrbit *bo;
					BuildStar *bs;

					if(sect->GetName() != NULL)
						cfg->sys.sect = sect->GetName();
					i = ((y / 10) * 4) + (x / 8);
					if(sect->GetSSName(i) != NULL)
						cfg->sys.ssect = sect->GetSSName(i);
					sprintf(loc_buff, "%02d%02d", x+1, y+1);
					cfg->sys.loc = loc_buff;
					cfg->sys.belts = cfg->sys.gg = -1;
					cfg->sys.port = sec_map[x][y];
					memcpy(cfg->sys.uwp, "\0\0\0\0\0\0\0\0", 
									sizeof(cfg->sys.uwp));
					memcpy(cfg->sys.stars[0], "R\0\0\0\0\0\0", 
									sizeof(cfg->sys.stars[0]));
					memcpy(cfg->sys.stars[1], "R\0\0\0\0\0\0", 
									sizeof(cfg->sys.stars[1]));
					memcpy(cfg->sys.stars[2], "R\0\0\0\0\0\0", 
									sizeof(cfg->sys.stars[2]));
					memcpy(cfg->sys.bases, "\0", 
									sizeof(cfg->sys.bases));
					memcpy(cfg->sys.alleg, "\0\0", 
									sizeof(cfg->sys.alleg));
					memcpy(cfg->sys.zone, "\0", 
									sizeof(cfg->sys.zone));

					if(NULL == ptr)
						sprintf(file_buff, "%02d%02d.sys", x+1, y+1);
					else
						sprintf(file_buff, "%s/%02d%02d.sys", ptr, x+1, y+1);

					sys_gen->Cancel(&cfg->sys);
					sys_gen->End(&cfg->sys);

					sys_gen->Save(file_buff, &cfg->sys);

					bo = sys_gen->GetMWOrbit();
					sprintf(uwp_buff, "%s", bo->GetData());
					star_buff[0][0] = star_buff[1][0] = star_buff[2][0] = 0;
					for(i = 0;i < 3;i++) {
						if((bs = sys_gen->GetStar(i)) != NULL) {
							bs->GetStarStr(star_buff[i]);
						}
					}

					hd->world = new main_world(buff, x, y, uwp_buff,
						sys_gen->GetNumBelts(), sys_gen->GetNumGG(),
						star_buff[0], star_buff[1], star_buff[2]);
					cfg->sys.loc = NULL;
					cfg->sys.sect = NULL;
					cfg->sys.ssect = NULL;
				}
				FeedBack(x, y, count, world_count);
				count++;
			}
		}
	}

	// make sure these are cleared for the dtor
	cfg->sys.loc = NULL;
	cfg->sys.sect = NULL;
	cfg->sys.ssect = NULL;

	FeedBack(-1, -1, 0, 0);
	return(TRUE);
}

/*
	base generation -- There are a few processes available to 
	handle this.  Unfortunately, most of them assume an Imperial
	sector.  I don't feel this is a reasonable assumption, so
	skip it......
*/
//void 
//Frame::CheckBases(void)
//{
//}

/*
 * generate base trade codes.  Table driven
 */
//#define DEBUG_TRADE
void 
Frame::CheckTrade(void)
{
char u[20];
int x,y,count=0;
HexData *hd;
main_world *mw;
TradeTable *tt;

	for(x = 0;x < MAX_X_HEX;x++) {
		for(y = 0;y < MAX_Y_HEX;y++) {
			if(((hd = sect->GetHex(x, y)) != NULL) &&
					((mw = hd->world) != NULL)) {
				tt = cfg->codes->GetTradeTable();
				mw->SetTradeCodes(tt->CheckTrade(u));
//				mw->SetTradeCodes(cfg->g_codes->CheckTrade(mw->GetUWPStr(u), 
//										cfg->codes->GetTradeTable()));
#ifdef DEBUG_TRADE
				fprintf(stderr, "%02d%02d: %s 0x%x\n", x, y, u,
					mw->GetTradeCodes());
#endif
				FeedBack(x, y, count, world_count);
				count++;
			}
		}
	}
	FeedBack(-1, -1, 0, 0);
}

/*
 * fill some stuff in.  Yes, it heriticly rolls a D9
 */
void 
Frame::SetPGB(void)
{
int x,y,i,count=0;
HexData *hd;
main_world *mw;

	for(x = 0;x < MAX_X_HEX;x++) {
		for(y = 0;y < MAX_Y_HEX;y++) {
			if(((hd = sect->GetHex(x, y)) != NULL) &&
					((mw = hd->world) != NULL)) {
				mw->SetPopNdx(dice.Roll(9, 1));
				if(!cfg->gen_sys) {		// already have this stuff?
					i = dice.Roll(6, 2);
					if(i < 5)
						mw->SetNumGG(0);
					else {
						i = dice.Roll(6, 2);
						if(i < 10)
							mw->SetNumGG(i / 2);
						else if(i > 10)
							mw->SetNumGG(5);
						else
							mw->SetNumGG(4);
					}
					i = dice.Roll(6, 2);
					if(i < 8)
						mw->SetNumBelts(0);
					else {
						i = dice.Roll(6, 2);
						if(i < 8)
							mw->SetNumBelts(1);
						else if(i < 12)
							mw->SetNumBelts(2);
						else
							mw->SetNumBelts(3);
					}
				}
				FeedBack(x, y, count, world_count);
				count++;
			}
		}
	}
	FeedBack(-1, -1, 0, 0);
}

/*
	Arbitrary zone setter primarily from MT Ref's Manual.
	Yes, it really is lame.
*/
void 
Frame::SetZone(void)
{
int x,y,g,l,count=0;
HexData *hd;
main_world *mw;

	for(x = 0;x < MAX_X_HEX;x++) {
		for(y = 0;y < MAX_Y_HEX;y++) {
			if(((hd = sect->GetHex(x, y)) != NULL) &&
					((mw = hd->world) != NULL)) {
				if(cfg->gen_zone) {
					g = mw->get_govt_val();
					l = mw->get_law_val();
					if((g + l) > 32)
						mw->SetZone('R');
					else if((g + l) > 30)
						mw->SetZone('A');
					else
						mw->SetZone(' ');
					FeedBack(x, y, count, world_count);
					count++;
				} else
					mw->SetZone(' ');
			}
		}
	}
	FeedBack(-1, -1, 0, 0);
}

/*
	SetStars()
	For greatest flexibility, get stars from the lib function.
	This way, if someone detests the Book 6 methods (I'm a 
	border-line case), another method can be used.  
	Optionally, this whole mess can be turned off from the command
	line.
*/
//#define DEBUG_STARS
bool 
Frame::SetStars(void)
{
char buff[MAX_FILE_LENGTH],u_buff[12];
int x,y,st_ndx,count=0,max,t_roll,s_roll;
HexData *hd;
main_world *mw;
GenStar *gs;

//fprintf(stderr, "starting GetStars()\n");
	// gen_sys => we already have this stuff, so don't bother....
	if((cfg->gen_stars) && (!cfg->gen_sys)) {
//		if(!StartSysgen(FALSE))
//			return(FALSE);
		gs = sys_gen->GetGenStar();

		// again, it might be good to verify the location.....
		count = 0;
		for(x = 0;x < MAX_X_HEX;x++) {
			for(y = 0;y < MAX_Y_HEX;y++) {
//fprintf(stderr, "Checking %02d%02d\n", x, y);
				if(((hd = sect->GetHex(x, y)) != NULL) &&
						((mw = hd->world) != NULL)) {
#ifdef DEBUG_STARS
					fprintf(stderr, "%02d%02d: ", x, y);
#endif
					max = gs->GenSystemNature();
					t_roll = s_roll = 0;
					for(st_ndx = 0;st_ndx < max;st_ndx++) {
						gs->GenerateStar(st_ndx, &t_roll, &s_roll,
							mw->GetUWPStr(u_buff), buff);
#ifdef DEBUG_STARS
						fprintf(stderr, "%s ", buff);
#endif
						mw->SetStar(st_ndx, buff);
					}
#ifdef DEBUG_STARS
					fprintf(stderr, "\n");
#endif
					FeedBack(x, y, count, world_count);
					count++;
				}
			}
		}
	}

	FeedBack(-1, -1, 0, 0);
	return(TRUE);
}

/*
Some caveats about SetRoute()
	1) obviously, it only looks in the sector we're working on
	2) this is the only place the sector is considered a grid
	3) I can't find any indication about "double checking"
	   (ie. once 0101->0202 has been checked, should 0202->0101
	   be checked?)  So, we only check directions 1, 2, and 3.
	4) it looks terribly confusing.....
	5) For these reasons, the user can turn this off.
*/
//#define DEBUG_ROUTES
void 
Frame::SetRoute(void)
{
int x,y,x1,y1,dir,dist,x2,y2,dir1,dist1,count=0;
HexData *hd,*hd1;
main_world *mw,*mw1;

	if(cfg->gen_routes) {
		for(x = 0;x < MAX_X_HEX;x++) {
			for(y = 0;y < MAX_Y_HEX;y++) {
				if(((hd = sect->GetHex(x, y)) != NULL) &&
						((mw = hd->world) != NULL)) {
#ifdef DEBUG_ROUTES
					fprintf(stderr, "%02d%02d: ", x, y);
#endif
					for(dir = 1;dir < 4;dir++) {
						for(dist = 1;dist < 5;dist++) {
							if(sect->TranslateCoord(x, y, dir, dist, &x1, &y1)){
								if(((hd1 = sect->GetHex(x1, y1)) != NULL) &&
									((mw1 = hd1->world) != NULL)) {
									if(cfg->jumps->CheckRoute(mw->get_port(),
											mw1->get_port(), dist)) {
										hd->AddRoute(x1, y1, 1);
#ifdef DEBUG_ROUTES
										fprintf(stderr, "%02d%02d: ", x1, y1);
#endif
									}
								}
							}
							// fill in....
							if((dist > 1) && (dir > 1)) {
								if(2 == dir) dir1 = 0;
								else dir1 = 1;
								for(dist1 = 1;dist1 < dist;dist1++) {
									if(sect->TranslateCoord(x1, y1, 
											dir1, dist1, &x2, &y2)) {
										if(((hd1 = sect->GetHex(x2, y2)) !=NULL)
											&& ((mw1 = hd1->world) != NULL)) {
											if(cfg->jumps->CheckRoute(mw->get_port(),
													mw1->get_port(), dist)) {
												hd->AddRoute(x2, y2, 1);
#ifdef DEBUG_ROUTES
											fprintf(stderr, "%02d%02d: ",x2,y2);
#endif
											}
										}
									}
								}
							}
						}
					}
#ifdef DEBUG_ROUTES
					fprintf(stderr, "\n");
#endif

					FeedBack(x, y, count, world_count);
					count++;
				}
			}
		}
	}
	FeedBack(-1, -1, 0, 0);
}

void 
Frame::Write(void)
{
	sect->WriteSec();
	sect->WriteDat(cfg->codes);

}

// ---------------------------------------------------------------
// let the user know something is going on
void
Frame::FeedBack(int x, int y, int c_count, int t_count)
{
char buff[80];

	if((x < 0) || (y < 0)) {
		msg2->SetLabel("initializing....");
		g2->SetValue(0);
	} else {
		sprintf(buff, "Working on hex %02d%02d (%d of %d)", x + 1, y + 1,
			c_count+1, t_count);
		msg2->SetLabel(buff);
		g2->SetValue(((c_count + 1) * 100) / t_count);
	}
	wxYield();
}

void
Frame::FeedBack3D(int x, int y, int z, int c_count, int t_count, int aux_count)
{
char buff[80];

	sprintf(buff, "Working on (%d %d %d) (%d of %d)", x, y, z, c_count + 1,
			t_count + 1);
	msg1->SetLabel(buff);
	g1->SetValue((loc_count * 100) / loc_tot);
	if(aux_count < 33)
		msg2->SetLabel(state_msgs[GS_INIT]);
	else if(aux_count < 66)
		msg2->SetLabel(state_msgs[GS_PRES]);
	else if(aux_count < 99)
		msg2->SetLabel(state_msgs[GS_STARS]);
	else
		msg2->SetLabel("Done");
	g2->SetValue(aux_count);
	wxYield();
}

// ---------------------------------------------------------------
bool
Frame::RollPresence(SYSTEM_DENSITY dens)
{
int num_dice,targ,roll;

	switch(dens) {
		case DENSE_DENSITY:
			num_dice = 1;
			targ = 3;
			break;
		case STANDARD_DENSITY:
			num_dice = 1;
			targ = 4;
			break;
		case SCATTERED_DENSITY:
			num_dice = 1;
			targ = 5;
			break;
		case SPARSE_DENSITY:
			num_dice = 1;
			targ = 6;
			break;
		case RIFT_DENSITY:
		default:			// ...to the least work
			num_dice = 2;
			targ = 12;
			break;
	}

	roll = dice.Roll(6, num_dice);
#ifdef DEBUG_ROLL
	fprintf(stderr, "d%d %dD t%d r%d\n", 
				dens, num_dice, targ, roll);
#endif
	if(roll >= targ)
		return(TRUE);
	else
		return(FALSE);
}

void
Frame::BuildMap(int v1, int v2, int v3, int v4)
{
int x, y;

	for(x = 0;x < MAX_X_HEX;x++) {
		for(y = 0;y < MAX_Y_HEX;y++) {
			sec_map[x][y] = -1;
		}
	}

	// first, use sec_map for density
	sec_map[0][0] = v1;
	sec_map[MAX_X_HEX-1][0] = v2;
	sec_map[0][MAX_Y_HEX-1] = v3;
	sec_map[MAX_X_HEX-1][MAX_Y_HEX-1] = v4;

	// recursively split each corner
	FillInMap(0, 0, MAX_X_HEX-1, MAX_Y_HEX-1);

	// fill in more, since MAX_X_HEX != MAX_Y_HEX
	for(x = 0;x < MAX_X_HEX;x++) {
		for(y = 0;y < MAX_Y_HEX;y++) {
			if(sec_map[x][y] < 0) {
				if(sec_map[x][y-1] < 0)
					sec_map[x][y] = sec_map[x][y+1];
				else if(sec_map[x][y+1] < 0)
					sec_map[x][y] = sec_map[x][y-1];
				else
					sec_map[x][y] = (sec_map[x][y-1] + sec_map[x][y+1]) / 2;
//fprintf(stderr, "%02d%02d: %d %d -> %d\n", x, y, sec_map[x][y-1], sec_map[x][y+1], sec_map[x][y]);
#if 0
				int i,tot=0,num=0,x1,y1;

				for(i = 0;i < 6;i++) {
					if((sect->GetAdjCoord(x, y, i, &x1, &y1)) && 
							(sec_map[x1][y1] > -1)) {
						num++;
						tot += sec_map[x1][y1];
					}
				}
				sec_map[x][y] = (int) ((float) (tot / num) + 0.5);
#endif
			}
		}
	}

}

/*
this is what I would like to see:
1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 
0 0 0 0 0 0 1 1 1 1 1 1 2 2 2 2 2 2 3 3 3 3 3 3 4 4 4 4 4 4 4 4
0 0 0 0 0 0 0 0 1 1 1 1 1 1 1 1 2 2 2 2 2 2 2 2 3 3 3 3 3 3 3 3
0 0 0 0 0 0 0 0 0 0 1 1 1 1 1 1 1 1 1 1 1 1 2 2 2 2 2 2 2 2 2 2
0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1

this is what I get:
0 1 1 1 1 1 1 1 2 2 2 2 2 2 2 2 3 3 3 3 3 3 3 3 4 4 4 4 4 4 4 4 
0 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 2 2 2 2 2 2 2 2 3 3 3 3 3 3 3 3 
0 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2
0 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1

*/
void
Frame::FillInMap(int x1, int y1, int x2, int y2)
{
int x3,y3;

	if(((x2 - x1) < 2) || ((y2 - y1) < 2))
		return;		// can't split any further

	x3 = x1 + ((x2 - x1) / 2);
	y3 = y1 + ((y2 - y1) / 2);

	sec_map[x1][y3] = (int) ((((float)sec_map[x1][y1] + (float)sec_map[x1][y2]) / 2.0) + 0.5);
	sec_map[x3][y1] = (int) ((((float)sec_map[x1][y1] + (float)sec_map[x2][y1]) / 2.0) + 0.5);
	sec_map[x2][y3] = (int) ((((float)sec_map[x2][y1] + (float)sec_map[x2][y2]) / 2.0) + 0.5);
	sec_map[x3][y2] = (int) ((((float)sec_map[x1][y2] + (float)sec_map[x2][y2]) / 2.0) + 0.5);
	sec_map[x3][y3] = (int) ((((float)sec_map[x1][y1] + (float)sec_map[x1][y2] + sec_map[x2][y1] + sec_map[x2][y2]) / 4.0) + 0.5);

	// yes, this does end up doing some edges twice....
	FillInMap(x1, y1, x3, y3);
	FillInMap(x1, y3, x3, y2);
	FillInMap(x3, y1, x2, y3);
	FillInMap(x3, y3, x2, y2);
}

#define FILE_DELEMITER		'/'
#define IMPORT_EXTENSION	".import"

bool
Frame::ImportFile(void)
{
bool err=FALSE;
char *ptr1,*ptr2,*file;
char buff[120],c;
int i,line=0,line1=0;
FILE *fp;

	ptr1 = sect->GetSecName();
	ptr2 = strrchr(ptr1, FILE_DELEMITER);
	if(NULL == ptr2) ptr2 = ptr1;
	else ptr2++;
	// kludge together a filename
	file = new char[strlen(ptr2) + strlen(IMPORT_EXTENSION) + 1];
	ptr1 = strchr(ptr2, '.');
	if(NULL == ptr1) ptr1 = ptr2 + strlen(ptr2);
	i = 0;
	while(ptr2 < ptr1) {
		file[i] = *ptr2;
		ptr2++;
		i++;
	}
	file[i] = 0;
	strcat(file, IMPORT_EXTENSION);
	if((fp = fopen(file, "r")) == NULL) {
		sprintf(buff, "Cannot open %s!  Quitting.", file);
		wxMessageBox(buff, "Secgen Alert !", wxCENTRE | wxICON_EXCLAMATION);
		delete file;
		return(FALSE);
	}

	world_count = 0;
	while(fgets(buff, 120, fp) != NULL) {
		line++;
		if('#' == buff[0]) continue;
		if(line1 >= MAX_Y_HEX) continue;
		line1++;
		i = 0;
		// allowed characters:
		// 1) IM_PRES: ' ' or '0' => no presence
		// 				anything else is assumed as presence
		// 2) IM_PORT: ' ' => no pressence
		// 				ABCDEX => port
		while((buff[i] >= ' ') && (i < MAX_X_HEX)) {
			c = buff[i];
			if((c != ' ') && (c != '0')) world_count++;
			if(IM_PRES == cfg->import) {
				if((' ' == c) || ('0' == c))
					c = 0;
				else
					c = 1;
			} else {
				if(strchr("ABCDEX ", c) == NULL)
					err = TRUE;
			}
			if(err) {
				sprintf(buff, "Error in %s on line %d character %d!  Quitting!",
						file, line, i);
				wxMessageBox(buff, "Secgen Alert !", 
						wxCENTRE | wxICON_EXCLAMATION);
				delete file;
				fclose(fp);
				return(FALSE);
			}
			sec_map[i][line1] = c;
			i++;
		}
	}

	fclose(fp);
	delete file;
	return(TRUE);
}

// ---------------------------------------------------------------

