/*
 * File:      enc_tbl.cpp
 * Purpose:	  generate Traveller animal encounter tables
 * Author:	
 * Created:	
 * Updated:	
 * Copyright: LGPL.
 *            Traveller is a registered trademark of Far Future Enterprises.	
 */

/* rcsid[] = "$RCSfile: enc_tbl.cpp,v $ $Revision: 1.10 $ $Author: man $ $Date: 2000/10/11 02:44:17 $" */

// For compilers that support precompilation, includes "wx.h".
#include "wx/wxprec.h"

#ifndef WX_PRECOMP
#include "wx/wx.h"
#endif

#include <stdlib.h>
#include "enc_tbl.h"
#include "d_sect.h"
#include "t_res.h"
#include "t_help.h"

IMPLEMENT_APP(EncTableApp)

#define BIG_BUFFER_SIZE			120

// button defines
enum {
	ID_TOOLBAR = 500,
	TB_CONFIG,
	TB_GEN,
	TB_SAVE,
	TB_PRINT,
	TB_ABOUT,
	TB_EXIT,
};

#ifdef __WXGTK__
#include "bitmaps/config.xpm"
#include "bitmaps/help.xpm"
#include "bitmaps/restart.xpm"
#include "bitmaps/print.xpm"
#include "bitmaps/exit.xpm"
#include "bitmaps/save.xpm"
#include "bitmaps/critter.xpm"
#endif

// handy references
EncTableApp *app=NULL;
EncFrame *frame = NULL;
wxFont *NormalFont = NULL;

// ===============================================================
bool
EncTableApp::OnInit(void)
{
struct conf_struct  cfg;
char config_file[MAX_FILE_LENGTH];
char str[MAX_FILE_LENGTH];
//char *cfg_ptr,*temp_ptr;
wxString ptr;

	app = this;
    // load general resource
	config = new wxConfig((const wxString) RESOURCE_FILE);
	wxConfigBase::Set(config);
	wxConfigBase::Get();

	sprintf(str, "/%s/%s", SECTION_DATA, ENTRY_DATA_DIR);
	if(config->Read((const wxString) str, &ptr)) {
		strcpy(config_file, ptr.GetData());
		strcat(config_file, "/");
		sprintf(str, "/%s/%s", SECTION_DATA, ENTRY_DATA_FILE);
		if(config->Read((const wxString) str, &ptr))
			strcat(config_file, ptr.GetData());
		else
			config_file[0] = 0;
	}

	// other init.
	// order/presedence:
	//	1) command line
	//	2) resources
	//	3) program
    // initialize default cfg
    cfg.name = NULL;
	cfg.sect = NULL;
	cfg.ssect = NULL;
	cfg.loc = NULL;
	memcpy(cfg.uwp, "A0000000", sizeof(cfg.uwp));
	cfg.method = EG_FULL;
	cfg.ter = TER_Beach;
	cfg.trav_version = TV_T4;			// default to T4
	cfg.out_file = NULL;
	cfg.die_size = 6;
	cfg.num_die = 1;
	cfg.table_1d6 = new char[strlen(DEFAULT_1D6) + 1];
	sprintf(cfg.table_1d6, DEFAULT_1D6);
	cfg.table_2d6 = new char[strlen(DEFAULT_2D6) + 1];
	sprintf(cfg.table_2d6, DEFAULT_2D6);
	cfg.table_1d20 = new char[strlen(DEFAULT_D20) + 1];
	sprintf(cfg.table_1d20, DEFAULT_D20);
	cfg.quiet_mode = FALSE;
	cfg.no_sleep = FALSE;

	// load resources
	sprintf(str, "/%s/%s", SECTION_ENCOUNTER, ENTRY_METHOD);
	if(config->Read((const wxString) str, &ptr))
		cfg.method = (ENC_GEN) atoi(ptr.GetData());
	sprintf(str, "/%s/%s", SECTION_ENCOUNTER, ENTRY_TRAV_VER);
	if(config->Read((const wxString) str, &ptr)) {
		if(strcmp(ptr.GetData(), "CT") == 0) {
			cfg.trav_version = TV_CT;
			cfg.die_size = 6;
			cfg.num_die = 2;
		} else if(strcmp(ptr.GetData(), "MT") == 0) {
			cfg.trav_version = TV_MT;
			cfg.die_size = 6;
			cfg.num_die = 2;
		} else if(strcmp(ptr.GetData(), "TNE") == 0) {
			cfg.trav_version = TV_TNE;
			cfg.die_size = 20;
			cfg.num_die = 1;
		} else if(strcmp(ptr.GetData(), "T4") == 0) {
			cfg.trav_version = TV_T4;
			cfg.die_size = 6;
			cfg.num_die = 2;
		}
	}

	sprintf(str, "/%s/%s", SECTION_ENCOUNTER, ENTRY_DICE);
	if(config->Read((const wxString) str, &ptr)) {
		if(strcmp(ptr.GetData(), "2D6") == 0) {
			cfg.die_size = 6;
			cfg.num_die = 2;
		} else if(strcmp(ptr.GetData(), "1D6") == 0) {
			cfg.die_size = 6;
			cfg.num_die = 1;
		} else if(strcmp(ptr.GetData(), "1D20") == 0) {
			cfg.die_size = 20;
			cfg.num_die = 1;
		}
	}

	sprintf(str, "/%s/%s", SECTION_ENCOUNTER, ENTRY_TABLE_1D6);
	if(config->Read((const wxString) str, &ptr)) {
		if((strspn(ptr.GetData(), "SHOC") == strlen(ptr.GetData())) &&
				(6 == strlen(ptr.GetData()))) {
			delete cfg.table_1d6;

			cfg.table_1d6 = new char[strlen(ptr.GetData()) + 1];
			strcpy(cfg.table_1d6, ptr.GetData());
		}
	}

	sprintf(str, "/%s/%s", SECTION_ENCOUNTER, ENTRY_TABLE_2D6);
	if(config->Read((const wxString) str, &ptr)) {
		if((strspn(ptr.GetData(), "SHOC") == strlen(ptr.GetData())) &&
				(11 == strlen(ptr.GetData()))) {
			delete cfg.table_2d6;

			cfg.table_2d6 = new char[strlen(ptr.GetData()) + 1];
			strcpy(cfg.table_2d6, ptr.GetData());
		}
	}

	sprintf(str, "/%s/%s", SECTION_ENCOUNTER, ENTRY_TABLE_D20);
	if(config->Read((const wxString) str, &ptr)) {
		if((strspn(ptr.GetData(), "SHOC") == strlen(ptr.GetData())) &&
				(20 == strlen(ptr.GetData()))) {
			delete cfg.table_1d20;

			cfg.table_1d20 = new char[strlen(ptr.GetData()) + 1];
			strcpy(cfg.table_1d20, ptr.GetData());
		}
	}

	// read command line
	ParseCommandLine(&cfg, config_file);
	
	// create fonts
    NormalFont = new wxFont(11, wxMODERN, wxNORMAL, wxNORMAL);

    frame = new EncFrame(&cfg, config_file);
	frame->CreateToolBar(wxNO_BORDER | wxHORIZONTAL | wxTB_DOCKABLE, 
			ID_TOOLBAR);
	frame->InitButtons(frame->GetToolBar());
#ifdef __WINDOWS__
	frame->SetIcon(wxIcon("critter_icn"));
#endif
#ifdef __WXGTK__
	frame->SetIcon(wxIcon(critter_xpm));
#endif
	
	frame->Show(TRUE);

	return(TRUE);
}

//EncTableApp::EncTableApp(void) :
//  wxApp()
//{
//}

void
EncTableApp::CleanUp(void)
{
	if(config != NULL) {
		delete wxConfigBase::Set((wxConfigBase *) NULL);
		config = NULL;
	}
}

bool
EncTableApp::ParseString(char **dest, char *src)
{
    if(*dest != NULL)
	    return(FALSE);
	if(src == NULL)
	    return(TRUE);
	*dest = new char[strlen(src) + 1];
	strcpy(*dest, src);
	return(TRUE);
}

bool
EncTableApp::CopyString(char *dest, char *src, unsigned int sz)
{
    if(strlen(src) > sz)
	    return(FALSE);
	strcpy(dest, src);
	return(TRUE);
}

void
EncTableApp::ParseCommandLine(struct conf_struct *cfg, char *cfg_ptr)
{
int i=1,star_count=0;
char buff[120], last_arg=0;
char *sect_file=NULL;

    buff[0] = 0;
    while(i < argc) {
		if('-' == argv[i][0]) {
			if(strchr("nSsluFoVItadNq", argv[i][1]) == NULL) {
				fprintf(stderr, "BadOption: %c\n", argv[i][1]);
				fprintf(stderr, "Valid Options:\n"
						" -n <system name>\n"
						" -S <Sector Name>\n"
						" -s <Subsector name>\n"
						" -l <location>\n"
						" -u <world UWP>\n"
						" -F <sector file>\n"
						" -o <output file>\n"
						" -V <Traveller version>\n"
						" These three mutually exlusive:\n"
						" -I <terrain type> (individual roll)\n"
						" -t <terrain type> (individual table)\n"
						" -a (all tables - default)\n"
						" -d <dice format>\n"
						" -T <table format>\n"
						" -q\n");
				exit(-1);
			}

			if(last_arg != 0)
				ProcessArgs(cfg, last_arg, buff, star_count, &sect_file);

			last_arg = argv[i][1];
			buff[0] = 0;
			if(argv[i][2] != 0)
			    strcat(buff, &argv[i][2]);
		}
		else
		    strcat(buff, argv[i]);
		i++;
	}

	if(last_arg != 0)
	    ProcessArgs(cfg, last_arg, buff, star_count, &sect_file);
	
	if(sect_file != NULL) {
		TCodes *codes;
		DetailSector *sect=NULL;

		if(cfg->loc == NULL) {
			fprintf(stderr, "No location.\n");
			exit(-4);
		}
		
		codes = new TCodes(cfg_ptr);
		sect = new DetailSector(codes);

		if(sect->LoadFile(sect_file) > 0) {
			int i,x,y;
			HexData *hex;

			i = atol(cfg->loc);
			x = i / 100;
			y = i % 100;
			hex = sect->GetHex(x - 1,y - 1);
			if(hex == NULL) {
				fprintf(stderr, "Problem with location %s.\n", cfg->loc);
				exit(-5);
			}
			ParseString(&cfg->name, hex->world->GetName(buff));
			ParseString(&cfg->sect, sect->GetName());
			i = (((x - 1) / 8)) + (((y - 1) / 10) * 4);
			ParseString(&cfg->ssect, sect->GetSSName(i));
			cfg->uwp[0] = hex->world->get_port();
			cfg->uwp[1] = hex->world->get_size();
			cfg->uwp[2] = hex->world->get_atmos();
			cfg->uwp[3] = hex->world->get_hydro();
			cfg->uwp[4] = hex->world->get_pop();
			cfg->uwp[5] = hex->world->get_govt();
			cfg->uwp[6] = hex->world->get_law();
			cfg->uwp[7] = hex->world->get_tech();
		} else {
			fprintf(stderr, "Problem with sector file %s\n", sect_file);
			exit(-3);
		}

		delete sect;
		delete codes;
		delete sect_file;
	}
}

bool
EncTableApp::ParseTerrain(TERRAIN_TYPE *t, char *src)
{
EncTerrain ter;
int i;

	for(i = 0;i < MAX_TERRAIN;i++) {
		if(strcmp(ter.GetTerrain((TERRAIN_TYPE) i), src) == 0) {
			*t = (TERRAIN_TYPE) i;
			return(TRUE);
		}
	}

	return(FALSE);
}

void
EncTableApp::ProcessArgs(struct conf_struct *cfg, char last_arg, char *buff,
					   int star_count, char **sect_file)
{
bool no_err=TRUE;			
char *ptr=NULL;

    switch(last_arg) {
		case 'n':
		    no_err = ParseString(&cfg->name, buff);
			break;
		case 'S':
		    no_err = ParseString(&cfg->sect, buff);
			break;
		case 's':
		    no_err = ParseString(&cfg->ssect, buff);
			break;
		case 'l':
		    no_err = ParseString(&cfg->loc, buff);
			break;
		case 'u': {
			int i,j;

			for(i = 0, j = 0; i < (int)strlen(buff);i++) {
				if(j > 8) {
					no_err = FALSE;
				} else if(isalnum(buff[i])) {
					cfg->uwp[j] = buff[i];
					j++;
				}
			}
			break;
		}
		case 'F':
		    no_err = ParseString(sect_file, buff);
			break;
		case 'o':
		    no_err = ParseString(&cfg->out_file, buff);
			break;
		case 'd':
			no_err = ParseString(&ptr, buff);
			if(no_err) {
				if(strcmp(ptr, "1D6") == 0) {
					cfg->die_size = 6;
					cfg->num_die = 1;
				} else if(strcmp(ptr, "2D6") == 0) {
					cfg->die_size = 6;
					cfg->num_die = 2;
				} else if(strcmp(ptr, "1D20") == 0) {
					cfg->die_size = 20;
					cfg->num_die = 1;
				}
				else
					no_err = FALSE;
			}
			if(no_err) {
				delete ptr;
				ptr = NULL;
			}
			break;
		case 'V':
			no_err = ParseString(&ptr, buff);
			if(no_err) {
				if(strcmp(ptr, "CT") == 0)
					cfg->trav_version = TV_CT;
				else if(strcmp(ptr, "MT") == 0)
					cfg->trav_version = TV_MT;
				else if(strcmp(ptr, "TNE") == 0)
					cfg->trav_version = TV_TNE;
				else if(strcmp(ptr, "T4") == 0)
					cfg->trav_version = TV_T4;
				else
					no_err = FALSE;
			}
			if(no_err) {
				delete ptr;
				ptr = NULL;
			}
			break;
		case 'I':
			no_err = ParseTerrain(&cfg->ter, buff);
			cfg->method = EG_SINGLE_ROLL;
			break;
		case 't':
			no_err = ParseTerrain(&cfg->ter, buff);
			cfg->method = EG_SINGLE_TABLE;
			break;
		case 'a':
			cfg->method = EG_FULL;
			break;
		case 'T':
			int l;

			if(6 == cfg->die_size) {
				if(1 == cfg->num_die) {
					l = 6;
					no_err = ParseString(&cfg->table_1d6, buff);
					ptr = cfg->table_1d6;
				} else {
					l = 11;
					no_err = ParseString(&cfg->table_2d6, buff);
					ptr = cfg->table_2d6;
				}
			} else {
				l = 20;
				no_err = ParseString(&cfg->table_1d20, buff);
				ptr = cfg->table_1d20;
			}

			if(no_err) {
				if((strspn(ptr, "SHOC") != strlen(ptr)) ||
					(l != (int)strlen(ptr)))
					no_err = FALSE;
			}
			break;
		case 'N':
		    cfg->no_sleep = TRUE;
		    break;
	}
			
	if(!no_err) {
		fprintf(stderr, "Problem with parameter: %c <%s>\n", last_arg, buff);
		exit(-2);
	}
}

// ===============================================================
EncTableFrame::EncTableFrame(struct conf_struct *c, char *cfg_ptr,
				int w, int h) :
  wxFrame(NULL, -1, "Generate Animal Encounters", 
  		wxPoint(0, 0), wxSize(w, h))
{
    // copy the cfg struct 
    cfg.name = c->name;
	cfg.sect = c->sect;
	cfg.ssect = c->ssect;
	cfg.loc = c->loc;
	memcpy(cfg.uwp, "\0\0\0\0\0\0\0\0", sizeof(cfg.uwp));
	strcpy(cfg.uwp, c->uwp);
	cfg.method = c->method;
	cfg.ter = c->ter;
	cfg.trav_version = c->trav_version;
	cfg.out_file = c->out_file;
	cfg.die_size = c->die_size;
	cfg.num_die = c->num_die;
	cfg.table_1d6 = c->table_1d6;
	cfg.table_2d6 = c->table_2d6;
	cfg.table_1d20 = c->table_1d20;
	cfg.quiet_mode = c->quiet_mode;
	cfg.no_sleep = c->no_sleep;

	tables = NULL;
	table = NULL;
	enc = NULL;
}

EncTableFrame::~EncTableFrame()
{
char str[MAX_FILE_LENGTH];
char str1[20];

	wxConfigBase::Set(app->config);
	wxConfigBase::Get();

	sprintf(str, "/%s/%s", SECTION_ENCOUNTER, ENTRY_TRAV_VER);
	switch(cfg.trav_version) {
		case TV_CT:
			app->config->Write((const wxString) str, (const wxString) "CT");
			break;
		case TV_MT:
			app->config->Write((const wxString) str, (const wxString) "MT");
			break;
		case TV_TNE:
			app->config->Write((const wxString) str, (const wxString) "TNE");
			break;
		case TV_T4:
			app->config->Write((const wxString) str, (const wxString) "T4");
			break;
	}
	sprintf(str, "/%s/%s", SECTION_ENCOUNTER, ENTRY_METHOD);
	app->config->Write((const wxString) str, (long int) cfg.method);

	sprintf(str, "/%s/%s", SECTION_ENCOUNTER, ENTRY_DICE);
	sprintf(str1, "%dD%d", cfg.num_die, cfg.die_size);
	app->config->Write((const wxString) str, (const wxString) str1);

	sprintf(str, "/%s/%s", SECTION_ENCOUNTER, ENTRY_TABLE_1D6);
	app->config->Write((const wxString) str, (const wxString) cfg.table_1d6);
	
	sprintf(str, "/%s/%s", SECTION_ENCOUNTER, ENTRY_TABLE_2D6);
	app->config->Write((const wxString) str, (const wxString) cfg.table_2d6);
	
	sprintf(str, "/%s/%s", SECTION_ENCOUNTER, ENTRY_TABLE_D20);
	app->config->Write((const wxString) str, (const wxString) cfg.table_1d20);

	CleanUp();
	app->CleanUp();
}

void
EncTableFrame::CleanUp(void)
{
	if(cfg.name != NULL)
	    delete cfg.name;
	if(cfg.sect != NULL)
	    delete cfg.sect;
	if(cfg.ssect != NULL)
	    delete cfg.ssect;
	if(cfg.loc != NULL)
	    delete cfg.loc;
	if(cfg.out_file != NULL)
	    delete cfg.out_file;
	if(cfg.table_1d6 != NULL)
	    delete cfg.table_1d6;
	if(cfg.table_2d6 != NULL)
	    delete cfg.table_2d6;
	if(cfg.table_1d20 != NULL)
	    delete cfg.table_1d20;
	if(tables != NULL)
		delete tables;
	if(table != NULL)
		delete table;
	if(enc != NULL)
		delete enc;
	cfg.name = cfg.sect = cfg.ssect = cfg.loc = cfg.out_file = 
			cfg.table_1d6 = cfg.table_2d6 = cfg.table_1d20 = NULL;
	tables = NULL;
	table = NULL;
	enc = NULL;
}

int
EncTableFrame::CalcTableCount(void)
{
	// the '3' should represent a larger font for the title (when it works)
	return(3 + (cfg.die_size * cfg.num_die) - (cfg.num_die - 1) + 1);
}

// ===============================================================
BEGIN_EVENT_TABLE(EncFrame, wxFrame)
//	EVT_SIZE(EncFrame::EncFrame::OnSize)
	EVT_MENU(TB_CONFIG, EncFrame::Configure)
	EVT_MENU(TB_GEN, EncFrame::Generate)
	EVT_MENU(TB_SAVE, EncFrame::Save)
	EVT_MENU(TB_PRINT, EncFrame::Print)
	EVT_MENU(TB_ABOUT, EncFrame::OnAbout)
	EVT_MENU(TB_EXIT, EncFrame::Exit)
	EVT_TOOL_ENTER(ID_TOOLBAR, EncFrame::OnToolEnter)
END_EVENT_TABLE()
// ---------------------------------------------------------------
#define BUILD_FRAME_WIDTH         500
#define BUILD_FRAME_HEIGHT        400
#define BUILD_PANEL_HEIGHT        80


// ---------------------------------------------------------------
EncFrame::EncFrame(struct conf_struct *c, char *cfg_ptr) :
	EncTableFrame(c, cfg_ptr, BUILD_FRAME_WIDTH, BUILD_FRAME_HEIGHT)
{
	// null out the dialogs (create on the fly)
	dlg = NULL;

	// now do the window stuff
	CreateStatusBar(2);

	canvas = new EncTableCanvas(this);
    canvas->SetScrollbars(4, 4, 300, 1700, 0, 0);
	canvas->SetBackgroundColour(*wxWHITE);
}

EncFrame::~EncFrame()
{
	CleanUp();
}

void
EncFrame::Configure(wxCommandEvent& event)
{
	if(NULL == dlg)
		dlg = new EncDialog(this);

	if(dlg->GetConfig(&cfg))
		canvas->Refresh();
}

void
EncFrame::Generate(wxCommandEvent& event)
{
char *ptr;

	ClearEnc();
	if((1 == cfg.die_size) && (20 == cfg.num_die))
		ptr = cfg.table_1d20;
	else if((1 == cfg.die_size) && (6 == cfg.num_die))
		ptr = cfg.table_2d6;
	else 
		ptr = cfg.table_1d6;

	switch(cfg.method) {
		case EG_SINGLE_ROLL:
			int i;
			Dice *dice;

			dice = new Dice(6, 2, 0);
			i = dice->Roll(cfg.die_size, cfg.num_die);
			delete dice;
			
			enc = new AnimalEncounter(cfg.ter, ptr[i],
					cfg.uwp);
			break;
		case EG_SINGLE_TABLE:
			table = new EncounterTable(cfg.ter, cfg.uwp,
							cfg.die_size, cfg.num_die, 
							ptr);
			break;
		case EG_FULL:
			tables = new EncounterTables(cfg.uwp,
							cfg.die_size, cfg.num_die, 
							ptr);
			break;
	}
	canvas->Refresh();
}

void
EncFrame::Print(wxCommandEvent& event)
{
	wxPrinter *printer;
	PrintOut *print_out;
	print_out = new PrintOut(&cfg);

	printer = new wxPrinter();
	printer->Print(this, print_out, TRUE);

	delete printer;
	delete print_out;
}

void
EncFrame::Save(wxCommandEvent& event)
{
char *s;

	if((enc != NULL) || (table != NULL) || (tables != NULL)) {
		if(cfg.out_file == NULL) {
			wxString ptr;

			ptr = wxFileSelector("Save Encounter", 
				NULL, 
				NULL,
				NULL,
				"*.txt",
				wxSAVE | wxHIDE_READONLY,
				this);
			s = (char *) ptr.GetData();
		} else
			s = cfg.out_file;

		if((s != NULL) && (s[0] != 0)) {
			FILE *fpx;

			if((fpx = fopen(s, "a+")) != NULL) {
				EncDisplay *ed;

				ed = new EncDisplay(fpx);
				if(enc != NULL)
					ed->DrawEnc(enc, &cfg);
				else if(table != NULL)
					ed->DrawTable(table, &cfg);
				else if(tables != NULL)
					ed->DrawTables(tables, &cfg);

				delete ed;
			}
		}
	}
}

void
EncFrame::Exit(wxCommandEvent& event)
{
	Close(TRUE);
}

void
EncFrame::OnAbout(wxCommandEvent& event)
{
AboutDialog *dlg;

	dlg = new AboutDialog(this);
	dlg->ShowAbout("Encounter Table Generator");
	delete dlg;
}

// ---------------------------------------------------------------
#define MAX_BUTTONS		6

static char *long_msgs[] = {
    "Configure Program",
	"Generate New Encounter(s)",
	"Save Current",
	"Print Current",
	"About The Encounter Table Program",
	"Exit Program" 
	};

static char *short_msgs[] = {
    "Config",
	"Generate",
	"Save",
	"Print",
	"About",
	"Exit",
	};

void 
EncFrame::InitButtons(wxToolBar *tb)
{
int i;
int currentX = DLG_OFFSET;
#ifdef __WXMSW__
  int width = 24;
#else
  int width = 16;
#endif
wxBitmap* bm[MAX_BUTTONS];

	// Set up toolbar
	tb->SetMargins(2, 2);

#ifdef __WXMSW__
	bm[0] = new wxBitmap("icon1");
	bm[1] = new wxBitmap("icon2");
	bm[2] = new wxBitmap("icon3");
	bm[3] = new wxBitmap("icon4");
	bm[4] = new wxBitmap("icon5");
	bm[5] = new wxBitmap("icon6");
#endif
#ifdef __WXGTK__
	bm[0] = new wxBitmap(config_xpm);
	bm[1] = new wxBitmap(restart_xpm);
	bm[2] = new wxBitmap(save_xpm);
	bm[3] = new wxBitmap(print_xpm);
	bm[4] = new wxBitmap(help_xpm);
	bm[5] = new wxBitmap(exit_xpm);
#endif

	for(i = 0;i < MAX_BUTTONS;i++) {
		tb->AddTool(TB_CONFIG + i, *(bm[i]), wxNullBitmap, FALSE,
			currentX, -1, NULL, 
			short_msgs[i], long_msgs[i]);
		currentX += width + DLG_OFFSET;
		if((0 == i) || (1 == i))
			tb->AddSeparator();
	}
		
	tb->Realize();

	// Can delete the bitmaps since they're reference counted
	for (i = 0; i < MAX_BUTTONS; i++)
		delete bm[i];
}

// ---------------------------------------------------------------
void
EncFrame::Draw(wxDC *dc)
{
EncDisplay *ed;

	ed = new EncDisplay(dc);
	if(enc != NULL)
		ed->DrawEnc(enc, &cfg);
	else if(table != NULL)
		ed->DrawTable(table, &cfg);
	else if(tables != NULL)
		ed->DrawTables(tables, &cfg);

	delete ed;
}

void
EncFrame::ClearEnc(void)
{
	if(tables != NULL) {
		delete tables;
		tables = NULL;
	}
	if(table != NULL) {
		delete table;
		table = NULL;
	}
	if(enc != NULL) {
		delete enc;
		enc = NULL;
	}
}

// ---------------------------------------------------------------
void
EncFrame::OnToolEnter(wxCommandEvent& event)
{
int i=TB_CONFIG;
bool fail=TRUE;

	while(i <= TB_EXIT)
		{
		if(event.GetSelection() == i)
			{
			wxString str;
			str.Printf("%s", long_msgs[i - TB_CONFIG]);
			SetStatusText(str, 0);
			fail = FALSE;
			break;
			}
		i++;
		}
	if(fail)
		SetStatusText("", 0);
}

// ---------------------------------------------------------------
void
EncFrame::UpdateStatus(void)
{
//char buff[BIG_BUFFER_SIZE];

	SetStatusText("");
}

// -------------------------------------------------------------------------
// =================================================================
// ===============================================================
// ===============================================================
EncTableCanvas::EncTableCanvas(wxWindow *frame,
						   int x, int y, int w, int h,
						   long style) :
    wxScrolledWindow(frame, -1, wxPoint(x, y), wxSize(w, h), style)
{
}

void
EncTableCanvas::OnDraw(wxDC& dc)
{
	if(frame != NULL) {
		dc.Clear();
		frame->Draw(&dc);
	}
	else
	    dc.Clear();
}

// =================================================================
EncPanel::EncPanel(wxWindow *frame, int x, int y, int w, int h) :
	wxPanel(frame, -1, wxPoint(x, y), wxSize(w, h))
{
}

EncPanel::~EncPanel()
{
}

// =================================================================
EncDisplay::~EncDisplay()
{
}

int 
EncDisplay::DrawEnc(AnimalEncounter *ae, conf_struct *cfg,
			char *msg1, char *msg2,
			int start_count, int total_count)
{
int ret=0;//,count=0;

	Start(msg1);

// pretty sure we don't need a line check here
	ret += DrawHeader(cfg, cfg->ter, FALSE);
	ret += DrawLine(ae, cfg->trav_version);

	Finish(msg2);
	return(ret);
}

int 
EncDisplay::DrawTable(EncounterTable *et, conf_struct *cfg,
			char *msg1, char *msg2,
			int start_count, int total_count)
{
//int i;
//int ret=0,count=0;

	Start(msg1);

// as w/ the single line, I'm pretty sure the check isn't neccasary
	DrawSingleTable(et, cfg, cfg->ter);

	Finish(msg2);
	return(frame->CalcTableCount());
}

int 
EncDisplay::DrawTables(EncounterTables *et, conf_struct *cfg,
			char *msg1, char *msg2,
			int start_count, int total_count)
{
int i, table_count;
int ret=0,count=0;

	table_count = frame->CalcTableCount();

	Start(msg1);

	for(i = 0;i < MAX_TERRAIN;i++) {
		if((count >= start_count) && (ret < total_count)) {
			DrawSingleTable(et->GetTable((TERRAIN_TYPE) i), cfg, 
						(TERRAIN_TYPE) i);
			ret += table_count;
		}
		count++;
	}

	Finish(msg2);
	return(table_count);
}

int
EncDisplay::DrawSingleTable(EncounterTable *at, conf_struct *cfg, 
		TERRAIN_TYPE t)
{
int i=0,j;
AnimalEncounter *ae;

	DrawHeader(cfg, t);

	j = cfg->num_die;
	while((ae = at->GetEnc(i)) != NULL) {
		DrawLine(ae, cfg->trav_version, j);
		i++;
		j++;
	}
	Line(" ");

	return(frame->CalcTableCount());
}


int 
EncDisplay::DrawLine(AnimalEncounter *ae, TRAV_VER v, int die_roll)
{
char buff1[2 * BIG_BUFFER_SIZE];
char buff2[2 * BIG_BUFFER_SIZE];
char buff3[20];

	ae->GetEnc(v, buff1);
	if(die_roll < 0) 
		strcpy(buff2, buff1);
	else {
		sprintf(buff3, "%d", die_roll);
		strfixsize(buff3, 4);
		if(die_roll < 10)
			strrot(buff3, 1, TRUE);
		sprintf(buff2, "%s%s", buff3, buff1);
	}
	Line(buff2);
	return(1);
}

int 
EncDisplay::DrawHeader(conf_struct *cfg, TERRAIN_TYPE t, bool table)
{
EncTerrain ter;
char buff[BIG_BUFFER_SIZE];
char buff1[BIG_BUFFER_SIZE];

	if(table)
		sprintf(buff, "Encounter Table for ");
	else
		sprintf(buff, "Encounter for ");

	if(cfg->name != NULL)
		strcat(buff, cfg->name);
	sprintf(buff1, " (%s), %s Terrain", cfg->uwp, ter.GetTerrain(t));
	strcat(buff, buff1);
	if(cfg->loc != NULL) {
		sprintf(buff1, "        %s ", cfg->loc);
		if(cfg->sect != NULL)
			strcat(buff1, cfg->sect);
		strcat(buff, buff1);
	}
	Header1(buff);

	ter.GetEncHeader(cfg->trav_version, buff, table);
	Header2(buff);

	return(2);
}

// =================================================================
#define LINES_PER_PAGE		90
PrintOut::PrintOut(conf_struct *c)
{
	cfg = c;
	if((frame->GetTable() != NULL) || (frame->GetEnc() != NULL))
		page_max = 1;
	else {
		int i;//,j;

		i = LINES_PER_PAGE / frame->CalcTableCount();
		page_max = MAX_TERRAIN / i;
		if(MAX_TERRAIN % i)  page_max++;
//fprintf(stderr, "i: %d  count: %d  max1: %d  max2:%d\n", i,
//	frame->CalcTableCount(), MAX_TERRAIN, page_max);
	}
}

bool
PrintOut::HasPage(int page)
{
	if(page <= page_max)
		return(TRUE);
	else
		return(FALSE);
}

void
PrintOut::GetPageInfo(int *minPage, int *maxPage,
						int *selPageFrom, int *selPageTo)
{
	*minPage = *selPageFrom = 1;
	*maxPage = *selPageTo = page_max;
}

bool 
PrintOut::OnPrintPage(int page)
{
wxDC *dc;
bool ret=FALSE;
char buff[80];

	dc = GetDC();
	if((dc) && (page > 0) && (page <= page_max)) {
		//int line_count,temp_count,i,cur_count;
		int line_count,cur_count;
		int w,h,a_h,a_w;
		float scale;
		EncDisplay *ed;
		EncounterTables *tables=NULL;
		EncounterTable *table=NULL;
		AnimalEncounter *enc;

		// figure out how much there is to work with
		dc->GetSize(&w,&h);
		a_h = h - 55;
		a_w = w - 35;
		
		ed = new EncDisplay(dc, 60, 20);
		sprintf(buff, "Page %d of %d", page, page_max);

		// fairly arbitrary, i would like about 100 lines......
		scale = (float)a_h / (float)(LINES_PER_PAGE * TEXT_Y_INCR);
		line_count = LINES_PER_PAGE - 4;
		dc->SetUserScale(scale, scale);
		cur_count = ((LINES_PER_PAGE / frame->CalcTableCount())) *
						(page - 1);

		// safely(?) assume that ind. rolls and table can fit
		if((enc = frame->GetEnc()) != NULL)
			ed->DrawEnc(enc, cfg, NULL, buff);
		else if((table = frame->GetTable()) != NULL)
			ed->DrawTable(table, cfg, NULL, buff);
		else if((tables = frame->GetTables()) != NULL) {
//fprintf(stderr, "page: %d cur:%d  line:%d\n", page, cur_count, line_count);
			ed->DrawTables(tables, cfg, NULL, buff, 
						cur_count, line_count);
		}

		delete ed;
		ret = TRUE;
	}

	return(ret);
}

