/*
 * File:      enc.cpp
 * Purpose:	  generate Traveller animal encounters
 * Author:	
 * Created:	
 * Updated:	
 * Copyright: LGPL.
 *            Traveller is a registered trademark of Far Future Enterprises.	
 */

/* rcsid[] = "$RCSfile: enc.cpp,v $ $Revision: 1.2 $ $Author: man $ $Date: 1999/04/06 04:25:07 $" */

#include "enc.h"
#include "string.h"
#include "str_util.h"

// inner workings of animal encounters
//  local to this module, unneeded elsewhere

// convience enums
// ---------------------------------------------------------------------------
// local structs; unneeded outside here
static struct animal_classes {
	ANIMAL_CLASS	cls;
	char			code;
	char			*desc; 
	} classes[MAX_ANIMAL_CLASS] = {
	{ AC_HERBIVORE, 'H', "Herbivore" },
	{ AC_OMNIVORE,  'O', "Omnivore" },
	{ AC_CARNIVORE, 'C', "Carnivore" },
	{ AC_SCAVENGER, 'S', "Scavenger" } };

// 3 Animal Type and Weight DM
static struct terrain_dm {
	char	*terr;
	char	*equiv;
	int		type_dm;
	int		wt_dm;
} terr_dms [MAX_TERRAIN] = {
	{ "Beach",    "Shore, Sea Edge",   3,  2 },
	{ "Bottom",   "Ocean, Sea",       -2,  0 },
	{ "Broken",   "Badlands",         -3, -3 },
	{ "Cave",     "Cavern",           -4,  1 },
	{ "Chasm",    "Crevasse, Abyss",  -1, -3 },
	{ "Clear",    "Road, Open",        3,  0 },
	{ "Crater",   "Hollow",            0, -1 },
	{ "Depths",   "Ocean, Sea",       -4,  0 },
	{ "Desert",   "Dunes",             3, -3 },
	{ "Forest",   "Woods",            -4, -4 },
	{ "Glacier",  "Ice Cap",           0, -1 },
	{ "Jungle",   "Rainforest",       -3, -2 },
	{ "Marsh",    "Wetland",           0, -1 },
	{ "Mountain", "Alpine",            0,  0 },
	{ "Prairie",  "Plain, Steppe",     4,  0 },
	{ "River",    "Stream, Creek",     1,  1 },
	{ "Rough",    "Hills, Foothills",  0,  0 },
	{ "Ruins",    "Old City",         -3,  0 },
	{ "Sargasso", "Seaweed",          -4, -2 },
	{ "Sea Cave", "Sea Cavern",       -2,  0 },
	{ "Shallows", "Ocean, Sea",        2,  2 },
	{ "Surface",  "Ocean, Sea",        2,  3 },
	{ "Swamp",    "Bog",              -2,  4 } };

// 4 Select Animal Table Format
//  	user

// 5 Animal Type and quantity
struct type_roll {
	ANIMAL_TYPE		type;
	int				num;		// number of dice
};

struct type_roll herb_roll[] = {
	{ AT_FILTER, 1 },
	{ AT_FILTER, 0 },
	{ AT_FILTER, 0 },
	{ AT_INTERMITTENT, 0 },
	{ AT_INTERMITTENT, 0 },
	{ AT_INTERMITTENT, 0 },
	{ AT_INTERMITTENT, 0 },
	{ AT_GRAZER, 0 },
	{ AT_GRAZER, 1 },
	{ AT_GRAZER, 2 },
	{ AT_GRAZER, 3 },
	{ AT_GRAZER, 2 },
	{ AT_GRAZER, 4 },
	{ AT_GRAZER, 5 } };

struct type_roll omni_roll[] = {
	{ AT_GATHERER, 0 },
	{ AT_GATHERER, 0 },
	{ AT_EATER, 0 },
	{ AT_GATHERER, 0 },
	{ AT_EATER,  2 },
	{ AT_GATHERER, 0 },
	{ AT_HUNTER, 0 },
	{ AT_HUNTER, 1 },
	{ AT_HUNTER, 0 },
	{ AT_GATHERER, 0 },
	{ AT_EATER, 1 },
	{ AT_HUNTER, 1 },
	{ AT_GATHERER, 0 },
	{ AT_GATHERER, 0 } };

struct type_roll carn_roll[] = {
	{ AT_SIREN, 0 },
	{ AT_POUNCER, 0 },
	{ AT_SIREN, 0 },
	{ AT_POUNCER, 0 },
	{ AT_KILLER, 1 },
	{ AT_TRAPPER, 0 },
	{ AT_POUNCER, 0 },
	{ AT_CHASER, 0 },
	{ AT_CHASER, 3 },
	{ AT_CHASER,  0 },
	{ AT_KILLER, 0 },
	{ AT_CHASER, 2 },
	{ AT_SIREN, 0 },
	{ AT_CHASER, 1 },
};

struct type_roll scav_roll[] = {
	{ AT_CARRION_EATER, 1 },
	{ AT_CARRION_EATER, 2 },
	{ AT_REDUCER, 1 },
	{ AT_HIJACKER, 1 },
	{ AT_CARRION_EATER, 2 },
	{ AT_INTIMIDATOR, 1 },
	{ AT_REDUCER, 0 },
	{ AT_CARRION_EATER, 1 },
	{ AT_REDUCER, 0 },
	{ AT_HIJACKER, 0 },
	{ AT_INTIMIDATOR, 0 },
	{ AT_REDUCER, 1 },
	{ AT_HIJACKER, 0 },
	{ AT_INTIMIDATOR, 1 },
};

// 6 Special Attributes
//		code
// 7 Animal Weight Effects
struct wt_roll {
	int		num_die;
	int		die_size;
	int		mod;
};
struct wt_roll1 {
	int		num_die;
	int		die_size;
};

struct wt_dm {
	int				wt;
	struct wt_roll	mt_hits;
	struct wt_roll	mt_wound;
	char			mt_wound_mod_op;
	struct wt_roll	mt_wound_mod;
	struct wt_roll	tne_hits;
	char			tne_dam_op;
	struct wt_roll	tne_dam;
	int				tne_init_dm;
	int				tne_arm_dm;
	struct wt_roll1	t4_hits;
	struct wt_roll1	t4_hits2;
	char			t4_wound_op;
	struct wt_roll1	t4_wound;
} wt_table[] = {
//     ------- MT ----------------- ----------- TNE ----------- ---- T4 ------
//                     Wound                                 Ar
//  Wt Hits    Wound   Modifer      Hits      Damage      In DM Hits        Wnds
{    1,{0,0,1},{0,0,0},'-',{1,6,-3},{ 0,0, 1},'/',{0,0,2}, 1,0,{ 1,6},{0,0},'-',{2, 6} },
{    3,{0,0,1},{0,0,1},'-',{1,6,-3},{ 0,0, 1},'/',{0,0,2}, 1,0,{ 1,6},{1,6},'-',{1, 6} },
{    6,{1,6,0},{0,0,1},'-',{0,0, 1},{ 1,3, 0},'/',{0,0,2}, 1,0,{ 1,6},{2,6},'-',{1, 6} },
{   12,{1,6,0},{0,0,2},0,  {0,0, 0},{ 1,6, 0},0,  {0,0,0}, 1,0,{ 2,6},{2,6},0,  {0,0} },
{   25,{1,6,0},{0,0,3},0,  {0,0, 0},{ 1,10,0},0,  {0,0,0}, 0,0,{ 3,6},{2,6},0,  {0,0} },
{   50,{1,6,0},{1,6,0},0,  {0,0, 0},{ 1,20,0},0,  {0,0,0}, 0,0,{ 4,6},{2,6},0,  {0,0} },
{  100,{1,6,0},{1,6,1},0,  {0,0, 0},{ 2,20,0},0,  {0,0,0}, 0,0,{ 5,6},{2,6},0,  {0,0} },
{  200,{2,6,0},{1,6,1},'+',{0,0, 1},{ 3,20,0},'+',{0,0,1}, 0,0,{ 5,6},{3,6},'+',{1, 6} },
{  400,{2,6,0},{1,6,1},'+',{1,6,-3},{ 4,20,0},'+',{0,0,2},-1,1,{ 6,6},{3,6},'+',{2, 6} },
{  800,{2,6,0},{1,6,3},'+',{1,6, 0},{ 5,20,0},'+',{1,6,0},-1,1,{ 7,6},{3,6},'+',{3, 6} },
{ 1600,{3,6,0},{1,6,3},'+',{2,6, 0},{ 6,20,0},'+',{2,6,0},-2,1,{ 8,6},{3,6},'+',{4, 6} },
{ 3200,{3,6,0},{1,6,4},'+',{3,6, 0},{ 7,20,0},'+',{3,6,0},-2,1,{ 8,6},{4,6},'+',{5, 6} },
{ 6000,{3,6,0},{2,6,0},'x',{1,6, 0},{ 8,20,0},'*',{1,6,0},-2,2,{ 9,6},{4,6},'*',{2, 0} },
{ 6000,{3,6,0},{2,6,0},'x',{1,6, 0},{ 8,20,0},'*',{1,6,0},-2,2,{ 9,6},{4,6},'*',{2, 0} },
{12000,{3,6,0},{2,6,2},'x',{1,6, 0},{ 9,20,0},'*',{1,6,0},-3,2,{10,6},{5,6},'*',{2, 0} },
{24000,{4,6,0},{3,6,0},'x',{1,6, 1},{10,20,0},'*',{1,6,0},-3,2,{12,6},{6,6},'*',{3, 0} },
{30000,{5,6,0},{3,6,0},'x',{2,6, 0},{11,20,0},'*',{2,6,0},-3,3,{14,6},{7,6},'*',{4, 0} },
{36000,{5,6,0},{3,6,3},'x',{2,6, 0},{12,20,0},'*',{2,6,0},-4,3,{15,6},{7,6},'*',{4, 0} },
{40000,{5,6,0},{3,6,6},'x',{2,6, 2},{13,20,0},'*',{2,6,0},-4,3,{16,6},{8,6},'*',{5, 0} },
{44000,{6,6,0},{3,6,6},'x',{3,6, 0},{14,20,0},'*',{3,6,0},-4,4,{17,6},{9,6},'*',{6, 0} } 
};

// 8 Animal Weaponry
//  code
// 9 Animal Armor
//  code
// 10 Animal Behavior
struct animal_roll {
	ROLL_TYPE	roll;
	int			min;
	};

struct animal_type {
	ANIMAL_TYPE         type;
	char				code;
	char				*desc;
	};

struct animal_sub_type {
	char				code;
	char				*sub_type;

	ROLL_TYPE			ct_attack;
	ROLL_TYPE			ct_flee;
	CT_SPEEDS			ct_speed;
	struct animal_roll	mt_attack;
	struct animal_roll	mt_flee;		// also t4
	struct animal_roll	mt_speed;
	ROLL_TYPE			tne_attack;
	ROLL_TYPE			tne_flee;
	int					tne_speed;		// always D20s
	} types[MAX_ANIMAL_TYPE] = {
 { 'H', "Filter",        RT_POSSIBLE, RT_8,         CS_ONE_ZERO,
		{ RT_POSSIBLE,  0 }, { RT_1D_P_2,   3 }, { RT_1D_M_5, 0 },
 						 RT_POSSIBLE, RT_1D,        1 },
 { 'H', "Intermittent",  RT_10,       RT_9,         CS_TWO,
		{ RT_1D_P_3,    0 }, { RT_1D_P_3,   4 }, { RT_1D_M_4, 1 },
 						 RT_2D,       RT_3D,        2 },
 { 'H', "Grazer",        RT_8,        RT_5,         CS_TWO,
		{ RT_1D_P_2,    0 }, { RT_1D_M_1,   0 }, { RT_1D_M_2, 2 },
 						 RT_2D,       RT_3D,        3 },
 { 'O', "Gatherer",      RT_9,        RT_8,         CS_ONE,
 		{ RT_1D_P_3,   4 }, { RT_1D_P_2,    0 }, { RT_1D_M_3, 1 },
 						 RT_2D,       RT_2D,        2 },
 { 'O', "Hunter",        RT_6,        RT_8,         CS_TWO,
 		{ RT_1D,       1 }, { RT_1D_P_2,    0 }, { RT_1D_M_4, 1 },
 						 RT_3D,       RT_2D,        3 },
 { 'O', "Eater",         RT_5,        RT_10,        CS_ONE_TWO,
 		{ RT_1D,       1 }, { RT_1D_P_3,    0 }, { RT_1D_M_3, 1 },
 						 RT_2D,       RT_1D,        2 },
 { 'C', "Pouncer",       RT_SURPRISE, RT_SURPRISED, CS_TWO,
 		{ RT_SURPRISE, 0 }, { RT_SURPRISED, 0 }, { RT_1D_M_4, 1 },
 						 RT_SURPRISE, RT_SURPRISED, 3 },
 { 'C', "Chaser",        RT_MORE,     RT_9,         CS_THREE_TWO,
 		{ RT_MORE,     0 }, { RT_1D_P_3,    0 }, { RT_1D_M_2, 2 },
                         RT_MORE,     RT_2D,        4 },
 { 'C', "Trapper",       RT_SURPRISE, RT_9,         CS_ONE_ZERO,
 		{ RT_SURPRISE, 0 }, { RT_1D_P_2,    0 }, { RT_1D_M_5, 0 },
                         RT_SURPRISE, RT_2D,        1 },
 { 'C', "Siren",         RT_SURPRISE, RT_10,        CS_ONE_ZERO,
 		{ RT_SURPRISE, 0 }, { RT_1D_P_3,    0 }, { RT_1D_M_4, 0 },
                         RT_SURPRISE, RT_2D,        1 },
 { 'C', "Killer",        RT_6,        RT_1D,        CS_TWO_ONE,
 		{ RT_1D,       1 }, { RT_1D_P_3,    0 }, { RT_1D_M_3, 1 },
                         RT_3D_P_2,   RT_1D,        3 },
 { 'S', "Hijacker",      RT_7,        RT_8,         CS_TWO,
 		{ RT_1D_P_1,   2 }, { RT_1D_P_2,    0 }, { RT_1D_M_4, 1 },
                         RT_3D,       RT_2D,        3 },
 { 'S', "Intimidator",   RT_8,        RT_7,         CS_TWO,
 		{ RT_1D_P_2,   3 }, { RT_1D_P_1,    0 }, { RT_1D_M_4, 1 },
                         RT_3D,       RT_2D,        2 },
 { 'S', "Carrion-eater", RT_11,       RT_8,         CS_ONE,
 		{ RT_1D_P_3,   4 }, { RT_1D_P_2,    0 }, { RT_1D_M_3, 1 },
                         RT_2D,       RT_3D,        3 },
 { 'S', "Reducer",       RT_10,       RT_8,         CS_TWO,
 		{ RT_1D_P_3,   4 }, { RT_1D_P_2,    0 }, { RT_1D_M_4, 1 },
                         RT_1D,       RT_1D,        1 } };


// =========================================================================
AnimalEncounter::AnimalEncounter(TERRAIN_TYPE t, char c, char *uwp) :
		Encounter()
{
int i;
ANIMAL_CLASS a=AC_HERBIVORE;

	for(i = 0;i < MAX_ANIMAL_CLASS;i++)
		if(classes[i].code == c)
			a = classes[i].cls;

	Init(t, a, uwp);
}

AnimalEncounter::AnimalEncounter(TERRAIN_TYPE t, ANIMAL_CLASS a, char *uwp) :
		Encounter()
{
	Init(t, a, uwp);
}

char *
AnimalEncounter::GetEnc(TRAV_VER v, char *buff)
{
//char *ptr;
char buff1[20];
char buff2[20];
char buff3[20];
int a,f,s;

	buff[0] = 0;

	sprintf(buff, "%d", num);
	strfixsize(buff, 3);
	switch(qualifier) {
		case AQ_NONE:
			break;
		case AQ_SWIMMER:
			strcat(buff, "Swimming ");
			break;
		case AQ_AMPHIBIAN:
			strcat(buff, "Amphibian ");
			break;
		case AQ_FLYER:
			strcat(buff, "Flying ");
			break;
		case AQ_TRIPHIBIAN:
			strcat(buff, "Triphibian ");
			break;
	}
	
	strcat(buff, types[type].sub_type);
	strfixsize(buff, 27);
	sprintf(buff1, "%dkg", mass);
	strfixsize(buff1, 8);
	strcat(buff, buff1);

	switch(v) {
		case TV_CT:
			GetCTEnc(buff);
			a = ct_attack;
			f = ct_flee;
			s = ct_speed;
			break;
		case TV_MT:
			GetMTEnc(buff);
			a = mt_attack;
			f = mt_flee;
			s = mt_speed;
			break;
		case TV_TNE:
			GetTNEEnc(buff);
			a = tne_attack;
			f = tne_flee;
			s = tne_speed;
			break;
		case TV_T4:
			GetT4Enc(buff);
			a = mt_attack;
			f = mt_flee;
			s = mt_speed;
			break;
	}

	if(type < AT_GATHERER)
		sprintf(buff1, "F%s A%s S%d", DecodeAttackFlee(f, buff3), 
				DecodeAttackFlee(a, buff2), s);
	else
		sprintf(buff1, "A%s F%s S%d", DecodeAttackFlee(a, buff3), 
				DecodeAttackFlee(f, buff2), s);
	strcat(buff, buff1);

	return(buff);
}

char *
AnimalEncounter::DecodeAttackFlee(int val, char *buff)
{
	if(val < 'a') sprintf(buff, "%d", val);
	else sprintf(buff, "%c", val);
	return(buff);
}

char *
AnimalEncounter::GetCTArmour(CT_ARMOUR_TYPE a)
{
char *ptr;

	switch(a) {
		case CAT_NONE:
			ptr = "--       ";
			break;
		case CAT_JACK:
			ptr = "Jack     ";
			break;
		case CAT_MESH:
			ptr = "Mesh     ";
			break;
		case CAT_CLOTH:
			ptr = "Cloth    ";
			break;
	}

	return(ptr);
}

char *
AnimalEncounter::GetMTArmour(MT_ARMOUR_TYPE a)
{
char *ptr;

	switch(a) {
		case MAT_NONE:
			ptr = "--       ";
			break;
		case MAT_JACK:
			ptr = "Jack     ";
			break;
		case MAT_MESH:
			ptr = "Mesh     ";
			break;
		case MAT_MESH1:
			ptr = "Mesh+1   ";
			break;
		case MAT_CLOTH:
			ptr = "Cloth    ";
			break;
		case MAT_CLOTH1:
			ptr = "Cloth+1  ";
			break;
		case MAT_REFLEC:
			ptr = "Reflect  ";
			break;
		case MAT_ABLAT:
			ptr = "Ablat    ";
			break;
		case MAT_BATTLE:
			ptr = "Battle   ";
			break;
		case MAT_COMBAT4:
			ptr = "Combat+4 ";
			break;
	}
		
	return(ptr);
}

char *
AnimalEncounter::GetTNEArmour(TNE_ARMOUR_TYPE a)
{
char *ptr;

	switch(a) {
		case TAT_NONE:
			ptr = "--          ";
			break;
		case TAT_HALF:
			ptr = "1/2         ";
			break;
		case TAT_HALF_HEAD1:
			ptr = "1/2(Head 1) ";
			break;
		case TAT_ONE_HEAD2:
			ptr = "1(Head 1)   ";
			break;
		case TAT_TWO_HEAD3:
			ptr = "2(Head 3)   ";
			break;
		case TAT_THREE:
			ptr = "3           ";
			break;
	}

	return(ptr);
}

char *
AnimalEncounter::GetT4Armour(T4_ARMOUR_TYPE a)
{
char *ptr;

	switch(a) {
		case T4AT_NONE:
			ptr = "None    ";
			break;
		case T4AT_ONE:
			ptr = "3       ";
			break;
		case T4AT_ONE_RIGID:
			ptr = "3 Rigid ";
			break;
		case T4AT_TWO:
			ptr = "3       ";
			break;
		case T4AT_TWO_RIGID:
			ptr = "3 Rigid ";
			break;
		case T4AT_THREE:
			ptr = "3       ";
			break;
		case T4AT_THREE_RIGID:
			ptr = "3 Rigid ";
			break;
		case T4AT_FOUR:
			ptr = "4        ";
			break;
		case T4AT_FOUR_RIGID:
			ptr = "4 Rigid ";
			break;
	}
		
	return(ptr);
}

char *
AnimalEncounter::GetMTWeapon(MT_WEAPON_TYPE w)
{
char *ptr;

	switch(w) {
		case MWT_HORNS_HOOVES:
			ptr = "Horns & Hooves    ";
			break;
		case MWT_HORNS:
			ptr = "Horns             ";
			break;
		case MWT_HOOVES_TEETH:
			ptr = "Hooves & Teeth    ";
			break;
		case MWT_HOOVES:
			ptr = "Hooves            ";
			break;
		case MWT_HORNS_TEETH:
			ptr = "Horns & Teeth     ";
			break;
		case MWT_THRASHER:
			ptr = "Thrasher          ";
			break;
		case MWT_CLAWS_TEETH:
			ptr = "Claws & Teeth     ";
			break;
		case MWT_TEETH:
			ptr = "Teeth             ";
			break;
		case MWT_CLAWS:
			ptr = "Claws             ";
			break;
		case MWT_CLAWS1:
			ptr = "Claws+1           ";
			break;
		case MWT_STINGER:
			ptr = "Stinger           ";
			break;
		case MWT_CLAWS1_TEETH1:
			ptr = "Claws+1 & Teeth+1 ";
			break;
		case MWT_TEETH1:
			ptr = "Teeth+1           ";
			break;
		case MWT_BLADE:
			ptr = "Blade             ";
			break;
		case MWT_PIKE:
			ptr = "Pike              ";
			break;
		case MWT_BROADSWORD:
			ptr = "Broadsword        ";
			break;
		case MWT_BODY_PISTOL:
			ptr = "Body Pistol       ";
			break;
	}

	return(ptr);
}

char *
AnimalEncounter::GetTNEWeapon(TNE_WEAPON_TYPE w)
{
char *ptr;

	switch(w) {
		case TWT_HOOVES_TEETH:
			ptr = "Hooves & Teeth   ";
			break;
		case TWT_TRAMPLE:
			ptr = "Trample          ";
			break;
		case TWT_HOOVES:
			ptr = "Hooves           ";
			break;
		case TWT_HORNS_HOOVES:
			ptr = "Horns & Hooves   ";
			break;
		case TWT_HORNS:
			ptr = "Horns            ";
			break;
		case TWT_QUILLS:
			ptr = "Quills           ";
			break;
		case TWT_TAIL:
			ptr = "Tail             ";
			break;
		case TWT_PROJECTILE:
			ptr = "Projectile       ";
			break;
		case TWT_ACID:
			ptr = "Acid             ";
			break;
		case TWT_CLAWS_TEETH:
			ptr = "Claws & Teeth    ";
			break;
		case TWT_COILS:
			ptr = "Coils            ";
			break;
		case TWT_CLAWS:
			ptr = "Claws            ";
			break;
		case TWT_TEETH:
			ptr = "Teeth            ";
			break;
		case TWT_POISON_TEETH:
			ptr = "Poison Teeth     ";
			break;
		case TWT_POISON_CLAWS:
			ptr = "Poison Claws     ";
			break;
		case TWT_POISON_QUILLS:
			ptr = "Poison Quills    ";
			break;
		case TWT_POISON_STINGER:
			ptr = "Poison Stinger   ";
			break;
		case TWT_TENTACLE_ACID:
			ptr = "Tentacle(Acid)   ";
			break;
		case TWT_TENTACLE_POISON:
			ptr = "Tentacle(Poison) ";
			break;
		case TWT_TENTACLE_TEETH:
			ptr = "Tentacle(Teeth)  ";
			break;
	}
		
	return(ptr);
}


void 
AnimalEncounter::GetCTEnc(char *buff)
{
char buff1[20];
//char *ptr;

	sprintf(buff1, "%d/%d", t4_hits[0], t4_hits[1]);
	strfixsize(buff1, 7);
	strcat(buff, buff1);

	strcat(buff, GetCTArmour(ct_armor));

	sprintf(buff1, "%d", ct_wounds);
	strfixsize(buff1, 7);
	strcat(buff, buff1);

	strcat(buff, GetMTWeapon(ct_weapon));
}

void 
AnimalEncounter::GetMTEnc(char *buff)
{
char buff1[20];
//char *ptr;

	sprintf(buff1, "%d", mt_hits);
	strfixsize(buff1, 7);
	strcat(buff, buff1);

	strcat(buff, GetMTArmour(mt_armor));

	sprintf(buff1, "%d", mt_wounds);
	strfixsize(buff1, 7);
	strcat(buff, buff1);

	strcat(buff, GetMTWeapon(ct_weapon));
}

void 
AnimalEncounter::GetTNEEnc(char *buff)
{
char buff1[20];
//char *ptr;

	sprintf(buff1, "%d", tne_hits);
	strfixsize(buff1, 7);
	strcat(buff, buff1);

	strcat(buff, GetTNEArmour(tne_armor));

	strcat(buff, GetTNEWeapon(tne_weapon));

	sprintf(buff1, "%d", tne_to_hit);
	strfixsize(buff1, 6);
	strcat(buff, buff1);

	if((tne_weapon >= TWT_POISON_TEETH) && (tne_weapon <= TWT_POISON_STINGER))
		sprintf(buff1, "%dp%d", tne_dam, tne_dam2);
	else
		sprintf(buff1, "%d", tne_dam);
	strfixsize(buff1, 5);
	strcat(buff, buff1);

	if(tne_pen < 1)
		sprintf(buff1, "Nil");
	else
		sprintf(buff1, "%d", tne_pen);
	strfixsize(buff1, 5);
	strcat(buff, buff1);

	if(0 == tne_range)
		sprintf(buff1, "Short ");
	else
		sprintf(buff1, "Long  ");
	strcat(buff, buff1);

	switch(tne_attack_as) {
		case TAA_ARMED_MELEE:
			strcat(buff, "Armed Melee   ");
			break;
		case TAA_DIVING_BLOW:
			strcat(buff, "Diving Blow   ");
			break;
		case TAA_DEFENSE_ONLY:
			strcat(buff, "Defense Only  ");
			break;
		case TAA_THROWN:
			strcat(buff, "Thrown Weapon ");
			break;
		case TAA_STRANGLE:
			strcat(buff, "Strangle      ");
			break;
	}
	
	sprintf(buff1, "%d", tne_init);
	strfixsize(buff1, 2);
	strcat(buff, buff1);

}

void 
AnimalEncounter::GetT4Enc(char *buff)
{
char buff1[20];
//char *ptr;

	sprintf(buff1, "%d/%d", t4_hits[0], t4_hits[1]);
	strfixsize(buff1, 7);
	strcat(buff, buff1);

	strcat(buff, GetT4Armour(t4_armor));

	sprintf(buff1, "%d", t4_wounds);
	strfixsize(buff1, 7);
	strcat(buff, buff1);

	strcat(buff, GetMTWeapon(t4_weapon));
}


void
AnimalEncounter::Init(TERRAIN_TYPE t, ANIMAL_CLASS a, char *uwp)
{
int i,j;//,k;
int wt_ndx;
int ct_roll,mt_roll,tne_roll,t4_roll;
int ct_a,mt_a,tne_a,t4_a;
int ct_w,mt_w,tne_w,t4_w;
int type_dm=0, wt_dm=0;

	dice = new Dice(6, 2, 0);

	type_dm = terr_dms[t].type_dm;
	wt_dm = terr_dms[t].wt_dm;

// animal type/qualifier
	// look ahead for armour and weapons
	ct_a = mt_a = tne_a = t4_a = ct_w = mt_w = tne_w = t4_w = 0;
	tne_init = dice->Roll(6, 1);
	tne_to_hit = dice->Roll(6, 2);

	i = dice->Roll(6, 2) + type_dm;
	if(i < 0) i = 0;
	if(i > 12) i = 12;
	switch (a) {
		case AC_HERBIVORE:
			ct_a = ct_w = -6;
			mt_a = 2;
			mt_w = -3;
			tne_a = 2;
			tne_w = -3;
			t4_a = 2;
			t4_w = -3;
			tne_init -= 2;
			type = herb_roll[i].type;
			j = herb_roll[i].num;
			break;
		case AC_OMNIVORE:
			mt_w = 4;
			tne_w = 3;
			t4_w = 4;
			tne_init++;
			type = omni_roll[i].type;
			j = omni_roll[i].num;
			break;
		case AC_CARNIVORE:
			mt_a = -1;
			mt_w = 8;
			tne_a = -1;
			tne_w = 5;
			t4_a = 1;
			t4_w = 8;
			ct_a = ct_w = 6;
			tne_init += 2;
			tne_to_hit = dice->Roll(6, 3);
			type = carn_roll[i].type;
			j = carn_roll[i].num;
			break;
		case AC_SCAVENGER:
			mt_a = 1;
			tne_a = 1;
			tne_w = 7;
			t4_a = 1;
			tne_init--;
			type = scav_roll[i].type;
			j = scav_roll[i].num;
			break;
	}

	if(j < 1) num = 1;
	else
		num = dice->Roll(6, j);

	qualifier = AQ_NONE;
	i = dice->Roll(6, 2);
	if(uwp[1] > '7') i--;
	else if(uwp[1] > '4') i++;
	else i += 2;
	if((uwp[2] > '1') && (uwp[2] < '4')) i--;
	else if(uwp[2] > '7') i += 2;
	switch(t) {
		case TER_Beach:
			if(i < 3) {
				qualifier = AQ_SWIMMER;
				wt_dm++;
			} else if(i < 5) {
				qualifier = AQ_AMPHIBIAN;
				wt_dm += 2;
			} else if(i > 11) {
				qualifier = AQ_FLYER;
				wt_dm -= 6;
			} else if(i > 10) {
				qualifier = AQ_FLYER;
				wt_dm -= 5;
			}
			break;
		case TER_Marsh:
			if(i < 3) {
				qualifier = AQ_SWIMMER;
				wt_dm -= 6;
			} else if(i < 5) {
				qualifier = AQ_AMPHIBIAN;
				if(i < 4) wt_dm += 2;
				else wt_dm++;
			} else if(i > 11) {
				qualifier = AQ_FLYER;
				wt_dm -= 6;
			} else if(i > 10) {
				qualifier = AQ_FLYER;
				wt_dm -= 5;
			}
			break;
		case TER_River:
			if(i < 3) {
				qualifier = AQ_SWIMMER;
				wt_dm++;
			} else if(i < 4) {
				qualifier = AQ_AMPHIBIAN;
				wt_dm++;
			} else if(i > 11) {
				qualifier = AQ_FLYER;
				wt_dm -= 6;
			}
			else if(i > 10) {
				qualifier = AQ_FLYER;
				wt_dm -= 5;
			}
			break;
		case TER_Bottom:
		case TER_Depths:
		case TER_SeaCave:
		case TER_Shallows:
		case TER_Surface:
			if((i > 4) && (i < 7)) qualifier = AQ_AMPHIBIAN;
			else if((i > 8) && (i < 11)) qualifier = AQ_TRIPHIBIAN;
			else if(i > 10) qualifier = AQ_FLYER;
			else qualifier = AQ_SWIMMER;
			if(i < 6) wt_dm += 2;
			else if(i == 7) wt_dm++;
			else if(i == 8) wt_dm--;
			else if(i == 9) wt_dm -= 7;
			else if((i == 10) || (i == 11)) wt_dm -= 6;
			else if(i == 12) wt_dm -= 5;
			break;
		case TER_Swamp:
			if(i < 3) {
				qualifier = AQ_SWIMMER;
				wt_dm -= 3;
			} else if(i < 4) {
				qualifier = AQ_AMPHIBIAN;
				wt_dm++;
			} else if(i > 11) {
				qualifier = AQ_FLYER;
				wt_dm -= 6;
			} else if(i > 10) {
				qualifier = AQ_FLYER;
				wt_dm -= 5;
			}
			break;
		default:
			qualifier = AQ_NONE;
			if(13 == i) wt_dm -= 2;
			if(12 == i) wt_dm -= 3;
			else if((11 == i) || (10 == i)) wt_dm -= 6;
			if(i > 10) qualifier = AQ_FLYER;
			break;
	}

// base hits
	j = 0;
	if(uwp[1] < '5') j = 1;
	if(uwp[1] > '7') j = -1;
	wt_ndx = dice->Roll(6, 2) + wt_dm + j;
	if(13 == wt_ndx) wt_ndx = dice->Roll(6, 2) + wt_dm + j + 6;
	if(13 == wt_ndx) wt_ndx = dice->Roll(6, 2) + wt_dm + j;
	if(13 == wt_ndx) wt_ndx = 14;
	if(wt_ndx < 0) wt_ndx = 0;
	else if(wt_ndx > 19) wt_ndx = 19;
	mass = wt_table[wt_ndx].wt;
	mt_hits = dice->Roll(wt_table[wt_ndx].mt_hits.num_die,
					wt_table[wt_ndx].mt_hits.die_size,
					wt_table[wt_ndx].mt_hits.mod);
	tne_hits = dice->Roll(wt_table[wt_ndx].tne_hits.num_die,
					wt_table[wt_ndx].tne_hits.die_size,
					wt_table[wt_ndx].tne_hits.mod);
	t4_hits[0] = dice->Roll(wt_table[wt_ndx].t4_hits.num_die,
					wt_table[wt_ndx].t4_hits.die_size,
					0);
	t4_hits[1] = dice->Roll(wt_table[wt_ndx].t4_hits2.num_die,
					wt_table[wt_ndx].t4_hits2.die_size,
					0);

// wounds and damage
	// need weapon first
	
// weapons
	ct_roll = mt_roll = tne_roll = t4_roll = dice->Roll(6, 2);
	ct_roll += ct_w;
	mt_roll += mt_w;
	tne_roll += tne_w;
	t4_roll += t4_w;
	// ct
	ct_weapon = TravWeapon(ct_roll);
	switch(ct_weapon) {
		case MWT_HORNS_HOOVES:
			i = dice->Roll(6, 2, -5) + dice->Roll(6, 2, -6);
			break;
		case MWT_HORNS:
			i = dice->Roll(6, 2, -5);
			break;
		case MWT_HOOVES_TEETH:
			i = dice->Roll(6, 2, -3) + dice->Roll(6, 2, -6);
			break;
		case MWT_HOOVES:
			i = dice->Roll(6, 2, -6);
			break;
		case MWT_HORNS_TEETH:
			i = dice->Roll(6, 2, -5) + dice->Roll(6, 2, -3);
			break;
		case MWT_THRASHER:
			i = dice->Roll(6, 2, 2);
			break;
		case MWT_CLAWS1_TEETH1:
		case MWT_CLAWS_TEETH:
			i = dice->Roll(6, 1, 3) + dice->Roll(6, 2, -3);
			break;
		case MWT_TEETH1:
		case MWT_TEETH:
			i = dice->Roll(6, 2, -3);
			break;
		case MWT_CLAWS:
		case MWT_CLAWS1:
			i = dice->Roll(6, 1, 3);
			break;
		case MWT_STINGER:
			i = dice->Roll(6, 3, -6);
			break;
		case MWT_BLADE:
			i = dice->Roll(6, 2);
			break;
		case MWT_PIKE:
			i = dice->Roll(6, 3);
			break;
		case MWT_BROADSWORD:
			i = dice->Roll(6, 4);
			break;
		case MWT_BODY_PISTOL:
			i = dice->Roll(6, 3, -8);
			break;
	}
	if(i < 1) i = 1;
	if(wt_table[wt_ndx].t4_wound.die_size > 0) 
		j = dice->Roll(wt_table[wt_ndx].t4_wound.die_size, 
						wt_table[wt_ndx].t4_wound.num_die);
	else j = wt_table[wt_ndx].t4_wound.num_die;
	if('-' == wt_table[wt_ndx].t4_wound_op)
		ct_wounds = i - j;
	else if('+' == wt_table[wt_ndx].t4_wound_op)
		ct_wounds = i + j;
	else if('*' == wt_table[wt_ndx].t4_wound_op)
		ct_wounds = i * j;
	else 
		ct_wounds = i;
	// mt
	mt_weapon = TravWeapon(mt_roll);
	if((MWT_STINGER == mt_weapon) || (MWT_BROADSWORD == mt_weapon) ||
		(MWT_PIKE == mt_weapon) || (MWT_BODY_PISTOL == mt_weapon))
		i = dice->Roll(6, 3);
	else i = dice->Roll(6, 2);
	if(wt_table[wt_ndx].mt_wound_mod.die_size > 0) 
		j = dice->Roll(wt_table[wt_ndx].mt_wound_mod.die_size, 
						wt_table[wt_ndx].mt_wound_mod.num_die,
						wt_table[wt_ndx].mt_wound_mod.mod);
	else j = wt_table[wt_ndx].mt_wound_mod.mod;
	if('-' == wt_table[wt_ndx].mt_wound_mod_op)
		mt_wounds = i - j;
	else if('+' == wt_table[wt_ndx].mt_wound_mod_op)
		mt_wounds = i + j;
	else if('*' == wt_table[wt_ndx].mt_wound_mod_op)
		mt_wounds = i * j;
	else 
		mt_wounds = i;
	// tne
	tne_dam = 1;
	tne_dam2 = -1;
	tne_pen = 0;
	tne_range = 0;
	tne_attack_as = TAA_ARMED_MELEE;
	if(tne_roll < 0) 
		tne_weapon = TWT_HOOVES_TEETH;
	else if(0 == tne_roll) {
		tne_weapon = TWT_TRAMPLE;
		tne_to_hit -= 3;
		tne_dam = dice->Roll(3, 1);
		tne_attack_as = TAA_DIVING_BLOW;
	} else if(1 == tne_roll) {
		tne_weapon = TWT_HOOVES_TEETH;
		tne_dam = dice->Roll(3, 1);
		tne_pen = wt_ndx / 10;
	} else if(2 == tne_roll) {
		tne_weapon = TWT_TRAMPLE;
		tne_to_hit -= 3;
		tne_attack_as = TAA_DIVING_BLOW;
	} else if(3 == tne_roll) {
		tne_weapon = TWT_HOOVES;
		tne_to_hit -= 2;
	} else if(4 == tne_roll) {
		tne_weapon = TWT_HORNS_HOOVES;
		tne_dam = 2;
		tne_pen = wt_ndx / 9;
		if(wt_ndx > 9) tne_range = 1;
	} else if(5 == tne_roll) {
		tne_weapon = TWT_HORNS;
		tne_to_hit -= 1;
		tne_dam = 2;
		tne_pen = wt_ndx / 7;
		if(wt_ndx > 9) tne_range = 1;
	} else if(6 == tne_roll) {
		tne_weapon = TWT_QUILLS;
		tne_to_hit -= 2;
		tne_attack_as = TAA_DEFENSE_ONLY;
	} else if(7 == tne_roll) {
		tne_weapon = TWT_TAIL;
		tne_to_hit -= 2;
		tne_pen = wt_ndx / 7;
		if(wt_ndx > 9) tne_range = 1;
	} else if(8 == tne_roll) {
		tne_weapon = TWT_PROJECTILE;
		tne_dam = dice->Roll(3, 1, -2);
		tne_range = (wt_ndx / 2 ) * dice->Roll(6, 1);
		tne_attack_as = TAA_THROWN;
	} else if(9 == tne_roll) {
		tne_weapon = TWT_TAIL;
		tne_to_hit -= 2;
		if(wt_ndx > 9) tne_range = 1;
	} else if(10 == tne_roll) {
		tne_weapon = TWT_ACID;
		tne_to_hit -= 1;
		tne_dam = dice->Roll(6, 1);
		tne_pen = dice->Roll(6, 1, -4);
	} else if(11 == tne_roll) {
		tne_weapon = TWT_CLAWS_TEETH;
		tne_to_hit += 2;
		tne_dam = dice->Roll(6, 1);
		tne_pen = wt_ndx / 8;
		if(wt_ndx > 8) tne_range = 1;
	} else if(12 == tne_roll) {
		tne_weapon = TWT_COILS;
		tne_to_hit += 1;
		tne_attack_as = TAA_STRANGLE;
	} else if(13 == tne_roll) {
		tne_weapon = TWT_CLAWS;
		tne_to_hit += 2;
		tne_dam = dice->Roll(6, 1, -1);
		tne_pen = wt_ndx / 9;
		if(wt_ndx > 8) tne_range = 1;
	} else if(14 == tne_roll) {
		tne_weapon = TWT_TEETH;
		tne_to_hit += 1;
		tne_dam = dice->Roll(6, 1);
		if(wt_ndx > 9) tne_range = 1;
	} else if(15 == tne_roll) {
		tne_dam2 = dice->Roll(3, 1);
		switch(dice->Roll(6, 1)) {
			case 1:
			case 2:
				tne_weapon = TWT_POISON_TEETH;
				tne_to_hit += 2;
				tne_dam = dice->Roll(3, 1);
				break;
			case 3:
				tne_weapon = TWT_POISON_CLAWS;
				tne_to_hit += 2;
				tne_dam = dice->Roll(3, 1);
				break;
			case 4:
			case 5:
				tne_weapon = TWT_POISON_QUILLS;
				tne_dam = 1;
				tne_attack_as = TAA_DEFENSE_ONLY;
				break;
			default:
				tne_weapon = TWT_POISON_STINGER;
				tne_to_hit += 1;
				tne_dam = 1;
				break;
		}
	} else if(16 == tne_roll) {
		switch(dice->Roll(6, 1)) {
			case 1:
				tne_weapon = TWT_TENTACLE_ACID;
				break;
			case 2:
				tne_weapon = TWT_TENTACLE_POISON;
				break;
			default:
				tne_weapon = TWT_TENTACLE_TEETH;
				break;
		}
		if(wt_ndx > 5) tne_range = 1;
	} else {
		tne_weapon = TWT_CLAWS;
		tne_to_hit += 1;
		tne_dam = dice->Roll(6, 1);
		tne_pen = wt_ndx / 9;
		if(wt_ndx > 8) tne_range = 1;
	}

	tne_init += wt_table[wt_ndx].tne_init_dm;

	if(wt_table[wt_ndx].tne_dam.die_size > 0) 
		j = dice->Roll(wt_table[wt_ndx].tne_dam.die_size, 
						wt_table[wt_ndx].tne_dam.num_die,
						wt_table[wt_ndx].tne_dam.mod);
	else j = wt_table[wt_ndx].tne_dam.mod;
	if('/' == wt_table[wt_ndx].tne_dam_op)
		tne_dam /= j;
	else if('+' == wt_table[wt_ndx].tne_dam_op)
		tne_dam += j;
	else if('*' == wt_table[wt_ndx].tne_dam_op)
		tne_dam *= j;

	if(tne_to_hit < 1) tne_to_hit = 1;
	if(tne_dam < 1) tne_dam = 1;
	if(tne_pen < 0) tne_pen = 0;
	if(tne_init < 0) tne_init = 0;

	// t4
	t4_weapon = TravWeapon(t4_roll);
	if((MWT_HORNS_HOOVES == t4_weapon) || (MWT_HOOVES_TEETH == t4_weapon) ||
		(MWT_HORNS_TEETH == t4_weapon) || (MWT_CLAWS_TEETH == t4_weapon) ||
		(MWT_CLAWS1_TEETH1 == t4_weapon)) i = dice->Roll(6, 4);
	else if(t4_weapon != MWT_BLADE) i = dice->Roll(6, 2);
	else i = dice->Roll(6, 1);
	if(wt_table[wt_ndx].t4_wound.die_size > 0) 
		j = dice->Roll(wt_table[wt_ndx].t4_wound.die_size, 
						wt_table[wt_ndx].t4_wound.num_die);
	else j = wt_table[wt_ndx].t4_wound.num_die;
	if('-' == wt_table[wt_ndx].t4_wound_op)
		t4_wounds = i - j;
	else if('+' == wt_table[wt_ndx].t4_wound_op)
		t4_wounds = i + j;
	else if('*' == wt_table[wt_ndx].t4_wound_op)
		t4_wounds = i * j;
	else 
		t4_wounds = i;

// armour
	ct_roll = mt_roll = tne_roll = t4_roll = dice->Roll(6, 2);
	i = dice->Roll(6, 2);	// rerolls
	j = dice->Roll(6, 2);	// rerolls
	ct_roll += ct_a;
	mt_roll += mt_a;
	tne_roll += tne_a;
	t4_roll += t4_a;
	// ct
	if(ct_roll < 2) ct_armor = CAT_JACK;
	else if(4 == ct_roll) ct_armor = CAT_MESH;
	else if(5 == ct_roll) ct_armor = CAT_CLOTH;
	else ct_armor = CAT_NONE;
	// mt
	if((mt_roll < 2) || (12 == mt_roll)) mt_roll = i + mt_a + 6;
	if((mt_roll < 2) || (12 == mt_roll)) mt_roll = j + mt_a;
	if((4 == mt_roll) || (10 == mt_roll)) mt_armor = MAT_JACK;
	else if((12 == mt_roll) || (13 == mt_roll)) mt_armor = MAT_MESH1;
	else if(14 == mt_roll) mt_armor = MAT_CLOTH1;
	else if(15 == mt_roll) mt_armor = MAT_MESH;
	else if(16 == mt_roll) mt_armor = MAT_CLOTH;
	else if(17 == mt_roll) mt_armor = MAT_COMBAT4;
	else if(18 == mt_roll) mt_armor = MAT_REFLEC;
	else if(19 == mt_roll) mt_armor = MAT_ABLAT;
	else if(mt_roll > 19) mt_armor = MAT_BATTLE;
	else mt_armor = MAT_NONE;
	// tne
	tne_a += wt_table[wt_ndx].tne_arm_dm;
	if((tne_roll < 2) || (12 == tne_roll)) tne_roll = i + tne_a + 6;
	if((tne_roll < 2) || (12 == tne_roll)) tne_roll = j + tne_a;
	if((4 == tne_roll) || (10 == tne_roll) || 
			(12 == tne_roll) || (13 == tne_roll))
		tne_armor = TAT_HALF;
	else if((tne_roll > 13) && (tne_roll < 17))
		tne_armor = TAT_HALF_HEAD1;
	else if((17 == tne_roll) || (18 == tne_roll))
		tne_armor = TAT_ONE_HEAD2;
	else if(19 == tne_roll)
		tne_armor = TAT_TWO_HEAD3;
	else if(tne_roll > 19)
		tne_armor = TAT_THREE;
	else tne_armor = TAT_NONE;
	// t4
	if((t4_roll < 2) || (12 == t4_roll)) t4_roll = i + t4_a + 6;
	if((t4_roll < 2) || (12 == t4_roll)) t4_roll = j + t4_a;
	if((4 == t4_roll) || (10 == t4_roll) || (12 == t4_roll) || (14 == t4_roll))
		t4_armor = T4AT_TWO;
	else if(13 == t4_roll) t4_armor = T4AT_THREE;
	else if(15 == t4_roll) t4_armor = T4AT_ONE_RIGID;
	else if(16 == t4_roll) t4_armor = T4AT_ONE;
	else if(17 == t4_roll) t4_armor = T4AT_TWO_RIGID;
	else if(18 == t4_roll) t4_armor = T4AT_FOUR;
	else if(19 == t4_roll) t4_armor = T4AT_THREE_RIGID;
	else if(t4_roll > 19) t4_armor = T4AT_FOUR_RIGID;
	else t4_armor = T4AT_NONE;
// behavior
	ct_attack = CrackTypeCode(types[type].ct_attack);
	ct_flee = CrackTypeCode(types[type].ct_flee);
	ct_speed = CrackSpeedCode(types[type].ct_speed);
	mt_attack = CrackTypeCode(types[type].mt_attack.roll);
	if(mt_attack < types[type].mt_attack.min)
		mt_attack = types[type].mt_attack.min;
	mt_flee = CrackTypeCode(types[type].mt_flee.roll);
	if(mt_flee < types[type].mt_flee.min)
		mt_flee = types[type].mt_flee.min;
	mt_speed = CrackTypeCode(types[type].mt_speed.roll);
	if(mt_speed < types[type].mt_speed.min)
		mt_speed = types[type].mt_speed.min;
	tne_attack = CrackTypeCode(types[type].tne_attack);
	tne_flee = CrackTypeCode(types[type].tne_flee);
	tne_speed = dice->Roll(20, types[type].tne_speed);
}

AnimalEncounter::~AnimalEncounter()
{
	delete dice;
}

int 
AnimalEncounter::CrackTypeCode(ROLL_TYPE t)
{
	switch(t) {
		case RT_POSSIBLE:
			return('p');
			break;
		case RT_SURPRISE:
			return('s');
			break;
		case RT_SURPRISED:
			return('s');
			break;
		case RT_MORE:
			return('m');
			break;
		case RT_1D_M_5:
			return(dice->Roll(6, 1, -5));
			break;
		case RT_1D_M_4:
			return(dice->Roll(6, 1, -4));
			break;
		case RT_1D_M_3:
			return(dice->Roll(6, 1, -3));
			break;
		case RT_1D_M_2:
			return(dice->Roll(6, 1, -2));
			break;
		case RT_1D_M_1:
			return(dice->Roll(6, 1, -1));
			break;
		case RT_1D:
			return(dice->Roll(6, 1));
			break;
		case RT_1D_P_1:
			return(dice->Roll(6, 1, 1));
			break;
		case RT_1D_P_2:
			return(dice->Roll(6, 1, 2));
			break;
		case RT_1D_P_3:
			return(dice->Roll(6, 1, 3));
			break;
		case RT_2D:
			return(dice->Roll(6, 2));
			break;
		case RT_3D:
			return(dice->Roll(6, 3));
			break;
		case RT_3D_P_2:
			return(dice->Roll(6, 3, 2));
			break;
		case RT_5:
			return(5);
			break;
		case RT_6:
			return(6);
			break;
		case RT_7:
			return(7);
			break;
		case RT_8:
			return(8);
			break;
		case RT_9:
			return(9);
			break;
		case RT_10:
			return(10);
			break;
		case RT_11:
			return(11);
			break;
	}
	return(0);
}

int 
AnimalEncounter::CrackSpeedCode(CT_SPEEDS s)
{
	switch(s) {
		case CS_ONE:
			return(1);
			break;
		case CS_TWO:
			return(2);
			break;
		case CS_THREE:
			return(3);
			break;
		case CS_ONE_ZERO:
			if(dice->Roll(6, 2) < 7)
				return(0);
			else
				return(1);
			break;
		case CS_ONE_TWO:
			if(dice->Roll(6, 2) < 7)
				return(2);
			else
				return(1);
			break;
		case CS_THREE_TWO:
			if(dice->Roll(6, 2) < 7)
				return(2);
			else
				return(3);
			break;
		case CS_TWO_ONE:
			if(dice->Roll(6, 2) < 7)
				return(1);
			else
				return(2);
			break;
	}
	return(0);
}

MT_WEAPON_TYPE
AnimalEncounter::TravWeapon(int roll)
{
MT_WEAPON_TYPE weapon;

	if(roll < 2) weapon = MWT_HORNS_HOOVES;
	else if(roll < 3) weapon = MWT_HORNS;
	else if(roll < 4) weapon = MWT_HOOVES_TEETH;
	else if(roll < 5) weapon = MWT_HOOVES;
	else if(roll < 6) weapon = MWT_HORNS_TEETH;
	else if(roll < 7) weapon = MWT_THRASHER;
	else if(roll < 8) weapon = MWT_CLAWS_TEETH;
	else if(roll < 9) weapon = MWT_TEETH;
	else if(roll < 11) weapon = MWT_CLAWS;
	else if(roll < 12) weapon = MWT_THRASHER;
	else if(roll < 13) weapon = MWT_CLAWS_TEETH;
	else if(roll < 14) weapon = MWT_CLAWS1;
	else if(roll < 15) weapon = MWT_STINGER;
	else if(roll < 16) weapon = MWT_CLAWS1_TEETH1;
	else if(roll < 17) weapon = MWT_TEETH1;
	else if(roll < 18) weapon = MWT_BLADE;
	else if(roll < 19) weapon = MWT_PIKE;
	else if(roll < 20) weapon = MWT_BROADSWORD;
	else weapon = MWT_BODY_PISTOL;

	return(weapon);
}
// =========================================================================
EncounterTable::EncounterTable(TERRAIN_TYPE t, char *uwp, 
								int s, int n, char *f) :
		wxList()
{
int i;
AnimalEncounter *ae;

	for(i = 0;i < (((n * s) - n) + 1);i++) {
		ae = new AnimalEncounter(t, f[i], uwp);
		Append(ae);
	}
}

EncounterTable::~EncounterTable()
{
wxNode *n;
AnimalEncounter *ae;

	n = First();
	while(n != NULL) {
		ae = (AnimalEncounter *)n->Data();
		delete ae;
		delete n;
		n = First();
	}
}

AnimalEncounter *
EncounterTable::GetEnc(int ndx)
{
//int count=0;
wxNode *n;
AnimalEncounter *ae=NULL;

	n = Nth(ndx);
	if(NULL != n)
		ae = (AnimalEncounter *)n->Data();

	return(ae);
}

// =========================================================================
EncounterTables::EncounterTables(char *uwp, int s, int n, char *f)
{
int i;

	die_size = s;
	die_num = n;
	format = new char[strlen(f) + 1];
	strcpy(format, f);
	
	dice = new Dice(die_size, die_num, 0);
	for(i = 0;i < MAX_TERRAIN;i++)
		table[i] = new EncounterTable((TERRAIN_TYPE) i, uwp, 
								die_size, die_num, format);
}

EncounterTables::~EncounterTables()
{
int i;

	delete dice;
	delete format;
	for(i = 0;i < MAX_TERRAIN;i++)
		delete table[i];
}

// =========================================================================
char *EncTerrain::GetTerrain(TERRAIN_TYPE t)
{
	return(terr_dms[t].terr);
}

char *EncTerrain::GetEquiv(TERRAIN_TYPE t)
{
	return(terr_dms[t].equiv);
}

char *
EncTerrain::GetEncHeader(TRAV_VER v, char *buff, bool table)
{
	buff[0] = 0;

	if(table)
		strcat(buff, "Die ");
	strcat(buff, "Animal                     Mass    ");
	switch(v) {
		case TV_CT:
			strcat(buff, "Hits   Armour   Wounds Weapons");
			break;
		case TV_MT:
			strcat(buff, "Hits   Armour   Wounds Weapons");
			break;
		case TV_TNE:
			strcat(buff, 
				"Hits   Armour      Weapons          ToHit Dam. Pen. Range TreatAs       Init.");
			break;
		case TV_T4:
			strcat(buff, "Hits   Armour  Wounds Weapons");
			break;
	}

	return(buff);
}

