/*
 * File:      	iso_map.cpp
 * Purpose:   	base mapping
 * Author:		Mark A. Nordstrand
 * Created:	
 * Updated:	
 * Copyright:	LGPL
Traveller is a registered trademark of Far Future Enterprises.
Portions based upon material Copyright 1977-1999 Far Future Enterprises.
 */

/* rcsid[] = "$RCSfile: iso_map.cpp,v $ $Revision: 1.1 $ $Author: man $ $Date: 2003/04/26 20:29:14 $" */

//#define STUPID_DEBUG_CODE
#ifdef STUPID_DEBUG_CODE
#include <stdio.h>
#endif

// various debugging stuff...
#define HEX_LINK_CHECK
//#define HEX_LINK_DUMP

#include <stdio.h>
//#include <stdlib.h>
#include "iso_map.h"
#include "t_dice.h"

// ---------------------------------------------------------------------------
// (hideous) look up tables
// mask is a helper for drawing (maybe it should be in map.h?)
unsigned char 
IsoMap::mask[ISO_MAP_SECTION][ISO_MAP_MAX_Y] = {
	{  0x0,  0xf,  0xf, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f,
	  0x1f, 0x1f, 0x1f, 0x1f, 0x10, 0x10,  0x0,  0x0,  0x0,  0x0,  0x0 },
	{  0x0,  0x0,  0x0,  0xf,  0xf, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 
	  0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x10, 0x10,  0x0,  0x0,  0x0 },
	{  0x0,  0x0,  0x0,  0x0,  0x0,  0xf,  0xf, 0x1f, 0x1f, 0x1f, 0x1f, 
	   0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x10, 0x10,  0x0 },
	{  0x0,  0x0,  0x0,  0x0,  0x0,  0x0,  0x1,  0xf, 0x1f, 0x1f, 0x1f, 
	  0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1e, 0x10 },
	{  0x0,  0x0,  0x0,  0x0,  0x1,  0x1, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 
	  0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1e, 0x1e,  0x0,  0x0 },
	{  0x0,  0x0,  0x1,  0x1, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f,
	  0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1e, 0x1e,  0x0,  0x0,  0x0,  0x0 },
	{  0x1,  0x1, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 
	  0x1f, 0x1f, 0x1f, 0x1e,  0x1e,  0x0, 0x0,  0x0,  0x0,  0x0,  0x0 }
};

// NOTE:  many of the following are dimensioned [ISO_MAP_SECTION][ISO_MAP_MAX_Y]
// 		  since the values effectively repeat.  access is by
// 		  [x % ISO_MAP_SECTION][y]

// face index is a quick way to determine which face a hex is on
// NOTE: -2 is a vertex, and -1 is a nonexistant hex
char
IsoMap::face_index[ISO_MAP_SECTION][ISO_MAP_MAX_Y] = {
	{-2,-1, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 2,-2, 3,-1,-1,-1,-1,-1,-1 },
	{-1,-1,-1,-1, 0, 0, 0, 0, 1, 1, 1, 2, 2, 2, 2, 3, 3, 3,-1,-1,-1,-1 },
	{-1,-1,-1,-1,-1,-1, 0, 0, 1, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3,-1,-1 },
	{-1,-1,-1,-1,-1,-1,-1,-2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3,-2 },
	{-1,-1,-1, 4, 4, 4, 4, 4, 5, 5, 2, 2, 2, 2, 2, 3, 3, 3, 3,-1,-1,-1 },
	{-1,-1, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 2, 2, 2, 3, 3,-1,-1,-1,-1,-1 },
	{-2, 4, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 5, 5, 2,-1,-1,-1,-1,-1,-1,-1 }
};

// half_hex is another mapping used to handle the hex halves 
// which occur where the triangle faces are apart (top and bottom
// 5 faces).  the value will be the amount to adjust (+/-) to get 
// the x coordinate for the other pair (hence, a zero has no 
// means no translation).
char
IsoMap::half_hex[ISO_MAP_SECTION][ISO_MAP_MAX_Y] = {
	{ 0, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,-1, 0, 0, 0, 0, 0, 0 },
	{ 0, 0, 5, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,-2,-3, 0, 0, 0, 0 },
	{ 0, 0, 0, 0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,-4,-5, 0, 0 },
	{ 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,-6, 0 },
	{ 0, 0, 0, 0, 0,-2,-1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 6, 0 },
	{ 0, 0, 0,-4,-3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 4, 0, 0, 0 },
	{ 0,-6,-5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0 }
};

// face_base is the absolute x,y of a face
// XXX shit, x and y are intuitively reversed....
char
IsoMap::face_base[MAX_FACES][2] = {
	{  0, -3 },
	{  7, -3 },
	{  7,  0 },
	{ 14,  0 },
	{  0,  4 },
	{  7,  4 },
	{  7,  7 },
	{ 14,  7 },
	{  0, 11 },
	{  7, 11 },
	{  7, 14 },
	{ 14, 14 },
	{  0, 18 },
	{  7, 18 },
	{  7, 21 },
	{ 14, 21 },
	{  0, 25 },
	{  7, 25 },
	{  7, 28 },
	{ 14, 28 }
};

// determines hex orientation relative to rotation
unsigned char
IsoMap::rotation[MAX_FACES][6] = {
	{ 0, 2, 1, 0, 5, 4 },		// 0
	{ 3, 4, 4, 3, 2, 2 },		// 1
	{ 0, 1, 1, 5, 5, 0 },		// 2
	{ 3, 4, 5, 1, 2, 3 },		// 3
	{ 0, 1, 0, 5, 4, 2 },		// 4
	{ 3, 4, 3, 2, 2, 4 },		// 5
	{ 0, 1, 5, 5, 0, 1 },		// 6
	{ 3, 5, 1, 2, 3, 4 },		// 7
	{ 0, 0, 5, 4, 2, 1 },		// 8
	{ 3, 3, 2, 2, 4, 4 },		// 9
	{ 0, 5, 5, 0, 1, 1 },		// 10
	{ 3, 1, 2, 3, 4, 5 },		// 11
	{ 0, 5, 4, 2, 1, 0 },		// 12
	{ 3, 2, 2, 4, 4, 3 },		// 13
	{ 0, 5, 0, 1, 1, 5 },		// 14
	{ 3, 2, 3, 4, 5, 1 },		// 15
	{ 0, 4, 2, 1, 0, 5 },		// 16
	{ 3, 2, 4, 4, 3, 2 },		// 17
	{ 0, 0, 1, 1, 5, 5 },		// 18
	{ 3, 3, 4, 5, 1, 2 }		// 19
};

// where each face lies for any orientation
unsigned char
IsoMap::face_xlat[MAX_FACES][6] = {
	{  0,  0,  1,  2,  5,  4 },
	{  1, 16, 18,  3,  6,  8 },
	{  2, 17, 19,  7,  9, 12 },
	{  3, 14, 15, 11, 10, 13 },
	{  4,  1,  2,  5,  4,  0 },
	{  5, 18,  3,  6,  8, 16 },
	{  6, 19,  7,  9, 12, 17 },
	{  7, 15, 11, 10, 13, 14 },
	{  8,  2,  5,  4,  0,  1 },
	{  9,  3,  6,  8, 16, 18 },
	{ 10,  7,  9, 12, 17, 19 },
	{ 11, 11, 10, 13, 14, 15 },
	{ 12,  5,  4,  0,  1,  2 },
	{ 13,  6,  8, 16, 18,  3 },
	{ 14,  9, 12, 17, 19,  7 },
	{ 15, 10, 13, 14, 15, 11 },
	{ 16,  4,  0,  1,  2,  5 },
	{ 17,  8, 16, 18,  3,  6 },
	{ 18, 12, 17, 19,  7,  9 },
	{ 19, 13, 14, 15, 11, 10 }
};

// rotational position
char
IsoMap::hex_xlat_even[6][7][7][2] = {
 {	{ {-1,-1}, {-1,-1}, { 2, 1}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, {-1,-1}, { 2, 2}, { 3, 2}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, { 1, 3}, { 2, 3}, { 3, 3}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, { 1, 4}, { 2, 4}, { 3, 4}, { 4, 4}, {-1,-1}, {-1,-1} },
	{ { 0, 5}, { 1, 5}, { 2, 5}, { 3, 5}, { 4, 5}, {-1,-1}, {-1,-1} },
	{ { 0, 6}, { 1, 6}, { 2, 6}, { 3, 6}, { 4, 6}, { 5, 6}, {-1,-1} },
	{ { 0, 7}, { 1, 7}, { 2, 7}, { 3, 7}, { 4, 7}, { 5, 7}, {-1,-1} } },
 {	{ {-1,-1}, {-1,-1}, { 5, 0}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, {-1,-1}, { 4, 0}, { 5, 1}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, { 3, 0}, { 4, 1}, { 4, 2}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, { 2, 0}, { 3, 1}, { 3, 2}, { 4, 3}, {-1,-1}, {-1,-1} },
	{ { 1, 0}, { 2, 1}, { 2, 2}, { 3, 3}, { 3, 4}, {-1,-1}, {-1,-1} },
	{ { 0, 0}, { 1, 1}, { 1, 2}, { 2, 3}, { 2, 4}, { 3, 5}, {-1,-1} },
	{ { 0, 1}, { 0, 2}, { 1, 3}, { 1, 4}, { 2, 5}, { 2, 6}, {-1,-1} } },
 {	{ {-1,-1}, {-1,-1}, { 6, 6}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, {-1,-1}, { 5, 5}, { 5, 6}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, { 5, 4}, { 4, 5}, { 4, 6}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, { 4, 3}, { 4, 4}, { 3, 5}, { 3, 6}, {-1,-1}, {-1,-1} },
	{ { 4, 2}, { 3, 3}, { 3, 4}, { 2, 5}, { 2, 6}, {-1,-1}, {-1,-1} },
	{ { 3, 1}, { 3, 2}, { 2, 3}, { 2, 4}, { 1, 5}, { 1, 6}, {-1,-1} },
	{ { 2, 1}, { 2, 2}, { 1, 3}, { 1, 4}, { 0, 5}, { 0, 6}, {-1,-1} } },
 {	{ {-1,-1}, {-1,-1}, { 2, 6}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, {-1,-1}, { 3, 5}, { 2, 5}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, { 3, 4}, { 2, 4}, { 1, 4}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, { 4, 3}, { 3, 3}, { 2, 3}, { 1, 3}, {-1,-1}, {-1,-1} },
	{ { 4, 2}, { 3, 2}, { 2, 2}, { 1, 2}, { 0, 2}, {-1,-1}, {-1,-1} },
	{ { 5, 1}, { 4, 1}, { 3, 1}, { 2, 1}, { 1, 1}, { 0, 1}, {-1,-1} },
	{ { 5, 0}, { 4, 0}, { 3, 0}, { 2, 0}, { 1, 0}, { 0, 0}, {-1,-1} } },
 {	{ {-1,-1}, {-1,-1}, { 0, 7}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, {-1,-1}, { 1, 7}, { 1, 6}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, { 2, 7}, { 2, 6}, { 1, 5}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, { 3, 7}, { 3, 6}, { 2, 5}, { 2, 4}, {-1,-1}, {-1,-1} },
	{ { 4, 7}, { 4, 6}, { 3, 5}, { 3, 4}, { 2, 3}, {-1,-1}, {-1,-1} },
	{ { 5, 7}, { 5, 6}, { 4, 5}, { 4, 4}, { 3, 3}, { 3, 2}, {-1,-1} },
	{ { 6, 6}, { 5, 5}, { 5, 4}, { 4, 3}, { 4, 2}, { 3, 1}, {-1,-1} } },
 {	{ {-1,-1}, {-1,-1}, { 0, 1}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, {-1,-1}, { 0, 2}, { 1, 1}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, { 1, 3}, { 1, 2}, { 2, 1}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, { 1, 4}, { 2, 3}, { 2, 2}, { 3, 1}, {-1,-1}, {-1,-1} },
	{ { 2, 5}, { 2, 4}, { 3, 3}, { 3, 2}, { 4, 1}, {-1,-1}, {-1,-1} },
	{ { 2, 6}, { 3, 5}, { 3, 4}, { 4, 3}, { 4, 2}, { 5, 1}, {-1,-1} },
	{ { 3, 6}, { 4, 5}, { 4, 4}, { 5, 3}, { 5, 2}, { 6, 1}, {-1,-1} } }
};

char
IsoMap::hex_xlat_odd[6][7][7][2] = {
 {	{ {-1,-1}, {-1,-1}, { 2, 1}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, {-1,-1}, { 2, 2}, { 3, 2}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, { 1, 1}, { 2, 3}, { 3, 3}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, { 1, 2}, { 2, 4}, { 3, 4}, { 4, 4}, {-1,-1}, {-1,-1} },
	{ { 0, 5}, { 1, 5}, { 2, 5}, { 3, 5}, { 4, 5}, {-1,-1}, {-1,-1} },
	{ { 0, 6}, { 1, 6}, { 2, 6}, { 3, 6}, { 4, 6}, { 5, 6}, {-1,-1} },
	{ {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1} } },
 {	{ {-1,-1}, {-1,-1}, { 6, 1}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, {-1,-1}, { 5, 1}, { 5, 2}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, { 4, 1}, { 4, 2}, { 5, 3}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, { 3, 1}, { 3, 2}, { 4, 3}, { 4, 4}, {-1,-1}, {-1,-1} },
	{ { 2, 1}, { 2, 2}, { 3, 3}, { 3, 4}, { 4, 5}, {-1,-1}, {-1,-1} },
	{ { 1, 1}, { 1, 2}, { 2, 3}, { 2, 4}, { 3, 5}, { 3, 6}, {-1,-1} },
	{ {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1} } },
 {	{ {-1,-1}, {-1,-1}, { 5, 7}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, {-1,-1}, { 5, 6}, { 4, 7}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, { 4, 5}, { 4, 6}, { 3, 7}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, { 4, 4}, { 3, 5}, { 3, 6}, { 2, 7}, {-1,-1}, {-1,-1} },
	{ { 3, 3}, { 3, 4}, { 2, 5}, { 2, 6}, { 1, 7}, {-1,-1}, {-1,-1} },
	{ { 3, 2}, { 2, 3}, { 2, 4}, { 1, 5}, { 1, 6}, { 0, 7}, {-1,-1} },
	{ {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1} } },
 {	{ {-1,-1}, {-1,-1}, { 2, 6}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, {-1,-1}, { 3, 5}, { 2, 5}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, { 3, 4}, { 2, 4}, { 1, 4}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, { 4, 3}, { 3, 3}, { 2, 3}, { 1, 3}, {-1,-1}, {-1,-1} },
	{ { 4, 2}, { 3, 2}, { 2, 2}, { 1, 2}, { 0, 2}, {-1,-1}, {-1,-1} },
	{ { 5, 1}, { 4, 1}, { 3, 1}, { 2, 1}, { 1, 1}, { 0, 1}, {-1,-1} },
	{ {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1} } },
 {	{ {-1,-1}, {-1,-1}, { 0, 6}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, {-1,-1}, { 1, 6}, { 0, 5}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, { 2, 6}, { 1, 5}, { 1, 4}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, { 3, 6}, { 2, 5}, { 2, 4}, { 1, 3}, {-1,-1}, {-1,-1} },
	{ { 4, 6}, { 3, 5}, { 3, 4}, { 2, 3}, { 2, 2}, {-1,-1}, {-1,-1} },
	{ { 5, 6}, { 4, 5}, { 4, 4}, { 3, 3}, { 3, 2}, { 2, 1}, {-1,-1} },
	{ {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1} } },
 {	{ {-1,-1}, {-1,-1}, { 0, 0}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, {-1,-1}, { 1, 1}, { 1, 0}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, { 1, 2}, { 2, 1}, { 2, 0}, {-1,-1}, {-1,-1}, {-1,-1} },
	{ {-1,-1}, { 2, 3}, { 2, 2}, { 3, 1}, { 3, 0}, {-1,-1}, {-1,-1} },
	{ { 2, 4}, { 3, 3}, { 3, 2}, { 4, 1}, { 4, 0}, {-1,-1}, {-1,-1} },
	{ { 3, 5}, { 3, 4}, { 4, 3}, { 4, 2}, { 5, 1}, { 5, 0}, {-1,-1} },
	{ {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1}, {-1,-1} } },
};

// vertex absolute coordinate positioning
unsigned char
IsoMap::vertex_xlat[6][MAX_VERTICES][2] = {
 {	{ 28,  0 }, {  3,  7 }, { 10,  7 }, { 17,  7 }, { 24,  7 }, { 31,  7 },
	{ 31, 21 }, {  7, 14 }, { 14, 14 }, { 21, 14 }, { 28, 14 }, {  0, 14 } },
 {	{  3,  7 }, { 31,  7 }, {  0, 14 }, {  7, 14 }, { 10,  7 }, { 28,  0 },
	{ 21, 14 }, { 28, 14 }, { 31, 21 }, { 14, 14 }, { 17,  7 }, { 24,  7 } },
 {	{  3,  7 }, {  0, 14 }, {  7, 14 }, { 10,  7 }, { 28,  0 }, { 31,  7 },
	{ 21, 14 }, { 31, 21 }, { 14, 14 }, { 17,  7 }, { 24,  7 }, { 28, 14} },
 {	{  3,  7 }, {  7, 14 }, { 10,  7 }, { 28,  0 }, { 31,  7 }, {  0, 14 }, 
	{ 21, 14 }, { 14, 14 }, { 17,  7 }, { 24,  7 }, { 28, 14 }, { 31, 21 } },
 {	{  3,  7 }, { 10,  7 }, { 28,  0 }, { 31,  7 }, {  0, 14 }, {  7, 14 }, 
	{ 21, 14 }, { 17,  7 }, { 24,  7 }, { 28, 14 }, { 31, 21 }, { 14, 14 } },
 {	{  3,  7 }, { 28,  0 }, { 31,  7 }, {  0, 14 }, {  7, 14 }, { 10,  7 }, 
	{ 21, 14 }, { 24,  7 }, { 28, 14 }, { 31, 21 }, { 14, 14 }, { 17,  7 } }
};

unsigned char 
IsoMap::vertex_rotation[6][MAX_VERTICES] = {
	{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
	{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
	{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
	{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
	{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
	{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }
};

// ---------------------------------------------------------------------------
IsoMap::IsoMap() : 
		MapObject(NULL, MOT_MAP)
{
int i,j,f;

	for(i = 0;i < MAX_PLATES;i++)
		plates[i] = NULL;
	
	for(i = 0;i < MAX_FACES;i++) {
		faces[i] = new MapFace(this, (i & 1));
//		faces[i]->DumpFace();
	}

	vertices[N_VERTEX_INDEX] = new MapHex(this, HT_N_POLE);
	vertices[S_VERTEX_INDEX] = new MapHex(this, HT_S_POLE);
	j = 1;
	for(i = 0;i < S_VERTEX_INDEX;i++) {
		vertices[j] = new MapHex(this, HT_N_VERTEX);
		vertices[j + S_VERTEX_INDEX] = new MapHex(this, HT_S_VERTEX);
		j++;
	}
	for(f = 0;f < MAX_FACES;f++) {
		for(i = 0;i < MAP_FACE_MAX_X;i++) {
			for(j = 0;j < MAP_FACE_MAX_Y;j++) {
				LinkHex(f, i, j);
			}
		}
	}
	for(i = 0;i < MAX_VERTICES;i++) {
		LinkHex(VERTEX_FACE_INDEX, i, VERTEX_FACE_INDEX);
	}
#ifdef PLATE_GROW_CODE
	blank_no = -1;
#endif
#ifdef HEX_LINK_CHECK
#ifdef HEX_LINK_DUMP
	// dump everything
	// use stdout for quick redirection
	for(i = 0;i < MAX_VERTICES;i++) {
		int k,f1,x1,y1;
		MapHex *hex,*hex1;

		if((hex = GetHex(VERTEX_FACE_INDEX, i, VERTEX_FACE_INDEX)) != NULL) {
			fprintf(stdout, "[%d (%d %d)] ", VERTEX_FACE_INDEX, i,
							VERTEX_FACE_INDEX);
			for(k = 0;k < 6;k++) {
				hex->GetAdj(k, &f1, &x1, &y1);
				if((hex1 = GetHex(f1, x1, y1)) != NULL) {
					fprintf(stdout, "%d:[%d (%d %d)] ", k, f1, x1, y1);
				}
			}
		}
		fprintf(stdout, "\n");
	}
	for(f = 0;f < MAX_FACES;f++) {
		for(i = 0;i < MAP_FACE_MAX_X;i++) {
			for(j = 0;j < MAP_FACE_MAX_Y;j++) {
				int k,f1,x1,y1;
				MapHex *hex,*hex1;

				if((hex = GetHex(f, i, j)) != NULL) {
					fprintf(stdout, "[%d (%d %d)] ", f, i, j);
					for(k = 0;k < 6;k++) {
						hex->GetAdj(k, &f1, &x1, &y1);
						if((hex1 = GetHex(f1, x1, y1)) != NULL) {
							fprintf(stdout, "%d:[%d (%d %d)] ", k, f1, x1, y1);
						}
					}
					fprintf(stdout, "\n");
				}
			}
		}
	}
#endif

#if 0
	for(i = (S_VERTEX_INDEX + 1);i < MAX_VERTICES;i++) {
		int j,k,f1,x1,y1,x,y;
		MapHex *hex,*hex1;

		if((hex = GetHex(VERTEX_FACE_INDEX, i, VERTEX_FACE_INDEX)) != NULL) {
			fprintf(stderr, "========== vertex %d =============\n", i);
			for(k = 0;k < 6;k++) {
				hex->GetAdj(k, &f, &x, &y);
				if((hex1 = GetHex(f, x, y)) != NULL) {
					fprintf(stderr, "%d: [%d (%d %d)]  ", k, f, x, y);
					for(j = 0;j < 6;j++) {
						hex1->GetAdj(j, &f1, &x1, &y1);
						fprintf(stderr, "%d: [%d (%d %d)]  ", j, f1, x1, y1);
					}
					fprintf(stderr, "\n");
				} else
					fprintf(stderr, "Bad dir %d\n", k);
			}
		}
	}
#endif

	// layout/link verification
	// poles only should have 1 'correct' location
	// N vertices will return 'bad' for direction 5
	// S vertices will return 'bad' for direction 3
	// so, these must be verified by hand....
	for(i = 1;i < MAX_VERTICES;i++) {
		int j,k,f1,x1,y1,x,y;
		MapHex *hex,*hex1;

		if(S_VERTEX_INDEX == i)
			continue;
		if((hex = GetHex(VERTEX_FACE_INDEX, i, VERTEX_FACE_INDEX)) != NULL) {
			for(k = 0;k < 6;k++) {
				hex->GetAdj(k, &f, &x, &y);
				if((i < S_VERTEX_INDEX) && (5 == k))
					continue;
				if((i >= S_VERTEX_INDEX) && (3 == k))
					continue;
				if((hex1 = GetHex(f, x, y)) != NULL) {
					j = k;
					if((f & 1) == 0) {
						j = k - 3;
						if(j < 0)
							j += 6;
					}
					hex1->GetAdj(j, &f1, &x1, &y1);
					if((f1 != VERTEX_FACE_INDEX) && (x1 != i) && 
									(y1 != VERTEX_FACE_INDEX)) 
						fprintf(stderr, "Oh-oh!!!  vertex %d dir %d points to "
										"[%d (%d %d)] and this points (%d) "
										"back to [%d (%d %d)\n", i, k,
										f, x, y, j, f1, x1, y1);
				} else
					fprintf(stderr, 
						"Oh-oh!!!  Vertex %d dir %d points to a NULL hex\n", 
						i, k);
			}
		} else
			fprintf(stderr, "Oh-oh!!!  No vertex at location %d.\n", i);
	}

	for(f = 0;f < MAX_FACES;f++) {
		for(i = 0;i < MAP_FACE_MAX_X;i++) {
			for(j = 0;j < MAP_FACE_MAX_Y;j++) {
				int j,k,f1,x1,y1,f2,x2,y2;
				MapHex *hex,*hex1;

				if((hex = GetHex(f, i, j)) != NULL) {
					for(k = 0;k < 6;k++) {
						hex->GetAdj(k, &f1, &x1, &y1);
						if((hex1 = GetHex(f1, x1, y1)) != NULL) {
							j = k;
							if((f & 1) == (f1 & 1)) {
								j -= 3;
								if(j < 0)
									j += 6;
							}
							hex1->GetAdj(j, &f2, &x2, &y2);
							if((f != f2) || (i != x2) || (j != y2))
								fprintf(stderr, "Oh-oh!!!  hex [%d (%d %d)] dir"
												" %d points to [%d (%d %d)] "
												"which points (%d) back to "
												"[%d (%d %d)]\n",
												f, i, j, k, f1, x1, y1, j,
												f2, x2, y2);
						} else
							fprintf(stderr, "Oh-oh!!!  hex [%d (%d %d)] dir %d"
											" points to a NULL hex.\n",
											f, i, j, k);
					}
				}
			}
		}
	}
#endif
}

IsoMap::~IsoMap()
{
int i;

	for(i = 0;i < MAX_PLATES;i++) {
		if(plates[i] != NULL)
			delete plates[i];
	}
	
	for(i = 0;i < MAX_FACES;i++) {
		if(faces[i] != NULL)
			delete faces[i];
	}

	for(i = 0;i < MAX_VERTICES;i++) {
		if(vertices[i] != NULL)
			delete vertices[i];
	}
}

// ---------------------------------------------------------------------------
unsigned char
IsoMap::GetHexMask(int x, int y)
{
	if((y > ISO_MAP_MAX_Y) || (x > ISO_MAP_MAX_X) || (y < 0) || (x < 0)) {
		return(0);
	}

	return(mask[x % ISO_MAP_SECTION][y]);
}

#if 0
MapHex *
IsoMap::GetHex(int x, int y)
{
int face;

	if((y > ISO_MAP_MAX_Y) || (x > ISO_MAP_MAX_X) || (y < 0) || (x < 0))
		return(NULL);

	if((face = face_index[x % ISO_MAP_SECTION][y]) >= 0) {
		int n_x,n_y;


		return(faces[face]->GetHex(n_x, n_y));
	}

	// fell through
	return(NULL);
}
#endif

MapHex *
IsoMap::GetHex(int face, int x, int y)
{
	if((VERTEX_FACE_INDEX == face) && (VERTEX_FACE_INDEX == y)) {
		return(vertices[x]);
	}
	if((face >= 0) && (face < MAX_FACES)) {
		return(faces[face]->GetHex(x, y));
	}

	return(NULL);
}

// ----------------------------------------------------------------------------
#define TOTAL_HEXES			500
#define TOTAL_HEX_THRESHOLD	(TOTAL_HEXES - 100)
#define HEX_DECREMENT		5
// probably a bad name.....
#define HEX_DIAMETER		35

void
IsoMap::SetPlates(int n)
{
#ifdef DUMP_FACE
int x,y;
int n_x,n_y,f;

	for(y = 0;y < ISO_MAP_MAX_Y;y++) {
		for(x = 0;x < ISO_MAP_MAX_X;x++) {
			if(XYToFaceCoord(x, y, &f, &n_x, &n_y)) {
				fprintf(stderr, "X");
			} else {
				fprintf(stderr, " ");
			}
		}
		fprintf(stderr, "\n");
	}
#endif
//only one (or none) of these:
//#define FIXED_PLATE_CODE
//#define LATITUDE_PLATE_CODE
//#define FACE_PLATE_CODE

// oh, man this one is lame
#if defined LATITUDE_PLATE_CODE
int x,y,f,k,val;

	k = 0;
	ForcePlateNum(VERTEX_FACE_INDEX, N_VERTEX_INDEX, VERTEX_FACE_INDEX, k);
	k++;
	for(y = 0;y < MAP_FACE_MAX_Y;y++) {
		for(x = 0;x < MAP_FACE_MAX_X;x++) {
			for(f = 0;f < MAX_FACES;f++) {
				switch(f & 3) {
					case 0:
						val = k;
						break;
					case 1:
						val = (2 * MAP_FACE_MAX_Y) - k;
						break;
					case 2:
						val = MAP_FACE_MAX_Y + k;
						break;
					case 3:
						val = (3 * MAP_FACE_MAX_Y) - k;
						break;
				}

				ForcePlateNum(f, x, y, val);
			}
		}
		k++;
	}

	for(x = 0;x < S_VERTEX_INDEX;x++) {
		ForcePlateNum(VERTEX_FACE_INDEX, x, VERTEX_FACE_INDEX, MAP_FACE_MAX_Y);
		ForcePlateNum(VERTEX_FACE_INDEX, x + S_VERTEX_INDEX, VERTEX_FACE_INDEX, 
						2 * MAP_FACE_MAX_Y);
	}

	ForcePlateNum(VERTEX_FACE_INDEX, S_VERTEX_INDEX, VERTEX_FACE_INDEX, 
					3 * MAP_FACE_MAX_Y);
#elif defined FIXED_PLATE_CODE
int f,i,x,y,num,tot_hexes;
Dice dice;

int dummy[7][3] = {
		{ 1, 2, 1 },
//		{ 11, 5, 6 },
		{ 2, 5, 6 },
		{ 4, 1, 5 },
		{ 7, 2, 1 },
		{ 12, 2, 3 },
		{ 15, 1, 3 },
		{ 13, 2, 0 }
};

	for(f = 0;f < MAX_FACES;f++) {
		for(x = 0;x < MAP_FACE_MAX_X;x++) {
			for(y = 0;y < MAP_FACE_MAX_Y;y++) {
				ForcePlateNum(f, x, y, -1);
			}
		}
	}

	num = 7;
	tot_hexes = 0;
	for(i = 0;i < num;i++) {
		int j,y,x,f,n_x,n_y;

		f = dummy[i][0];
		n_x = dummy[i][1];
		n_y = dummy[i][2];
		x = n_x;
		y = n_y;
		plates[i] = new TectPlate(dice.Roll(6, 2) * j, dice.Roll(6,1) - 1, 
				f, x, y);
		ForcePlateNum(f, x, y, i);
		tot_hexes += plates[i]->GetNumHexes();
	}
#elif defined FACE_PLATE_CODE
int f,x,y;

	for(f = 0;f < MAX_FACES;f++) {
		for(x = 0;x < MAP_FACE_MAX_X;x++) {
			for(y = 0;y < MAP_FACE_MAX_Y;y++) {
				if(y < 4)
					ForcePlateNum(f, x, y, f + 1);
				else
					ForcePlateNum(f, x, y, 0);
			}
		}
	}

	for(x = 0;x < S_VERTEX_INDEX;x++) {
		ForcePlateNum(VERTEX_FACE_INDEX, x, VERTEX_FACE_INDEX, 0);
		ForcePlateNum(VERTEX_FACE_INDEX, x + S_VERTEX_INDEX, VERTEX_FACE_INDEX, 
						0);
	}

#else
// this is the *real* code
int f,i,x,y,num,tot_hexes;
Dice dice;

	for(f = 0;f < MAX_FACES;f++) {
		for(x = 0;x < MAP_FACE_MAX_X;x++) {
			for(y = 0;y < MAP_FACE_MAX_Y;y++) {
				ForcePlateNum(f, x, y, -1);
			}
		}
	}

	num = n - 1;				// plate 0 counts as a plate
	if(num > MAX_PLATES)
		num = MAX_PLATES;

	tot_hexes = 0;
	for(i = 0;i < num;i++) {
		int j,y,x,f,n_x,n_y;

		switch(dice.Roll(6, 1)) {
			default:
				j = 5;
				break;
			case 2:
				j = 10;
				break;
			case 3:
				j = 15;
				break;
			case 4:
				j = 20;
				break;
			case 5:
				j = 25;
				break;
			case 6:
				j = 30;
				break;
		}

		// try to spread them evenly
//		x = ((HEX_DIAMETER / num) * i) + (num / 2);
		x = i * ((HEX_DIAMETER / num));
		// note:  any particular 'y' only has 14 valid hexes......
		//  but, the N & S vertices complicate things so, make it 12
		y = LowestY(x) + dice.Roll(12, 1);
		XYToFaceCoord(x, y, &f, &n_x, &n_y);
		x = n_x;
		y = n_y;
//fprintf(stderr, "%d: f:%d c(%d %d)\n", i, f, x, y);
		plates[i] = new TectPlate(dice.Roll(6, 2) * j, dice.Roll(6,1) - 1, 
				dice.Roll(10, 1), f, x, y);
		ForcePlateNum(f, x, y, i);
		tot_hexes += plates[i]->GetNumHexes();
	}
#endif

#if !defined LATITUDE_PLATE_CODE && !defined FACE_PLATE_CODE
#ifndef PLATE_GROW_CODE
	GrowPlates();
#endif
	// always one plate, place it at the end....
	plates[i] = new TectPlate(TOTAL_HEXES - tot_hexes, dice.Roll(6, 1) - 1, 
		// XXX arbitrary placement!!!!
		dice.Roll(10, 1), 0, 34, 3);
//		ForcePlateNum(i, 0, 11);
#ifdef PLATE_GROW_CODE
	blank_no = i;
#endif
	// ...and use all the remaining hexes
	for(f = 0;f < MAX_FACES;f++) {
		for(x = 0;x < MAP_FACE_MAX_X;x++) {
			for(y = 0;y < MAP_FACE_MAX_Y;y++) {
				MapHex *mh;

				if(((mh = faces[f]->GetHex(x, y)) != NULL) && 
						(mh->GetPlateNo() < 0)) {
					ForcePlateNum(f, x, y, i);
				}
			}
		}
	}
	for(x = 0;x < MAX_VERTICES;x++) {
		MapHex *mh;

		if(((mh = vertices[x]) != NULL) && (mh->GetPlateNo() < 0)) {
			ForcePlateNum(VERTEX_FACE_INDEX, x, VERTEX_FACE_INDEX, i);
		}
	}

//fprintf(stderr, "%d: f:%d c(%d %d)\n", i, 0, 3, 3);
#endif
}

void
IsoMap::GrowPlates(void)
{
int mag=1;

	while(TRUE) {
		if(!ExpandPlates(mag))
			break;
		mag++;
	}
}

bool
IsoMap::ExpandPlates(int mag)
{
int plate,f,x,y,dir,grown;
Dice dice;

	grown = 0;
	plate = 0;
//fprintf(stderr, "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX\n");
	while(plates[plate] != NULL) {
		plates[plate]->GetCenter(&f, &x, &y);
//fprintf(stderr, "=====================================\n");
//fprintf(stderr, "plate %d: f:%d (%d %d)\n", plate, f, x, y);
		for(dir = 0;dir < 6;dir++) {
			int new_f,new_x,new_y,alt_dir;

			alt_dir = dir;
			if(plates[plate]->CanPlace()) {
//fprintf(stderr, "----> GetNewCoord() %d (%d %d) d:%d m:%d ", f, x, y, alt_dir, mag);
				if(GetNewCoord(f, x, y, mag, 
							&alt_dir, &new_f, &new_x, &new_y)) {
//fprintf(stderr, "[%d (%d %d)] ", new_f, new_x, new_y);
					if(SetPlateNum(new_f, new_x, new_y, plate)) {
//fprintf(stderr, "Placed\n");
						plates[plate]->IncPlacedHexes();
						grown++;
					}
//else fprintf(stderr, "Not Placed\n");
				}
//else fprintf(stderr, "\n");
				if(mag > 1) {
					// backfill
					int tmp_mag,tmp_dir,tmp_f,tmp_x,tmp_y;

					for(tmp_mag = 1;tmp_mag < mag;tmp_mag++) {
//fprintf(stderr, "  --> GetNewCoord() %d (%d %d) d:%d m:%d ", new_f, new_x, new_y, tmp_dir, tmp_mag);
						tmp_dir = alt_dir - 2;		// default
						if(tmp_dir < 0)
							tmp_dir += 6;
						if(tmp_dir >= 6)
							tmp_dir -= 6;
						if(GetNewCoord(new_f, new_x, new_y, tmp_mag, &tmp_dir,
									&tmp_f, &tmp_x, &tmp_y)) {
//fprintf(stderr, "[%d (%d %d)] ", tmp_f, tmp_x, tmp_y);
							if(plates[plate]->CanPlace()) {
								if(SetPlateNum(tmp_f, tmp_x, tmp_y,plate)) {
//fprintf(stderr, "Placed\n");
									plates[plate]->IncPlacedHexes();
									grown++;
								}
//else fprintf(stderr, "Not Placed\n");
							}
						}
//else fprintf(stderr, "\n");
					}
				}
			}
		}
		plate++;
	}

	if(grown)
		return(TRUE);
	return(FALSE);
}

bool
IsoMap::ForcePlateNum(int f, int x, int y, int plate_num)
{
MapHex *mh;

	if((VERTEX_FACE_INDEX == f) && (VERTEX_FACE_INDEX == y)) {
		if(vertices[x] != NULL) {
			vertices[x]->SetPlateNo(plate_num);
			return(TRUE);
		}
	}

	if((mh = faces[f]->GetHex(x, y)) != NULL) {
		mh->SetPlateNo(plate_num);
		return(TRUE);
	}

	return(FALSE);
}

bool
IsoMap::SetPlateNum(int f, int x, int y, int plate_num)
{
MapHex *mh;

//fprintf(stderr, "=============================\n");
//fprintf(stderr, "SetPlateNum() ---->[%d (%d %d)]: set:%d ", f, x, y, plate_num);
	if((VERTEX_FACE_INDEX == f) && (VERTEX_FACE_INDEX == y))
		mh = vertices[x];
	else 
		mh = faces[f]->GetHex(x, y);

	if(mh != NULL) {
//fprintf(stderr, "no:%d", mh->GetPlateNo());
#ifdef PLATE_GROW_CODE
		if(mh->GetPlateNo() == blank_no) {
#else
		if(mh->GetPlateNo() == -1) {
#endif
			int i;

			// don't set unless there is already an adjacent hex with
			//  the same value
			for(i = 0;i < 6;i++) {
				int f1,x1,y1;
				MapHex *mh1;

				if(GetAdjHex(f, x, y, i, &f1, &x1, &y1)) {
					if((VERTEX_FACE_INDEX == f1) && (VERTEX_FACE_INDEX == y1))
						mh1 = vertices[x1];
					else 
						mh1 = faces[f1]->GetHex(x1, y1);
					if(mh1 != NULL) {
//fprintf(stderr, "\n  %d: [%d (%d %d)] no:%d", i, f1, x1, y1, mh1->GetPlateNo());
						if(mh1->GetPlateNo() == plate_num) {
							mh->SetPlateNo(plate_num);
//fprintf(stderr, "OK!!!\n");
							return(TRUE);
						}
					}
				}
			}
		}
	}
//fprintf(stderr, "Failed!!!\n");
	return(FALSE);
}

// ---------------------------------------------------------------------------
void
IsoMap::DoPlateCollision(void)
{
int f,x,y,f1,x1,y1,i;
int plate,p_vel,p_dir,a_plate,a_vel,a_dir;
MapHex *hex,*adj_hex;

	for(f = 0;f < MAX_FACES;f++) {
		for(x = 0;x < MAP_FACE_MAX_X;x++) {
			for(y = 0;y < MAP_FACE_MAX_Y;y++) {
				if((hex = GetHex(f, x, y)) != NULL) {
					plate = hex->GetPlateNo();
					// XXX
					p_dir = plates[plate]->GetDirection();
					p_vel = plates[plate]->GetVelocity();
					for(i = 0;i < 6;i++) {
						if(GetAdjHex(f, x, y, i, &f1, &x1, &y1)) {
							if((adj_hex = GetHex(f1, x1, y1)) != NULL) {
								if((a_plate = adj_hex->GetPlateNo()) != plate) {
									a_dir = plates[a_plate]->GetDirection();
									a_vel = plates[a_plate]->GetVelocity();
									hex->SetAlt(i, p_dir, p_vel, a_dir, a_vel);
								}
							}
						}
					}
				}
			}
		}
	}
}

// ---------------------------------------------------------------------------
int
IsoMap::GetFaceIndex(int x, int y)
{
int face;

	if((y > ISO_MAP_MAX_Y) || (x > ISO_MAP_MAX_X) || (y < 0) || (x < 0))
		return(-1);

	if((face = face_index[x % ISO_MAP_SECTION][y]) >= 0) {
		face += (x / ISO_MAP_SECTION) * 4;
		return(face);
	}

	// fell through
	return(-1);
}

bool
IsoMap::XYToFaceCoord(int x, int y, int *f, int *n_x, int *n_y)
{
int max_y;

//fprintf(stderr, "(%d %d) ", x, y);
	if((y > ISO_MAP_MAX_Y) || (x > ISO_MAP_MAX_X) || (y < 0) || (x < 0)) {
//fprintf(stderr, "Bad xy!!!\n");
		return(FALSE);
	}

	if(((*f) = face_index[x % ISO_MAP_SECTION][y]) >= 0) {
		(*f) += (x / ISO_MAP_SECTION) * 4;
		if((*f) >= MAX_FACES)
			(*f) -= MAX_FACES;

		if((*f) & 1)
			max_y = MAP_FACE_MAX_Y - 2;
		else
			max_y = MAP_FACE_MAX_Y - 1;

		switch((*f) & 3) {
			case 0:
				(*n_x) = ((x + 3) % ISO_MAP_SECTION);
				(*n_y) = y - 1;
				break;
			case 1:
				(*n_x) = MAP_FACE_MAX_X - ((x + 3) % ISO_MAP_SECTION) - 1;
				(*n_y) = max_y - (y - 8);
				if(((*n_y) & 1) == 0)
					(*n_x)--;
				break;
			case 2:
				(*n_x) = (x % ISO_MAP_SECTION);
				(*n_y) = y - 8;
				if(((*n_y) & 1) == 0)
					(*n_x)--;
				break;
			case 3:
				(*n_x) = MAP_FACE_MAX_X - (x % ISO_MAP_SECTION) - 1;
				(*n_y) = max_y - (y - 15);
				break;
		}
		// XXX vertex check
		if((*n_x) >= MAP_FACE_MAX_X)
			(*n_x) -= MAP_FACE_MAX_X;
		if((*n_x) < 0)
			(*n_x) += MAP_FACE_MAX_X;
//fprintf(stderr, "%d (%d %d)\n", *f, *n_x, *n_y);
		return(TRUE);
	}

//fprintf(stderr, "Failed!!!\n");
	// fell through
	return(FALSE);
}

int 
IsoMap::LowestY(int x)
{
int i=0;

	while(face_index[x % ISO_MAP_SECTION][i] < 0) {
		i++;
	}

	return(i);
}

// set up adjacent hexes to speed up future adjacency checks.
//  yes, a pointer to the hex would have been another way to
//  do this, but I believe there is a case the latitude is
//  needed.
// NOTE:  the displayed half vertex is #11
void
IsoMap::LinkHex(int f, int x, int y)
{
int i;
MapHex *hex;
struct MAP_COORD links[6];

	if((hex = GetHex(f, x, y)) != NULL) {
		// trap vertices and poles
		if((VERTEX_FACE_INDEX == f) && (VERTEX_FACE_INDEX == y)) {
			if(N_VERTEX_INDEX == x) {
				links[0].f = 12;
				links[1].f = 8;
				links[2].f = 4;
				links[3].f = 0;
				links[4].f = 16;
				links[5].f = 12;
				links[0].x = 2;
				links[1].x = 2;
				links[2].x = 2;
				links[3].x = 2;
				links[4].x = 2;
				links[5].x = 2;
				links[0].y = 0;
				links[1].y = 0;
				links[2].y = 0;
				links[3].y = 0;
				links[4].y = 0;
				links[5].y = 0;
			} else if(S_VERTEX_INDEX == x) {
				links[0].f = 3;
				links[1].f = 7;
				links[2].f = 11;
				links[3].f = 11;
				links[4].f = 15;
				links[5].f = 19;
				links[0].x = 2;
				links[1].x = 2;
				links[2].x = 2;
				links[3].x = 2;
				links[4].x = 2;
				links[5].x = 2;
				links[0].y = 0;
				links[1].y = 0;
				links[2].y = 0;
				links[3].y = 0;
				links[4].y = 0;
				links[5].y = 0;
			} else if(x < S_VERTEX_INDEX) {
				links[0].f = 4;
				links[1].f = 4;
				links[2].f = 5;
				links[3].f = 2;
				links[4].f = 0;
				links[5].f = 4;
				links[0].x = 0;
				links[1].x = 0;
				links[2].x = 5;
				links[3].x = 2;
				links[4].x = 5;
				links[5].x = 0;
				links[0].y = 5;
				links[1].y = 6;
				links[2].y = 5;
				links[3].y = 0;
				links[4].y = 6;
				links[5].y = 5;
				for(i = 0;i < 6;i++) {
					links[i].f += ((x - 1) * 4);
					if((5 == x) && ((i < 3) || (i > 4)))
						links[i].f -= MAX_FACES;
				}
			} else {
				links[0].f = 6 + ((x - 7) * 4);
				links[1].f = 6 + ((x - 7) * 4);
				links[2].f = 7 + ((x - 7) * 4);
				links[3].f = 7 + ((x - 7) * 4);
				links[4].f = 2 + ((x - 7) * 4);
				links[5].f = 5 + ((x - 7) * 4);
				links[0].x = 0;
				links[1].x = 0;
				links[2].x = 5;
				links[3].x = 5;
				links[4].x = 5;
				links[5].x = 2;
				links[0].y = 5;
				links[1].y = 6;
				links[2].y = 5;
				links[3].y = 5;
				links[4].y = 6;
				links[5].y = 0;
			}
		} else {
			// do a face hex
			//  NOTE:  many bulk adjustments are made up front,
			//         and cleaned up further along.  Arguable
			//         which is faster, but since this is a one
			//         time function, don't worry about it.

			// bulk settings
			links[0].f = f;
			links[1].f = f;
			links[2].f = f;
			links[3].f = f;
			links[4].f = f;
			links[5].f = f;
			if(y & 1) {
				links[0].x = x;
				links[1].x = x + 1;
				links[2].x = x;
				links[3].x = x - 1;
				links[4].x = x - 1;
				links[5].x = x - 1;
			} else {
				links[0].x = x + 1;
				links[1].x = x + 1;
				links[2].x = x + 1;
				links[3].x = x;
				links[4].x = x - 1;
				links[5].x = x;
			}
			links[0].y = y - 1;
			links[1].y = y;
			links[2].y = y + 1;
			links[3].y = y + 1;
			links[4].y = y;
			links[5].y = y - 1;

			// bottom edge check
			if(((f & 1) == 0) && (6 == y)) {
				links[2].f = f + 1;
				links[3].f = f + 1;
				links[2].x = 4 - x;
				links[3].x = 5 - x;
				links[2].y = 5;
				links[3].y = 5;
			} else if(((f & 1) == 1) && (5 == y)) {
				links[2].f = f - 1;
				links[3].f = f - 1;
				links[2].x = 4 - x;
				links[3].x = 5 - x;
				links[2].y = 6;
				links[3].y = 6;
			}

			// other edge checks
			switch(f & 3) {
				case 0:
					if(((((2 == x) && ((0 == y) || (1 == y))))) ||
						 ((((1 == x) && ((2 == y) || (3 == y))))) ||
						 ((((0 == x) && ((4 == y) || (5 == y)))))) {
					// left edge check
						links[4].f = f - 4;
						links[5].f = f - 4;
						links[4].x = 5 - x;
						links[5].x = x + y;
						links[4].y = y + 1;
						links[5].y = y;
					} 
					if(((((2 == x) && (0 == y)))) ||
						 ((((3 == x) && ((1 == y) || (2 == y))))) ||
						 ((((4 == x) && ((3 == y) || (4 == y))))) ||
						 ((((5 == x) && ((5 == y) || (6 == y)))))) {
					// right edge check
						links[0].f = f + 4;
						links[1].f = f + 4;
						links[0].x = 5 - x;
						links[1].x = x - y;
						links[0].y = y - 1;
						links[1].y = y;
					}
					// special corner and vertice checks
					if((2 == x) && (0 == y)) {
						links[0].f = VERTEX_FACE_INDEX;
						links[0].x = N_VERTEX_INDEX;
						links[0].y = VERTEX_FACE_INDEX;
						links[1].f = f + 4;
						links[1].x = x;
						links[1].y = y;
					} else if(0 == x) {
						if(5 == y) {
							links[3].f = VERTEX_FACE_INDEX;
							if(0 == f)
								links[3].x = 5;
							else
								links[3].x = N_VERTEX_INDEX + (f / 4);
							links[3].y = VERTEX_FACE_INDEX;
						} else if(6 == y) {
							links[4].f = VERTEX_FACE_INDEX;
							links[4].x = N_VERTEX_INDEX + (f / 4);
							links[4].y = VERTEX_FACE_INDEX;
						}
					} else if((5 == x) && (6 == y)) {
						links[1].f = VERTEX_FACE_INDEX;
						if(0 == f)
							links[1].x = N_VERTEX_INDEX + 1;
						else
							links[1].x = N_VERTEX_INDEX + (f / 4) + 1;
						links[1].y = VERTEX_FACE_INDEX;
						links[2].f = f + 2;
						links[2].x = 2;
						links[2].y = 0;
					}
					break;
				case 1:
					if(((((2 == x) && ((0 == y) || (1 == y))))) ||
						 ((((1 == x) && ((2 == y) || (3 == y))))) ||
						 ((((0 == x) && ((4 == y) || (5 == y)))))) {
					// left edge check
						links[4].f = f + 1;
						links[5].f = f + 1;
						links[4].x = 2 - x;
						links[5].x = y + x - 3;
						links[4].y = 5 - y;
						links[5].y = 6 - y;
					} 
					if(((((2 == x) && (0 == y)))) ||
						 ((((3 == x) && ((1 == y) || (2 == y))))) ||
						 ((((4 == x) && ((3 == y) || (4 == y))))) ||
						 ((((5 == x) && ((5 == y)))))) {
					// right edge check
						links[0].f = f - 3;
						links[1].f = f - 3;
						links[0].x = x - y + 3;
						links[1].x = 7 - x;
						links[0].y = 6 - y;
						links[1].y = 5 - y;
					}
					// special corner and vertice checks
					if((2 == x) && (0 == y)) {
						links[5].f = VERTEX_FACE_INDEX;
						if(1 == f)
							links[5].x = MAX_VERTICES - 1;
						else
							links[5].x = S_VERTEX_INDEX + (f / 4);
						links[5].y = VERTEX_FACE_INDEX;
					} else if((5 == x) && (5 == y)) {
						links[2].f = VERTEX_FACE_INDEX;
						links[2].x = N_VERTEX_INDEX + (f / 4);
						links[2].y = VERTEX_FACE_INDEX;
					}
					break;
				case 2:
					if(((((2 == x) && ((0 == y) || (1 == y))))) ||
						 ((((1 == x) && ((2 == y) || (3 == y))))) ||
						 ((((0 == x) && ((4 == y) || (5 == y)))))) {
					// left edge check
						links[4].f = f - 1;
						links[5].f = f - 1;
						links[4].x = 2 - x;
						links[5].x = y + x - 3;
						links[4].y = 5 - y;
						links[5].y = 6 - y;
					} 
					if(((((2 == x) && (0 == y)))) ||
						 ((((3 == x) && ((1 == y) || (2 == y))))) ||
						 ((((4 == x) && ((3 == y) || (4 == y))))) ||
						 ((((5 == x) && ((5 == y) || (6 == y)))))) {
					// right edge check
						links[0].f = f + 3;
						links[1].f = f + 3;
						links[0].x = 3 + (x - y);
						links[1].x = 7 - x;
						links[0].y = 6 - y;
						links[1].y = 5 - y;
					}
					// special corner and vertice checks
					if((2 == x) && (0 == y)) {
						links[0].f = VERTEX_FACE_INDEX;
							links[0].x = N_VERTEX_INDEX + (f / 4);
						links[0].y = VERTEX_FACE_INDEX;
						links[5].f = f - 2;
						links[5].x = 5;
						links[5].y = 6;
					} else if(0 == x) {
						if(5 == y) {
							links[3].f = VERTEX_FACE_INDEX;
							if(2 == f)
								links[3].x = MAX_VERTICES - 1;
							else
								links[3].x = S_VERTEX_INDEX + (f / 4);
							links[3].y = VERTEX_FACE_INDEX;
						} else if(6 == y) {
							links[4].f = VERTEX_FACE_INDEX;
							if(2 == f)
								links[4].x = MAX_VERTICES - 1;
							else
								links[4].x = S_VERTEX_INDEX + (f / 4);
							links[4].y = VERTEX_FACE_INDEX;
						}
					} else if((5 == x) && (6 == y)) {
						links[1].f = VERTEX_FACE_INDEX;
						links[1].x = S_VERTEX_INDEX + (f / 4) + 1;
						links[1].y = VERTEX_FACE_INDEX;
						links[2].f = f + 5;
						links[2].x = 5;
						links[2].y = 5;
					}
					break;
				case 3:
					// left edge check
					if(((((2 == x) && ((0 == y) || (1 == y))))) ||
						 ((((1 == x) && ((2 == y) || (3 == y))))) ||
						 ((((0 == x) && ((4 == y) || (5 == y)))))) {
						links[4].f = f + 4;
						links[5].f = f + 4;
						links[4].x = x + y;
						links[5].x = 4 - x;
						links[4].y = y;
						links[5].y = y - 1;
					} 
					if(((((2 == x) && (0 == y)))) ||
						 ((((3 == x) && ((1 == y) || (2 == y))))) ||
						 ((((4 == x) && ((3 == y) || (4 == y))))) ||
						 ((((5 == x) && ((5 == y)))))) {
					// right edge check
						links[0].f = f - 4;
						links[1].f = f - 4;
						links[0].x = x - y;
						links[1].x = 4 - x;
						links[0].y = y;
						links[1].y = y + 1;
					}
					// special corner and vertice checks
					if((2 == x) && (0 == y)) {
						links[4].f = f + 4;
						links[4].x = x;
						links[4].y = y;
						links[5].f = VERTEX_FACE_INDEX;
						links[5].x = S_VERTEX_INDEX;
						links[5].y = VERTEX_FACE_INDEX;
					} else if((5 == x) && (5 == y)) {
						links[1].f = f - 5;
						links[1].x = 5;
						links[1].y = 6;
						links[2].f = VERTEX_FACE_INDEX;
						if(3 == f)
							links[2].x = MAX_VERTICES - 1;
						else
							links[2].x = S_VERTEX_INDEX + (f / 4);
						links[2].y = VERTEX_FACE_INDEX;
					}
					break;
			}

		}

		// final sanity checks
		for(i = 0;i < 6;i++) {
			if((VERTEX_FACE_INDEX != links[i].f) &&
				(VERTEX_FACE_INDEX != links[i].y)) {
				if(links[i].f >= MAX_FACES)
					links[i].f -= MAX_FACES;
				if(links[i].f < 0)
					links[i].f += MAX_FACES;
			}
		}

		for(i = 0;i < 6;i++) {
#ifdef HEX_LINK_CHECK
			// basic error check....
			//   better ones in caller, but this doesn't hurt
			if(GetHex((int) links[i].f, (int) links[i].x, (int) links[i].y) == 
							NULL)
				fprintf(stderr, "Oh-oh!!! hex [%d (%d %d)] has bad "
								"link in dir %d [%d (%d %d)]\n",
						f, x, y, i, links[i].f, links[i].x, links[i].y);
#endif

			hex->SetAdj(i, (int)links[i].f, (int)links[i].x, (int)links[i].y);
		}
	}
}

bool
IsoMap::GetAdjHex(int f, int x, int y, int dir,
			int *new_f, int *new_x, int *new_y)
{
MapHex *hex;

	if((hex = GetHex(f, x, y)) != NULL) {
		return(hex->GetAdj(dir, new_f, new_x, new_y));
	}

	return(FALSE);
}

bool 
IsoMap::GetNewCoord(int f, int x, int y, int mag, int *base_dir,
			int *new_f, int *new_x, int *new_y)
{
// XXX should use the face/hex functions.....
int f1,x1,y1,m1,dir,alt_dir;

	m1 = mag;
	dir = *base_dir;
//fprintf(stderr, "=========================================\n");
//fprintf(stderr, "GetNewCoord() [%d (%d %d)] d:%d m:%d ", f, x, y, dir, mag);
	while(m1) {
		if(!GetAdjHex(f, x, y, dir, &f1, &x1, &y1)) {
//fprintf(stderr, "Failed! (%d)\n", m1);
			return(FALSE);
		}

		alt_dir = 0;
		if(f != f1) {
			if(VERTEX_FACE_INDEX == f) {
				if(f < S_VERTEX_INDEX) {
					if(f1 & 1)
						alt_dir = 3;
				} else {
					if((f1 & 1) == 0) 
						alt_dir = 3;
				}
			} else if (VERTEX_FACE_INDEX == f1) {
				if(f1 < S_VERTEX_INDEX) {
					if(f & 1)
						alt_dir = 3;
				} else {
					if((f & 1) == 0) 
						alt_dir = 3;
				}
			} else if((f & 1) != (f1 & 1))
				alt_dir = 3;
			else if(((f & 3) == 0) && ((f1 & 3) == 0)) {
				if(((f - f1) == 4) || ((f - f1) == -16))
					alt_dir = -1;
				else
					alt_dir = 1;
			} else if(((f & 3) == 3) && ((f1 & 3) == 3)) {
				if(((f - f1) == 4) || ((f - f1) == -16))
					alt_dir = 1;
				else
					alt_dir = -1;
			}
		}

		dir += alt_dir;
		if(dir < 0)
			dir += 6;
		if(dir >= 6)
			dir -= 6;
		f = f1;
		x = x1;
		y = y1;

//fprintf(stderr, "[%d (%d %d)] a:%d d:%d ", f1, x1, y1, alt_dir, dir);
		m1--;
	}

	*(new_f) = f1;
	*(new_x) = x1;
	*(new_y) = y1;
	*(base_dir) = dir;
//fprintf(stderr, "OK\n");
	return(TRUE);
}

bool 
IsoMap::GetRelativeCoord(int f, int x, int y, int d, 
		int *n_f, int *n_x, int *n_y, int *n_d)
{
	(*n_d) = d;
	switch(f % 4) {
		case 0:
			switch(d) {
				case 0:
				case 2:
				case 4:
					// XXX vertices
					(*n_f) = -99;
					break;
				case 1:
					(*n_f) = f + 4;
					(*n_d) = 2;
					break;
				case 3:
					(*n_f) = f + 1;
					(*n_x) = MAP_FACE_MAX_X - x;
					(*n_y) = MAP_FACE_MAX_Y - y;
					(*n_d) = 0;
					break;
				case 5:
					(*n_f) = f - 4;
					(*n_d) = 4;
					break;
			}
			break;
		case 1:
			switch(d) {
				case 0:
				case 2:
				case 4:
					// XXX vertices
					(*n_f) = -99;
					break;
				case 1:
					(*n_f) = f - 3;
					(*n_x) = MAP_FACE_MAX_X - x;
					(*n_y) = MAP_FACE_MAX_Y - y;
					(*n_d) = 5;
					break;
				case 3:
					(*n_f) = f - 1;
					(*n_x) = MAP_FACE_MAX_X - x;
					(*n_y) = MAP_FACE_MAX_Y - y;
					(*n_d) = 0;
					break;
				case 5:
					(*n_f) = f + 1;
					(*n_x) = MAP_FACE_MAX_X - x;
					(*n_y) = MAP_FACE_MAX_Y - y;
					(*n_d) = 1;
					break;
			}
			break;
		case 2:
			switch(d) {
				case 0:
				case 2:
				case 4:
					// XXX vertices
					(*n_f) = -99;
					break;
				case 1:
					(*n_f) = f + 3;
					(*n_x) = MAP_FACE_MAX_X - x;
					(*n_y) = MAP_FACE_MAX_Y - y;
					(*n_d) = 5;
					break;
				case 3:
					(*n_f) = f + 1;
					(*n_x) = MAP_FACE_MAX_X - x;
					(*n_y) = MAP_FACE_MAX_Y - y;
					(*n_d) = 0;
					break;
				case 5:
					(*n_f) = f - 1;
					(*n_x) = MAP_FACE_MAX_X - x;
					(*n_y) = MAP_FACE_MAX_Y - y;
					(*n_d) = 1;
					break;
			}
			break;
		case 3:
			switch(d) {
				case 0:
				case 2:
				case 4:
					// XXX vertices
					(*n_f) = -99;
					break;
				case 1:
					(*n_f) = f - 4;
					(*n_d) = 2;
					break;
				case 3:
					(*n_f) = f - 1;
					(*n_x) = MAP_FACE_MAX_X - x;
					(*n_y) = MAP_FACE_MAX_Y - y;
					(*n_d) = 0;
					break;
				case 5:
					(*n_f) = f + 4;
					(*n_d) = 4;
					break;
			}
			break;
	}
	if((*n_f) > -99) {	//XXX
		if((*n_f) >= MAX_FACES)
			(*n_f) -= MAX_FACES;
		if((*n_f) < 0)
			(*n_f) += MAX_FACES;
		return(TRUE);
	}

	return(FALSE);
}

// ---------------------------------------------------------------------------
// given face:x,y axis and rotation, gives absolute x, y positoning
bool
IsoMap::XlatCoords(int b_f, int b_x, int b_y, int axis, int rot, 
			int *n_x, int *n_y)
{
int adj_axis,rot_fac,adj_face;
int hex_xlat[2];
//#define DEBUG_ISO_XLAT
//#define STUPID_DEBUG_CODE
#ifdef STUPID_DEBUG_CODE
int i,j,k;
static bool flag=FALSE;
	
	if(!flag) {
		flag = TRUE;
		for(k = 0;k < 6;k++) {
			for(i = 0;i < 7;i++) {
				for(j = 0;j < 7;j++) {
					fprintf(stderr, "{");
					if(hex_xlat[k][i][j][0] < 0)
						fprintf(stderr, "%d,", hex_xlat[k][i][j][0]);
					else
						fprintf(stderr, " %d,", hex_xlat[k][i][j][0]);
					if(hex_xlat[k][i][j][1] < 0)
						fprintf(stderr, "%d", hex_xlat[k][i][j][1]);
					else
						fprintf(stderr, " %d", hex_xlat[k][i][j][1]);
					fprintf(stderr, "}, ");
				}
				fprintf(stderr, "\n");
			}
			fprintf(stderr, "\n");
		}
	}
#endif

	if((VERTEX_FACE_INDEX == b_f) && (VERTEX_FACE_INDEX == b_y)) {
		if(b_x >= MAX_VERTICES)
			return(FALSE);

		(*n_x) = vertex_xlat[axis][b_x][0];
		(*n_y) = vertex_xlat[axis][b_x][1];

		if((b_x != N_VERTEX_INDEX) && (b_x != N_VERTEX_INDEX)) {
			*n_x += ((rot) * 7);
			if(*n_x < 0)
				(*n_x) += (ISO_MAP_MAX_X);
			if(*n_x >= ISO_MAP_MAX_X)
				(*n_x) -= (ISO_MAP_MAX_X);
		}

//fprintf(stderr, "XlatCoords (%d (%d %d) a:%d r:%d --> (%d %d)\n", b_f, b_x, b_y, axis, rot, *n_x, *n_y);
		return(TRUE);
	}

	if((b_f < 0) || (b_f >= MAX_FACES) || 
		(b_x < 0) || (b_x >= MAP_FACE_MAX_X) ||
		(b_y < 0) || (b_y >= MAP_FACE_MAX_Y)) {
			return(FALSE);
	}

//if((b_f & 1) == 0) return(FALSE);

	adj_axis = axis;
	adj_face = face_xlat[b_f][adj_axis];
	rot_fac = rotation[b_f][adj_axis];
	if(b_f & 1) {
		hex_xlat[0] = hex_xlat_odd[rot_fac][b_y][b_x][0];
		hex_xlat[1] = hex_xlat_odd[rot_fac][b_y][b_x][1];
	} else {
		hex_xlat[0] = hex_xlat_even[rot_fac][b_y][b_x][0];
		hex_xlat[1] = hex_xlat_even[rot_fac][b_y][b_x][1];
	}

#ifdef DEBUG_ISO_XLAT
fprintf(stderr, "%d: (%d %d) a:%d r:%d ---> nf: %d aa:%d rf:%d fb:(%d %d) ",
		b_f, b_x, b_y, axis, rot, adj_face, adj_axis, rot_fac,
		face_base[adj_face][0], face_base[adj_face][1]);
#endif
#ifdef DEBUG_ISO_XLAT
fprintf(stderr, "[%d][%d][%d]:(%d,%d) ", rot_fac, b_y, b_x,
		hex_xlat[0], hex_xlat[1]);
#endif
		*n_x = face_base[adj_face][1] + hex_xlat[0];
		*n_y = face_base[adj_face][0] + hex_xlat[1];

	if(rot)
		*n_x += ((rot) * 7);

	if((hex_xlat[0] < 0) && (hex_xlat[1] < 0)) {
#ifdef DEBUG_ISO_XLAT
fprintf(stderr, "Failed!!!!\n");
#endif
		return(FALSE);
	}

	if((adj_face & 3) > 1) {
		if(((*n_y) & 1) == 0) {
			(*n_x)++;
		}
	}

	if(*n_x < 0)
		(*n_x) += (ISO_MAP_MAX_X);
	if(*n_x >= ISO_MAP_MAX_X)
		(*n_x) -= (ISO_MAP_MAX_X);
#ifdef DEBUG_ISO_XLAT
fprintf(stderr, "---> (%d %d)\n", *n_x, *n_y);
#endif
	return(TRUE);
}

unsigned char 
IsoMap::GetRotation(int face, int rot)
{
	if((face >= 0) && (face <  MAX_FACES) && (rot >= 0) && (rot < 6))
		return(rotation[face][rot]);
	return(0);
}

bool
IsoMap::CheckPair(int x, int y, int *new_x)
{
int offset;

	if((offset = (int)half_hex[x % ISO_MAP_SECTION][y]) != 0) {
//fprintf(stderr, "----> (%d %d) o:%d ", x, y, offset);
		x += offset;
		if(x >= ISO_MAP_MAX_X)
			x -= ISO_MAP_MAX_X;
		if(x < 0)
			x += ISO_MAP_MAX_X;
//fprintf(stderr, "n_x:%d\n", x);
		*new_x = x;
		return(TRUE);
	}

	return(FALSE);
}

// ----------------------------------------------------------------------------
TectPlate *
IsoMap::GetPlate(int ndx)
{
	if((ndx < 0) || (ndx >= MAX_PLATES))
		return(NULL);
	return(plates[ndx]);
}

// ============================================================================
TectPlate::TectPlate(int size, int dir, int v, int f, int x, int y)
{
	direction = dir;
	velocity = v;
	num_hexes = size;
	placed_hexes = 0;
	face = f;
	x_cent = x;
	y_cent = y;
}

TectPlate::~TectPlate()
{
}

void
TectPlate::GetCenter(int *f, int *x, int *y)
{
	*f = face;
	*x = x_cent;
	*y = y_cent;
}

bool
TectPlate::IncPlacedHexes(void)
{
	if(CanPlace()) {
		placed_hexes++;
		return(TRUE);
	}

	return(FALSE);
}

#ifdef DEBUG_PLATES
void
TectPlate::DumpPlate(void)
{
	fprintf(stderr, "f:(%d %d) d: %d  n:%d(%d)\n", 
		face, x_cent, y_cent, direction, num_hexes, placed_hexes);
}
#endif


