//
// File:		uwpgen.cpp
// Purpose:		common star generation
// Author:		Mark A. Nordstrand
// Created:
// Updated:	
// Copyright:	LGPL
// Traveller is a registered trademark of Far Future Enterprises.
// Portions based upon material Copyright 1977-2002 Far Future Enterprises.
//
// rcsid[] = "$RCSfile: uwpgen.cpp,v $ $Revision: 1.1 $ $Author: man $ $Date: 2002/04/13 13:56:44 $"

#include <stdio.h>
#include <stdlib.h>
#include "uwpgen.h"
#include "parse.h"
#include "str_util.h"

#ifndef TRUE
#define TRUE		1
#endif

UWPGen::UWPGen(TECH_GEN tg, int n, int x)
{
	tech_gen = new GenTech(tg, n, x);
}

UWPGen::~UWPGen()
{
	if(tech_gen != NULL)
		delete tech_gen;
}

// ---------------------------------------------------------------------------
// 1st pass for system generation
//  generate:  size, atmos, hydro, pop
//  port is ' ' all others 0
//  NOTE:  rely on p_uwp containing "L*a*rge GG" or "S*m*all GG" for GG 
//   detection.
void
UWPGen::GenSystem1(int orb, int zone, char *p_uwp, char *uwp, bool type_m_star)
{
int i,dm;

	ClearUWP(uwp);

	uwp[UWPI_PORT] = ' ';
	if((p_uwp != NULL) && (p_uwp[UWPI_PORT] != ' ') &&
			(p_uwp[UWPI_PORT] != 0)) {
		if('m' == p_uwp[1])
			i = dice.Roll(6, 2) - 4;
		else if('a' == p_uwp[1])
			i = dice.Roll(6, 2) - 6;
		else {
			if(p_uwp[UWPI_SIZE] > 'S') {
				dm = 0;
			} else if(p_uwp[UWPI_SIZE] > '9') {
				dm = p_uwp[UWPI_SIZE] - 'A' + 10;
			} else
				dm = p_uwp[UWPI_SIZE] - '0';
			i = dm - dice.Roll(6, 1);
		}
		uwp[UWPI_SIZE] = SizeConv(i, TRUE);
	} else {
		i = dice.Roll(6, 2);
		if(orb < 1)
			i -= 5;
		else if(orb < 2)
			i -= 4;
		else if(orb < 3)
			i -= 2;
		if(type_m_star)
			i -= 2;
		uwp[UWPI_SIZE] = SizeConv(i, FALSE);
	}

	if(i <= 0) {
		uwp[UWPI_ATMOS] = uwp[UWPI_HYDRO] = '0';
	} else {
		dm = i;
		i = dice.Roll(6, 2);
		if((zone > 1) && (i > 11)) {
			uwp[UWPI_ATMOS] = 'A';
		} else {
			i -= 7;
			i += dm;
			if(zone < 0)
				i -= 2;
			else if(zone > 0)
				i -= 4;
			uwp[UWPI_ATMOS] = SimpleConv(i);
		}

		if(zone < 0)
			uwp[UWPI_HYDRO] = '0';
		else {
			if(i < 0)
				dm = 0;
			else
				dm = i;
			if(zone > 0)
				dm -= 2;
			if((uwp[UWPI_ATMOS] < '2') || (uwp[UWPI_ATMOS] > '9'))
				dm -= 4;
			i = dice.Roll(6, 2) - 7 + dm;
			uwp[UWPI_HYDRO] = SimpleConv(i);
		}
	}

	uwp[UWPI_POP] = GenPop(zone, uwp);
}

// second pass system generation
//  create a main world
void
UWPGen::GenSystem2(char p, char *b, char *uwp)
{
	ClearUWP(uwp);

	if((p != 0) && (strchr("ABCDEX", p) != NULL))
		uwp[UWPI_PORT] = p;
	else 
		uwp[UWPI_PORT] = GenPort();
	uwp[UWPI_SIZE] = b[UWPI_SIZE];
	uwp[UWPI_ATMOS] = b[UWPI_ATMOS];
	uwp[UWPI_HYDRO] = b[UWPI_HYDRO];
	uwp[UWPI_POP] = b[UWPI_POP];
	uwp[UWPI_GOVT] = GenGovt(uwp);
	uwp[UWPI_LAW] = GenLaw(uwp);
	uwp[UWPI_TECH] = GenerateTech(uwp, 0, TRUE);
}

// third pass system generation
//  fill in port, govt, law, and tech
void
UWPGen::GenSystem3(char *m_uwp, char *b, char *uwp, bool b_pres)
{
int i,dm;

	ClearUWP(uwp);

	uwp[UWPI_SIZE] = b[UWPI_SIZE];
	uwp[UWPI_ATMOS] = b[UWPI_ATMOS];
	uwp[UWPI_HYDRO] = b[UWPI_HYDRO];
	uwp[UWPI_POP] = b[UWPI_POP];
	if(' ' == uwp[UWPI_POP]) {
		uwp[UWPI_GOVT] = uwp[UWPI_LAW] = uwp[UWPI_TECH] = ' ';
	} else {
		dm = 0;
		if(m_uwp[UWPI_POP] > '6')
			dm += 1;
		else if('6' == m_uwp[UWPI_POP]) {
			if(uwp[UWPI_POP] > '9')
				dm = uwp[UWPI_POP] - 'A' + 10;
			else
				dm = uwp[UWPI_POP] - '0';
		}
		i = dice.Roll(6, 1) + dm - 1;
		if(i < 0)
			uwp[UWPI_GOVT] = '0';
		else if(i > 3)
			uwp[UWPI_GOVT] = '6';
		else
			uwp[UWPI_GOVT] = i + '0';
		
		if(m_uwp[UWPI_LAW] > '9')
			dm = m_uwp[UWPI_LAW] - 'A' + 10;
		else
			dm = m_uwp[UWPI_LAW] - '0';
		i = dice.Roll(6, 1) - 3 + dm;
		uwp[UWPI_LAW] = SimpleConv(i);

		i = 0;
		if(b_pres)
			i++;
		uwp[UWPI_TECH] = GenerateTech(m_uwp, i, FALSE);
	}

	dm = 0;
	if(uwp[UWPI_POP] < '0')
		dm -= 4;
	else if(uwp[UWPI_POP] < '1')
		dm -= 3;
	else if(uwp[UWPI_POP] < '2')
		dm -= 2;
	else if(uwp[UWPI_POP] > '5')
		dm += 2;
	i = dice.Roll(6, 1) + dm;
	if(i < 3)
		uwp[UWPI_PORT] = 'Y';
	else if(i < 4)
		uwp[UWPI_PORT] = 'H';
	else if(i < 6)
		uwp[UWPI_PORT] = 'G';
	else
		uwp[UWPI_PORT] = 'F';
}

// standard main world generation
void
UWPGen::GenUWP(char p, char *uwp)
{
int i,dm;

	ClearUWP(uwp);

	if((p != 0) && (strchr("ABCDEX", p) != NULL))
		uwp[UWPI_PORT] = p;
	else 
		uwp[UWPI_PORT] = GenPort();

	i = dice.Roll(6, 2) - 2;
	uwp[UWPI_SIZE] = SimpleConv(i);
	if(i < 1) {
		uwp[UWPI_ATMOS] = uwp[UWPI_HYDRO] = '0';
	} else {
		dm = i;
		i = dice.Roll(6, 2) - 7 + dm;
		uwp[UWPI_ATMOS] = SimpleConv(i);
		if((i < 2) && (i > 9))
			dm -= 4;
		i = dice.Roll(6, 2) - 7 + dm;
		uwp[UWPI_HYDRO] = SimpleConv(i);
	}

	i = dice.Roll(6, 2) - 2;
	uwp[UWPI_POP] = SimpleConv(i);

	uwp[UWPI_GOVT] = GenGovt(uwp);
	uwp[UWPI_LAW] = GenLaw(uwp);
	uwp[UWPI_TECH] = GenerateTech(uwp, 0, TRUE);
}

// ---------------------------------------------------------------------------
//  misc common generation functions
char
UWPGen::GenPort(void)
{
int i;

	i = dice.Roll(6, 2);
	if(i < 5)
		return('A');
	else if(i < 7)
		return('B');
	else if(i < 9)
		return('C');
	else if(i < 10)
		return('D');
	else if(i < 12)
		return('E');
	return('X');
}

char
UWPGen::GenGovt(char *uwp)
{
int i,dm;

	if(uwp[UWPI_POP] > '9')
		dm = uwp[UWPI_POP] - 'A' + 10;
	else
		dm = uwp[UWPI_POP] - '0';
	i = dice.Roll(6, 2) - 7 + dm;
	return(SimpleConv(i));
}

char
UWPGen::GenLaw(char *uwp)
{
int i,dm;

	if(uwp[UWPI_GOVT] > '9')
		dm = uwp[UWPI_GOVT] - 'A' + 10;
	else
		dm = uwp[UWPI_GOVT] - '0';
	i = dice.Roll(6, 2) - 7 + dm;
	return(SimpleConv(i));
}

char
UWPGen::GenerateTech(char *uwp, int ext_dm, bool mw)
{
	return(tech_gen->Generate(uwp, ext_dm, mw));
}

char
UWPGen::GenPop(int zone, char *uwp)
{
int dm,i;

	dm = 0;
	if(zone < 0)
		dm -= 5;
	else if(zone > 0)
		dm -= 3;
	if((uwp[UWPI_ATMOS] != '0') || (uwp[UWPI_ATMOS] != '5') || 
			(uwp[UWPI_ATMOS] != '6') || (uwp[UWPI_ATMOS] != '8'))
		dm -= 2;
	i = dice.Roll(6, 2) - 2 + dm;
	if(i < 0)
		return(' ');
	return(SimpleConv(i));
}

// --------------------------------------------------------------------------
static char simple_conv_str[] =
	{ "0123456789ABCDEFGHJKLMNPQRSTUVWXYZ" };

char
UWPGen::SimpleConv(int v)
{
	if(v < 0)
		return('0');
	else
		return(simple_conv_str[v]);
}

char
UWPGen::SizeConv(int v, bool for_sat)
{
	if(v < 1) {
		if((for_sat) && (0 == v))
			return('R');
		return('S');
	} else
		return(simple_conv_str[v]);
}

void
UWPGen::ClearUWP(char *uwp)
{
int i;

	for(i = 0;i < UWP_BUFFER_SIZE;i++)
		uwp[i] = 0;
}

