
/*
 * File:    syst.cpp
 * Purpose:	
 * Author:	
 * Created:	
 * Updated:	
 * Copyright: LGPL.
 *            Traveller is a registered trademark of Far Future Enterprises.
 */

/* rcsid[] = "$RCSfile: syst.cpp,v $ $Revision: 1.8 $ $Author: man $ $Date: 2001/12/02 18:18:39 $" */



#include <stdio.h>
#define __USE_GNU
#include <string.h>
#include <ctype.h>
#include <math.h>
#include "syst.h"
#include "str_util.h"

// misc. parsing consts.
#define DEFAULT_ORB_START		0
#define DEFAULT_NAME_START		14
#define DEFAULT_UWP_START		32
#define DEFAULT_BASE_START		41
#define DEFAULT_CODE_START		43

#define DEFAULT_ORB_END			13
#define DEFAULT_NAME_END		31
#define DEFAULT_UWP_END			39
#define DEFAULT_BASE_END		41
#define DEFAULT_CODE_END		80

#define DEFAULT_PRIMARY_ORB			3
#define DEFAULT_PRIMARY_SAT_ORB		5
#define DEFAULT_SECONDARY_ORB		6
#define DEFAULT_SECONDARY_SAT_ORB	8
#define DEFAULT_TERTIARY_ORB		9
#define DEFAULT_TERTIARY_SAT_ORB	11

// general buffer decl.
#define BIG_BUFFER_SIZE		120
// ===========================================================================
#define MILLION_KM__PER_AU       149.6

float StarOrbit::AUOrbit[MAX_AU_TABLE] = {
  0.2,
  0.4,
  0.7,
  1.0,
  1.6,
  2.8,
  5.2,
  10.0,
  19.6,
  38.8,
  77.2,
  154.0,
  307.6,
  614.8,
  1229.2,
  2458.0,
  4915.6,
  9830.8,
  19661.2,
  39322.0
};

// ===========================================================================
#if 0
static void dump_parse(short *start, short *end)
{
int i;

fprintf(stderr, "start:");
for(i = 0; i < SYS_PARSE_ALLEG + 1;i++)
    fprintf(stderr, " %d", start[i]);
fprintf(stderr, "\n  end:");
for(i = 0; i < SYS_PARSE_ALLEG + 1;i++)
    fprintf(stderr, " %d", end[i]);
fprintf(stderr, "\n");

fprintf(stderr, " %d %d %d %d %d %d %d\n",
	SYS_PARSE_ORBIT,
	SYS_PARSE_NAME,
	SYS_PARSE_UWP,
	SYS_PARSE_BASE,
	SYS_PARSE_CODE,
	SYS_PARSE_ZONE,
	SYS_PARSE_ALLEG);
}
#endif

// ===========================================================================
BaseOrbit::BaseOrbit(BaseOrbit *p, unsigned short mask_val, float o) :
	ListData()
{
	orbit = o;
	orb_mask = mask_val;
	parent = p;
	v_id = 0l;
	i_id = 0;
}

BaseOrbit::~BaseOrbit()
{
}

char *
BaseOrbit::GetFormatedOrbit(char *buffer, bool prim_flag)
{
char l_buff[BIG_BUFFER_SIZE];

    buffer[0] = 0;
    // nice reursion
    if(parent != NULL)
	    parent->GetFormatedOrbit(buffer, TRUE);

	strcat(buffer, FormatOrbit(l_buff));

	if(prim_flag == TRUE)
	    strcat(buffer, ":");

	return(buffer);
}

char *
BaseOrbit::FormatOrbit(char *buffer)
{
	if(orbit < 0) {
		if(orb_mask & ORBIT_IS_PRIMARY)
		    sprintf(buffer, "Primary");
		else
		    sprintf(buffer, "Close");
	} else {
		if(orb_mask & ORBIT_IS_SAT) {
			sprintf(buffer, "%.0f", orbit);
		} else {
			sprintf(buffer, "%.1f", orbit);
		}
	}

	return(buffer);
}

char *
BaseOrbit::FormatOrbit(char *buffer, short params[], int width)
{
int i;

	// make sure our space is empty
	strfixsize(buffer, width);
	FormatOrbit(buffer);

	if(orb_mask & (ORBIT_IS_PRIMARY | ORBIT_ORBITS_PRIMARY))
		i = 0;
	else if(orb_mask & ORBIT_ORBITS_COMP)
		i = 2;
	else
		i = 4;

	if(orbit >= 0) {
		if(orb_mask & ORBIT_IS_SAT)
			i++;
		strfixsize(buffer, width);
		while(!isdigit(buffer[params[i]]))
		    strrot(buffer, 1, TRUE);
	}

	strfixsize(buffer, width);

	return(buffer);
}

int
BaseOrbit::GetHabitible(void)
{
    if(parent != NULL) {
		if(GetOrbType() == ORBIT_IS_SAT)
		    return(parent->GetHabitible());
		else {
			int hab;

			hab = (((StarOrbit *) parent)->GetStar())->CheckHabit(orbit);
			if((GetOrbOrbitType() & ORBIT_ORBITS_PRIMARY) == 0) {
				int hab1;
				
			    hab1 = parent->GetHabitible();
				if(hab1 < hab)
				    hab = hab1;
			}
			return(hab);
		}
	}
	
// must be primary!
	return(HAB_UNKNOWN);
}

STAR_INDEX
BaseOrbit::GetStarIndex(void)
{
BaseOrbit *bo;

    if(parent == NULL)
	    return(SI_PRIMARY);            // special case?

	bo = parent;
	while((bo != NULL) && (bo->GetEnumType() != OT_STAR))
	    bo = bo->GetParent();

	if(bo == NULL)
	    return(SI_UNKNOWN);
	if(bo->GetOrbType() & ORBIT_IS_PRIMARY)
	    return(SI_PRIMARY);
	if(bo->GetOrbType() & ORBIT_ORBITS_PRIMARY)
	    {
		if(bo->GetOrbType() & ORBIT_SECOND_STAR)
		    return(SI_2ND_COMP);
		else
		    return(SI_1ST_COMP);
		}
	if(bo->GetOrbType() & ORBIT_ORBITS_COMP)
	    {
		if(bo->GetOrbType() & ORBIT_SECOND_STAR)
		    return(SI_2ND_TERT);
		else
		    return(SI_1ST_TERT);
		}
	
	// what is it?
	return(SI_UNKNOWN);
}

void
BaseOrbit::GetOrbitArray(int *orbs)
{
int i,j;
BaseOrbit *parent,*o;

	for(i = 1;i < 4;i++)
		orbs[i] = -1;
	j = 0;
	orbs[0] = (int) (orbit * 10.0);
	o = this;

	while(((parent = o->GetParent()) != NULL) &&
				(0 == (parent->GetOrbType() & ORBIT_IS_PRIMARY))) {
		for(i = 3;i > 0;i--)
			orbs[i] = orbs[i-1];
		orbs[0] = (int) (parent->GetOrbit() * 10.0);
		o = parent;
	}
}

#ifdef SYS_DEBUG
void
BaseOrbit::dump(FILE *fp)
{
	fprintf(fp, "--------------------------------\n");
	fprintf(fp, " <<<<< Base Orbit >>>>> parent: 0x%x this: 0x%x\n", 
			parent, this);
	fprintf(fp, " orbit: %f  orb_mask: 0x%x\n", orbit, orb_mask);
}
#endif

// ---------------------------------------------------------------------------
SatOrbit::SatOrbit(BaseOrbit *p, unsigned short mask_val, float o, planet *s) :
	BaseOrbit(p, mask_val | ORBIT_IS_SAT, o)
{
	sat = s;
}

SatOrbit::~SatOrbit()
{
	if(sat != NULL)
		delete sat;
}

char *
SatOrbit::GetDesc(char *buffer)
{
	buffer[0] = 0;





// XXX

	return(buffer);
}

char *
SatOrbit::FormatName(char *buffer, int width)
{
	sat->GetName(buffer);
	if(width > -1)
	    strfixsize(buffer, width);
	return(buffer);
}

char *
SatOrbit::FormatUWP(char *buffer, int width)
{
	sat->GetUWPStr(buffer);
	strfixsize(buffer, width);
	return(buffer);
}

char *
SatOrbit::FormatBase(char *buffer, int width)
{
	sprintf(buffer, "%c", sat->GetBase());
	strfixsize(buffer, width);
	return(buffer);
}

void
SatOrbit::GetCodes(unsigned long *co, unsigned long *cx, unsigned long *cs)
{
	*co = sat->GetTradeCodes();
	*cx = sat->GetOtherCodes();
	*cs = sat->GetSysFlags();
}

char *
SatOrbit::FormatZone(char *buffer, int width)
{
	sprintf(buffer, "%c", sat->GetZone());
	strfixsize(buffer, width);
	return(buffer);
}

char *
SatOrbit::FormatAlleg(char *buffer, int width)
{
	sprintf(buffer, "%s", sat->GetAlleg());
	strfixsize(buffer, width);
	return(buffer);
}

bool
SatOrbit::IsRing(void)
{
    if(sat->get_size_val() == SIZE_RING)
	    return(TRUE);
	return(FALSE);
}

ORBIT_TYPE
SatOrbit::GetEnumType(void)
{
    if(IsRing())
	    return(OT_RING);
	else
	    return(OT_SATILLITE);
}

#ifdef SYS_DEBUG
void
SatOrbit::dump(FILE *fp)
{
	fprintf(fp, "--------------------------------\n");
	fprintf(fp, " <<<<< Sat Orbit >>>>> sat: 0x%x\n", sat);
	BaseOrbit::dump(fp);
}
#endif

// ---------------------------------------------------------------------------
SimpleOrbit::SimpleOrbit(BaseOrbit *p, unsigned short mask_val, float o) :
	BaseOrbit(p, mask_val, o)
{
	orbits = new OrbitTable(mask_val);
}

SimpleOrbit::~SimpleOrbit()
{
	if(orbits != NULL)
		delete orbits;
}

bool
SimpleOrbit::AddOrbit(BaseOrbit *bo, float o)
{
unsigned short mask;

    if(orbits == NULL)
	    return(FALSE);

	mask = GetOrbOrbitType();
	if(orb_mask & ORBIT_IS_STAR)
	    mask = ORBIT_IS_WORLD;
	else
	    mask = ORBIT_IS_SAT;
	bo->SetMask(mask);

	bo->SetOrbit(o);

	return(orbits->AppendOrbit(bo));
}

BaseOrbit *
SimpleOrbit::AppendToOrbitTable(unsigned short mask, float o, 
		planet *p)
{
ADD_ORBIT_RESULT stat;
BaseOrbit *bo;

	if(orbits == NULL)
		return(FALSE);
	if((bo = ValidateOrbit(&stat, mask, o, p)) != NULL) {
		if(CheckOrbitStat(stat, bo))
		    return(bo);
	}

	return(NULL);
}

BaseOrbit *
SimpleOrbit::AppendToOrbitTable(unsigned short mask, float o,
		DetailStar *r)
{
ADD_ORBIT_RESULT stat;
BaseOrbit *bo;

	if(orbits == NULL)
		return(FALSE);
	if((bo = ValidateOrbit(&stat, mask, o, r)) != NULL) {
		if(CheckOrbitStat(stat, bo))
		    return(bo);
	}

	return(NULL);
}

bool
SimpleOrbit::CheckOrbitStat(ADD_ORBIT_RESULT stat, BaseOrbit *bo)
{
	if(stat == ADD_ORBIT_BAD)
		return(FALSE);
	else {
		if(stat == ADD_ORBIT_OK)
			return(orbits->AppendOrbit(bo));
	}
	return(TRUE);
}

BaseOrbit *
SimpleOrbit::GetNthOrbit(int ndx)
{
ListNode *n;

	n = orbits->Nth(ndx);
	if(n != NULL)
		return((SatOrbit *)n->Data());
	else
		return(NULL);
}

BaseOrbit *
SimpleOrbit::GetFirstOrbit(void)
{
ListNode *n;

	n = orbits->First();
	if(n != NULL)
		return((SatOrbit *)n->Data());
	else
		return(NULL);
}

bool
SimpleOrbit::DeleteOrbit(BaseOrbit *bo)
{
ListNode *n;
BaseOrbit *o;
SatOrbit *a;
StarOrbit *s;
WorldOrbit *w;

// ok, this is kludgy.....
	n = orbits->First();
	while(n != NULL) {
		o = (BaseOrbit *) n->Data();

		if(o != bo)
		    n = n->Next();
		else {
			switch(o->GetOrbType()) {
			    case ORBIT_IS_STAR:
					s = (StarOrbit *) o;
					delete s;
					break;
				case ORBIT_IS_WORLD:
					w = (WorldOrbit *) o;
					delete w;
					break;
				case ORBIT_IS_SAT:
					a = (SatOrbit *) o;
					delete a;
					break;
				default:
					delete o;
					break;
			}
			delete n;
			return(TRUE);
		}
	}

	return(FALSE);
}

#ifdef SYS_DEBUG
void
SimpleOrbit::dump(FILE *fp)
{
	fprintf(fp, "--------------------------------\n");
	fprintf(fp, " <<<<< Simple Orbit >>>>> \n");
	BaseOrbit::dump(fp);
	if(orbits != NULL)
		orbits->dump(fp);
}
#endif

// ---------------------------------------------------------------------------
WorldOrbit::WorldOrbit(BaseOrbit *p, unsigned short mask_val, float o, 
		planet *w) :
	SimpleOrbit(p, mask_val | ORBIT_IS_WORLD, o)
{
	world = w;
}

WorldOrbit::~WorldOrbit()
{
	if(world != NULL)
		delete world;
}

char *
WorldOrbit::GetDesc(char *buffer)
{
	buffer[0] = 0;



// XXX


	return(buffer);
}

ORBIT_TYPE
WorldOrbit::GetEnumType(void)
{
    if(IsLargeGG() || IsSmallGG())
	    return(OT_GG);
	else if(IsBelt())
	    return(OT_BELT);
	else
	    return(OT_WORLD);
}

char *
WorldOrbit::FormatName(char *buffer, int width)
{
	world->GetName(buffer);
	if(width > -1)
	    strfixsize(buffer, width);
	return(buffer);
}

char *
WorldOrbit::FormatUWP(char *buffer, int width)
{
	world->GetUWPStr(buffer);
	strfixsize(buffer, width);
	return(buffer);
}

char *
WorldOrbit::FormatBase(char *buffer, int width)
{
	sprintf(buffer, "%c", world->GetBase());
	strfixsize(buffer, width);
	return(buffer);
}

void
WorldOrbit::GetCodes(unsigned long *co, unsigned long *cx, unsigned long *cs)
{
	*co = world->GetTradeCodes();
	*cx = world->GetOtherCodes();
	*cs = world->GetSysFlags();
}

char *
WorldOrbit::FormatZone(char *buffer, int width)
{
	sprintf(buffer, "%c", world->GetZone());
	strfixsize(buffer, width);
	return(buffer);
}

char *
WorldOrbit::FormatAlleg(char *buffer, int width)
{
	sprintf(buffer, "%s", world->GetAlleg());
	strfixsize(buffer, width);
	return(buffer);
}

bool
WorldOrbit::IsLargeGG(void)
{
    if(world->get_size_val() == SIZE_LARGE_GG)
	    return(TRUE);
	return(FALSE);
}

bool
WorldOrbit::IsSmallGG(void)
{
    if(world->get_size_val() == SIZE_SMALL_GG)
	    return(TRUE);
	return(FALSE);
}

bool
WorldOrbit::IsBelt(void)
{
    if(world->get_size_val() == 0)
	    return(TRUE);
	return(FALSE);
}

BaseOrbit *
WorldOrbit::ValidateOrbit(ADD_ORBIT_RESULT *stat, 
        unsigned short mask, float orb, planet *p)
{
SatOrbit *so=NULL;

	*stat = ADD_ORBIT_BAD;			// assume the worse
	if(((mask & ORBIT_ORBIT_FLAGS) == (orb_mask & ORBIT_ORBIT_FLAGS)) ||
		(mask & ORBIT_IS_SAT)) {
		*stat = ADD_ORBIT_OK;
		so = new SatOrbit(this, mask, orb, p);
	}

	return(so);
}

BaseOrbit *
WorldOrbit::ValidateOrbit(ADD_ORBIT_RESULT *stat, 
        unsigned short mask, float orb, DetailStar *r)
{
// can't do this
	*stat = ADD_ORBIT_BAD;
	return(NULL);
}

SatOrbit *
WorldOrbit::GetSat(short index)
{
ListNode *n;

	n = orbits->Nth(index);
	if(n != NULL)
		return((SatOrbit *)n->Data());
	else
		return(NULL);
}

float
WorldOrbit::GetKmOrbit(float orb)
{
    return(world->GetWorldRadius() * orb);
}

#ifdef SYS_DEBUG
void
WorldOrbit::dump(FILE *fp)
{
	fprintf(fp, "--------------------------------\n");
	fprintf(fp, " <<<<< World Orbit >>>>> world: 0x%x\n", world);
	SimpleOrbit::dump(fp);
}
#endif

// ---------------------------------------------------------------------------
StarOrbit::StarOrbit(BaseOrbit *p, unsigned short mask_val, float o, 
		DetailStar *s) :
	SimpleOrbit(p, mask_val | ORBIT_IS_STAR, o)
{
	star = s;
	world = NULL;
}

StarOrbit::~StarOrbit()
{
	if(star != NULL)
		delete star;
}

char *
StarOrbit::GetDesc(char *buffer)
{
	buffer[0] = 0;




// XXX

	return(buffer);
}

char *
StarOrbit::FormatName(char *buffer, int width)
{
	star->GetName(buffer);
	if(width > -1)
	    strfixsize(buffer, width);
	return(buffer);
}

char *
StarOrbit::FormatUWP(char *buffer, int width)
{
	star->GetStarStr(buffer);
	strfixsize(buffer, width);
	return(buffer);
}

char *
StarOrbit::FormatBase(char *buffer, int width)
{
    buffer[0] = 0;
    strfixsize(buffer, width);
	return(buffer);
}

void
StarOrbit::GetCodes(unsigned long *co, unsigned long *cx, unsigned long *cs)
{
	*co = *cx = *cs = 0;
}

char *
StarOrbit::FormatZone(char *buffer, int width)
{
    buffer[0] = 0;
	strfixsize(buffer, width);
	return(buffer);
}

char *
StarOrbit::FormatAlleg(char *buffer, int width)
{
    buffer[0] = 0;
	strfixsize(buffer, width);
	return(buffer);
}

WorldOrbit *
StarOrbit::CheckWorldOrbit(ADD_ORBIT_RESULT *stat, 
        unsigned short mask, float orb, planet *p)
{
WorldOrbit *wo=NULL;

	*stat = ADD_ORBIT_BAD;			// assume the worse

	if((mask & ORBIT_IS_SAT) && (world != NULL)) {
		*stat = ADD_ORBIT_NO;
		world->AppendToOrbitTable(mask, orb, p);
	} else if(mask & ORBIT_IS_WORLD) {
		*stat = ADD_ORBIT_OK;
		world = wo = new WorldOrbit(this, mask, orb, p);
	}

	return(wo);
}

SimpleOrbit *
StarOrbit::GetOrbit(short index)
{
ListNode *n;

	n = orbits->Nth(index);
	if(n != NULL)
		return((SimpleOrbit *)n->Data());
	else
		return(NULL);
}

float
StarOrbit::GetAUOrbit(float orb)
{
float num,rem;
float hi,lo;
int index;

    num = floor(orb);
	index = (int)num;
	rem = floor((orb - num) * 10.0);

	if(index > MAX_AU_TABLE)
	    lo = AUOrbit[MAX_AU_TABLE - 1] * 2 * (index - MAX_AU_TABLE);
	else
	    lo = AUOrbit[index];

	if((index + 1) > MAX_AU_TABLE)
	    hi = AUOrbit[MAX_AU_TABLE - 1] * 2 * (index + 1 - MAX_AU_TABLE);
	else
	    hi = AUOrbit[index + 1];

	return(lo + ((orb - num) * (hi - lo)));
}

#ifdef SYS_DEBUG
void
StarOrbit::dump(FILE *fp)
{
	fprintf(fp, "--------------------------------\n");
	fprintf(fp, " <<<<< Star Orbit >>>>> star: 0x%x\n", star);
	SimpleOrbit::dump(fp);
}
#endif

// ---------------------------------------------------------------------------
TertOrbit::TertOrbit(BaseOrbit *p, unsigned short mask_val, float o, 
		DetailStar *r) :
	StarOrbit(p, mask_val, o, r)
{
}

TertOrbit::~TertOrbit()
{
}

BaseOrbit *
TertOrbit::ValidateOrbit(ADD_ORBIT_RESULT *stat, 
        unsigned short mask, float orb, DetailStar *r)
{
// can't do this
	*stat = ADD_ORBIT_BAD;
	return(NULL);
}

BaseOrbit *
TertOrbit::ValidateOrbit(ADD_ORBIT_RESULT *stat, 
        unsigned short mask, float orb, planet *p)
{
WorldOrbit *wo=NULL;

	*stat = ADD_ORBIT_BAD;			// assume the worse
	if((mask & ORBIT_ORBIT_FLAGS) == (orb_mask & ORBIT_ORBIT_FLAGS)) {
		wo = CheckWorldOrbit(stat, mask, orb, p);
	}

	return(wo);
}

#ifdef SYS_DEBUG
void
TertOrbit::dump(FILE *fp)
{
	fprintf(fp, "--------------------------------\n");
	fprintf(fp, " <<<<< Tert Orbit >>>>>\n");
	StarOrbit::dump(fp);
}
#endif

// ---------------------------------------------------------------------------
CompOrbit::CompOrbit(BaseOrbit *p, unsigned short mask_val, float o, 
		DetailStar *r) :
	StarOrbit(p, mask_val, o, r)
{
	tert = NULL;
}

CompOrbit::~CompOrbit()
{
}

BaseOrbit *
CompOrbit::ValidateOrbit(ADD_ORBIT_RESULT *stat, 
        unsigned short mask, float orb, planet *p)
{
WorldOrbit *wo=NULL;

	*stat = ADD_ORBIT_BAD;			// assume the worse
	if(mask & ORBIT_ORBITS_COMP) {
		wo = CheckWorldOrbit(stat, mask, orb, p);
	} else {
		if(tert != NULL) {
			if(tert->AppendToOrbitTable(mask, orb, p) != NULL)
				*stat = ADD_ORBIT_NO;
		}
	}
	return(wo);
}

BaseOrbit *
CompOrbit::ValidateOrbit(ADD_ORBIT_RESULT *stat, 
        unsigned short mask, float orb, DetailStar *r)
{
TertOrbit *to;

	*stat = ADD_ORBIT_BAD;			// assume the worse
	if(mask & ORBIT_ORBITS_COMP) {
		*stat = ADD_ORBIT_OK;
		tert = to = new TertOrbit(this, mask, orb, r);
	}

	return(to);
}

#ifdef SYS_DEBUG
void
CompOrbit::dump(FILE *fp)
{
	fprintf(fp, "--------------------------------\n");
	fprintf(fp, " <<<<< Comp Orbit >>>>>\n");
	StarOrbit::dump(fp);
}
#endif

// ---------------------------------------------------------------------------
PrimOrbit::PrimOrbit(unsigned short mask_val, DetailStar *r) :
	StarOrbit(NULL, mask_val, (float) -1.0, r)
{
	comp[0] = comp[1] = NULL;
}

PrimOrbit::~PrimOrbit()
{
}

BaseOrbit *
PrimOrbit::ValidateOrbit(ADD_ORBIT_RESULT *stat, 
        unsigned short mask, float o, planet *p)
{
WorldOrbit *wo=NULL;

	*stat = ADD_ORBIT_BAD;			// assume the worse
	if(mask & ORBIT_ORBITS_PRIMARY)
		wo = CheckWorldOrbit(stat, mask, o, p);
	else {
		if(comp[1] != NULL) {
			// adjust mask here, since this is the earliest we can
			// figure it out..........
			mask |= ORBIT_SECOND_STAR;
			if(comp[1]->AppendToOrbitTable(mask, o, p) != NULL)
				*stat = ADD_ORBIT_NO;
		} else if(comp[0] != NULL) {
			if(comp[0]->AppendToOrbitTable(mask, o, p) != NULL)
				*stat = ADD_ORBIT_NO;
		}
	}

	return(wo);
}

BaseOrbit *
PrimOrbit::ValidateOrbit(ADD_ORBIT_RESULT *stat, 
        unsigned short mask, float o, DetailStar *r)
{
int i;
CompOrbit *co;

	*stat = ADD_ORBIT_BAD;			// assume the worse
	if(mask & ORBIT_ORBITS_PRIMARY) {
		if(comp[0] == NULL)
			i = 0;
		else if(comp[1] == NULL)
			i = 1;

		*stat = ADD_ORBIT_OK;
		comp[i] = co = new CompOrbit(this, mask, o, r);
	} else if(mask & ORBIT_ORBITS_COMP) {
		if(comp[1] != NULL) {
			// adjust mask here, since this is the earliest we can
			// figure it out..........
			mask |= ORBIT_SECOND_STAR;
			if(comp[1]->AppendToOrbitTable(mask, o, r) != NULL)
				*stat = ADD_ORBIT_NO;
		} else if(comp[0] != NULL) {
			if(comp[0]->AppendToOrbitTable(mask, o, r) != NULL)
				*stat = ADD_ORBIT_NO;
		}
	}

	return(co);
}

CompOrbit *
PrimOrbit::GetComp(int index)
{
	if((index < 0) || (index > 1))
		return(NULL);

	return(comp[index]);
}

TertOrbit *
PrimOrbit::GetTert(int index)
{
	if((index < 0) || (index > 1) || (comp[index] == NULL))
		return(NULL);

	return(comp[index]->GetTert());
}

// ---------------------------------------------------------------------------

#ifdef SYS_DEBUG
void
PrimOrbit::dump(FILE *fp)
{
	fprintf(fp, "================================\n");
	fprintf(fp, " <<<<< Primary Orbit:>>>>> comp 1: 0x%x  comp 2: 0x%x\n", 
			comp[0], comp[1]);
	StarOrbit::dump(fp);
}
#endif

// ---------------------------------------------------------------------------
OrbitTable::OrbitTable(unsigned short allow_mask) :
	LinkedList()
{
	type = allow_mask;
}

OrbitTable::~OrbitTable()
{
ListNode *n;
BaseOrbit *o;
SatOrbit *a;
StarOrbit *s;
WorldOrbit *w;

// ok, this is kludgy.....
	n = First();
	while(n != NULL) {
		o = (BaseOrbit *) n->Data();

		switch(o->GetOrbType()) {
			case ORBIT_IS_STAR:
				s = (StarOrbit *) o;
				delete s;
				break;
			case ORBIT_IS_WORLD:
				w = (WorldOrbit *) o;
				delete w;
				break;
			case ORBIT_IS_SAT:
				a = (SatOrbit *) o;
				delete a;
				break;
			default:
				delete o;
				break;
		}

		Remove();

		n = First();
	}
}

bool
OrbitTable::AppendOrbit(BaseOrbit *orbit)
{
int i=0;
ListNode *node;
BaseOrbit *o;
float orb;

	orb = orbit->GetOrbit();

    node = First();
	while(node != NULL) {
		o = (BaseOrbit *) node->Data();

		if(orb > o->GetOrbit()) {
			node = node->Next();
			i++;
		    continue;
		} else {
		    Insert(orbit, i);
		    return(TRUE);
		}
	}

	if(node == NULL)
		Append(orbit);

	return(TRUE);
}

#ifdef SYS_DEBUG
void
OrbitTable::dump(FILE *fp)
{
ListNode *n;
BaseOrbit *bo;

	fprintf(fp, "--------------------------------\n");
	fprintf(fp, " <<<<< Orbit Table >>>>> type: %d\n", type);

	n = First();
	while(n != NULL)
		{
		bo = (BaseOrbit *)n->Data();
		switch(bo->GetOrbType())
			{
			case ORBIT_IS_STAR:
				if(bo->GetOrbOrbitType() & ORBIT_ORBITS_PRIMARY)
					{
					CompOrbit *po;
					po = (CompOrbit *)bo;
					po->dump(fp);
					}
				else if(bo->GetOrbOrbitType() & ORBIT_ORBITS_COMP)
					{
					TertOrbit *po;
					po = (TertOrbit *)bo;
					po->dump(fp);
					}
				else
					fprintf(fp, "Unkown Star(0x%x)!!!!\n", bo->GetOrbMask());
				break;
			case ORBIT_IS_WORLD:
				WorldOrbit *wo;
				wo = (WorldOrbit *)bo;
				wo->dump(fp);
				break;
			case ORBIT_IS_SAT:
				SatOrbit *so;
				so = (SatOrbit *)bo;
				bo->dump(fp);
				break;
			case ORBIT_IS_PRIMARY:
				PrimOrbit *po;
				po = (PrimOrbit *)bo;
				po->dump(fp);
				break;
			default:
				fprintf(fp, "Unkown(0x%x)!!!!\n", bo->GetOrbMask());
			}
		n = n->Next();
		}
}
#endif

// ===========================================================================
System::System(TCodes *tc)
{
	InitSystem();
// XXX
	sect = new DetailSector(tc);
	file_name = NULL;
}

System::~System()
{
	Clear();
}

#ifdef SYS_DEBUG
void
System::dump(FILE *fp)
{
	fprintf(fp, "================================\n");
	fprintf(fp, " <<<<< System:>>>>> \n");
	if(prim_orbit != NULL)
		{
		fprintf(fp, "prim_orbit: 0x%x\n", prim_orbit);
		prim_orbit->dump(fp);
		}
}
#endif

void
System::InitSystem(void)
{
//int i;

	name = NULL;

//	for(i = 0;i < MAX_STARS;i++)
//		stars[i] = NULL;
//	primary = NULL;
	prim_orbit = NULL;
	cur_orbit = NULL;

	start_parse[0] = DEFAULT_ORB_START;
	start_parse[1] = DEFAULT_NAME_START;
	start_parse[2] = DEFAULT_UWP_START;
	start_parse[3] = DEFAULT_BASE_START;
	start_parse[4] = DEFAULT_CODE_START;

	end_parse[0] = DEFAULT_ORB_END;
	end_parse[1] = DEFAULT_NAME_END;
	end_parse[2] = DEFAULT_UWP_END;
	end_parse[3] = DEFAULT_BASE_END;
	end_parse[4] = DEFAULT_CODE_END;

	orb_parse[0] = DEFAULT_PRIMARY_ORB;
	orb_parse[1] = DEFAULT_PRIMARY_SAT_ORB;
	orb_parse[2] = DEFAULT_SECONDARY_ORB;
	orb_parse[3] = DEFAULT_SECONDARY_SAT_ORB;
	orb_parse[4] = DEFAULT_TERTIARY_ORB;
	orb_parse[5] = DEFAULT_TERTIARY_SAT_ORB;
}

void
System::Clear(void)
{
	if(name != NULL)
		delete name;

// this should clear it all
	if(prim_orbit != NULL)
		delete prim_orbit;
	prim_orbit = NULL;

	InitSystem();
}

// ---------------------------------------------------------------------------
void
System::SetName(char *buff)
{
    if(name != NULL)
	    delete name;
	name = new char[strlen(buff) + 1];
	strcpy(name, buff);	
}

void
System::SetLoc(char *buff)
{
    if(loc != NULL)
	    delete loc;
	loc = new char[strlen(buff) + 1];
	strcpy(loc, buff);	
}

void
System::SetSectName(char *buff)
{
    if(sect_name != NULL)
	    delete sect_name;
	sect_name = new char[strlen(buff) + 1];
	strcpy(sect_name, buff);	
}

void
System::SetSSName(char *buff)
{
    if(ss_name != NULL)
	    delete ss_name;
	ss_name = new char[strlen(buff) + 1];
	strcpy(ss_name, buff);	
}

// ---------------------------------------------------------------------------
BaseOrbit *
System::GetIndexedOrbit(int index)
{
int count=0;

    return(GetIndexedOrbit((BaseOrbit *)prim_orbit, index, &count));
}

BaseOrbit *
System::GetIndexedOrbit(BaseOrbit *bo, int index, int *c)
{
int type,count;

    count = *c;
    while(bo != NULL) {
		if(index == count)
			return(bo);

		count++;
		type = bo->GetOrbType();

		if(type & ORBIT_IS_STAR) {
			int i=0;
			SimpleOrbit *so;
			StarOrbit *to;

			to = (StarOrbit *) bo;
			while((so = to->GetOrbit(i)) != NULL) {
				BaseOrbit *lbo;
				if((lbo =
					GetIndexedOrbit((BaseOrbit *)so, index, &count)) != NULL)
				  return(lbo);
				i++;
			}
			bo = NULL;
		} else if(type & ORBIT_IS_WORLD) {
			int i=0;
			SatOrbit *so;
			WorldOrbit *wo;

			wo = (WorldOrbit *) bo;
			while((so = wo->GetSat(i)) != NULL) {
			   if(index == count)
				   return((BaseOrbit *) so);
			   count++;
			   i++;
		   }
			bo = NULL;
		} else
		    bo = NULL;
	}

	*c = count;
	return(NULL);
}

void
System::SetCurrentOrbit(int index)
{
BaseOrbit *bo;
int count=0;

    if((bo = GetIndexedOrbit((BaseOrbit *)prim_orbit, index, &count)) != NULL)
	    cur_orbit = bo;
}

void
System::SetCurrentOrbitV(long id)
{
BaseOrbit *bo;
int count=0;

	while((bo = GetIndexedOrbit(count++)) != NULL) {
		if(bo->GetID() == id) {
	    	cur_orbit = bo;
			return;
		}
	}
}

// ---------------------------------------------------------------------------
// VIndexed routines look up an orbit based on a referenced v_id
BaseOrbit *
System::GetVIndexedOrbit(long v)
{
int count=0;
BaseOrbit *bo;

	while((bo = GetIndexedOrbit(count++)) != NULL) {
		if(v == bo->GetID()) 
			return(bo);
	}

	return(NULL);
}

int
System::TranslateVIndex(long v)
{
int count=0;
BaseOrbit *bo;

	while((bo = GetIndexedOrbit(count)) != NULL) {
		if(v == bo->GetID()) 
			return(count);
		count++;
	}

	return(-1);
}

int
System::TranslateParentVIndex(long v)
{
BaseOrbit *bo;

	if((bo = GetVIndexedOrbit(v)) != NULL) {
		if((bo = bo->GetParent()) != NULL) {
			return(GetOrbitIndex(bo));
		}
	}
	
	// fell through, either bad v_id, or no parent
	return(0);
}

// ---------------------------------------------------------------------------
// given the absolute index, translate it to an index relative 
//  to the given parent absolute index
int
System::TranslateRelIndex(int ndx, int pndx)
{
BaseOrbit *orb;
SimpleOrbit *parent;

	if((orb = GetIndexedOrbit(ndx)) != NULL) {
		if((parent = (SimpleOrbit *)orb->GetParent()) != NULL) {
			int count=0;
			BaseOrbit *bo;

			while((bo = parent->GetNthOrbit(count)) != NULL) {
				if(bo == orb)
					return(count);
				count++;
			}
		}
	}

	// fell through, bad parent or orbit
	return(0);
}

// ---------------------------------------------------------------------------
BaseOrbit *
System::Find(int orb1, int orb2, int orb3, int orb4)
{
int i,j;
float tests[4];
BaseOrbit *bo=NULL,*bo1=NULL;

	tests[0] = (float) (orb1 / 10) + (float) ((orb1 % 10) * 0.01);
	tests[1] = (float) (orb2 / 10) + (float) ((orb2 % 10) * 0.01);
	tests[2] = (float) (orb3 / 10) + (float) ((orb3 % 10) * 0.01);
	tests[3] = (float) (orb4 / 10) + (float) ((orb4 % 10) * 0.01);
	i = 0;
	bo1 = prim_orbit;
	for(i = 0;i < 4;i++) {
		j = 0;
		if(tests[i] < 0.0)
			break;
		while((bo = bo1->GetNthOrbit(j)) != NULL) {
			if(bo->GetOrbit() == tests[i])
				break;
			j++;
		}
		if(NULL == bo)
			break;
		bo1 = bo;
	}

	return(bo);
}

int
System::GetOrbitIndex(BaseOrbit *o)
{
int i=0;
BaseOrbit *bo;

    while((bo = GetIndexedOrbit(i)) != NULL) {
		if(bo == o)
		    return(i);
		i++;
	}
	
	return(0); // ????
}

BaseOrbit *
System::GetOrbit(STAR_INDEX st, short pl, short sat)
{
SimpleOrbit *so;
WorldOrbit *wo;
BaseOrbit *bo=NULL;
StarOrbit *sto=NULL;

	switch(st)
		{
		case SI_PRIMARY:
			sto = prim_orbit;
			break;
		case SI_1ST_COMP:
			sto = prim_orbit->GetComp(0);
			break;
		case SI_2ND_COMP:
			sto = prim_orbit->GetComp(1);
			break;
		case SI_1ST_TERT:
			sto = prim_orbit->GetTert(0);
			break;
		case SI_2ND_TERT:
			sto = prim_orbit->GetTert(1);
			break;
		default:
			break;
		}
	if(sto != NULL)
		{
		if(pl < 0)
			bo = sto;
		else
			{
			if((so = sto->GetOrbit(pl)) != NULL)
			    {
				if(sat < 0)
				    bo = so;
				else
				    {
					if(so->GetOrbType() & ORBIT_IS_WORLD)
					    {
						wo = (WorldOrbit *) so;
						bo = wo->GetSat(sat);
						}
					}
				}
			}
		}

	return(bo);
}

// used to iterate through all worlds
// returns NULL on invalid indecies
// returns an empty string if a star is referenced on pl
char *
System::GetListing(STAR_INDEX st, short pl, short sat, char *buff)
{
BaseOrbit *bo=NULL;

	buff[0] = 0;

	if((bo = GetOrbit(st, pl, sat)) == NULL)
		return(NULL);

	return(GetListing(bo, buff));
}

char *
System::GetListing(int ndx, char *buff)
{
BaseOrbit *bo=NULL;

    buff[0] = 0;

	if((bo = GetIndexedOrbit(ndx)) == NULL)
	    return(NULL);

	return(GetListing(bo, buff));
}

char *
System::GetListing(BaseOrbit *bo, char *buff)
{
int i;
char lbuff[120];
ListNode *n;
CodeTableEntry *ct;
unsigned long co, cx, cs, l;

    bo->FormatOrbit(buff, orb_parse,
				end_parse[SYS_PARSE_ORBIT] - start_parse[SYS_PARSE_ORBIT] + 1);
	strcat(buff, " ");

	bo->FormatName(lbuff,
				end_parse[SYS_PARSE_NAME] - start_parse[SYS_PARSE_NAME] + 1);
	strcat(buff, lbuff);
	strcat(buff, " ");

	if((end_parse[SYS_PARSE_UWP] - start_parse[SYS_PARSE_UWP]) < 9)
	    i = 9;
	else
	    i = end_parse[SYS_PARSE_UWP] - start_parse[SYS_PARSE_UWP] + 1;
	bo->FormatUWP(lbuff, i);
	strcat(buff, lbuff);
	strcat(buff, " ");

	bo->FormatBase(lbuff,
			   end_parse[SYS_PARSE_BASE] - start_parse[SYS_PARSE_BASE] + 1);
	strcat(buff, lbuff);
	strcat(buff, " ");

	// walk thru ea. code table and get any that are there
	bo->GetCodes(&co, &cx, &cs);
	lbuff[0] = 0;
	//  trade
	l = 1l;
	n = (ListNode *)sect->GetTradeTable()->First();
	while((n != NULL) && (l != 0))
	    {
		if(co & l)
		    {
			ct = (CodeTableEntry *) n->Data();
			strcat(lbuff, ct->GetCode());
			strcat(lbuff, " ");
			}
		l *= 2;
		n = n->Next();
		}
	//  other
	l = 1l;
	n = sect->GetCodeTable()->First();
	while((n != NULL) && (l != 0))
	    {
		if(cx & l)
		    {
			ct = (CodeTableEntry *) n->Data();
			strcat(lbuff, ct->GetCode());
			strcat(lbuff, " ");
			}
		l *= 2;
		n = n->Next();
		}
	//  trade
	l = 1l;
	n = sect->GetSystTable()->First();
	while((n != NULL) && (l != 0))
	    {
		if(cs & l)
		    {
			ct = (CodeTableEntry *) n->Data();
			strcat(lbuff, ct->GetCode());
			strcat(lbuff, " ");
			}
		l *= 2;
		n = n->Next();
		}
	strfixsize(lbuff,
			 end_parse[SYS_PARSE_CODE] - start_parse[SYS_PARSE_CODE] + 1);
	strcat(buff, lbuff);

	bo->FormatZone(lbuff,
			 end_parse[SYS_PARSE_ZONE] - start_parse[SYS_PARSE_ZONE] + 1);
	strcat(buff, lbuff);
	strcat(buff, " ");
	
	bo->FormatAlleg(lbuff,
			 end_parse[SYS_PARSE_ALLEG] - start_parse[SYS_PARSE_ALLEG] + 1);
	strcat(buff, lbuff);
	strcat(buff, " ");

	return(buff);
}

// ---------------------------------------------------------------------------
bool
System::MoveCurrentOrbit(STAR_INDEX si, float val)
{
StarOrbit *so;
BaseOrbit *bo;
int i;

    so = (StarOrbit *)GetOrbit(si, -1, -1);

	// see if it will be a sat
	while((bo = GetOrbit(si, i, -1)) != NULL) {
		if((bo->GetOrbit() == val) && (bo->GetOrbType() & ORBIT_IS_WORLD)) {
			// if already a sat just move it
			if(cur_orbit->GetOrbType() & ORBIT_IS_SAT)
			    return(((SimpleOrbit *)bo)->AddOrbit(cur_orbit,
					   bo->GetOrbit()));
			// not already a sat, add as one, and then delete it
			else {
				unsigned short mask;
				BaseOrbit *temp;
				planet *p;

				mask = cur_orbit->GetOrbOrbitType() | ORBIT_IS_SAT;
				p = new planet(((WorldOrbit *)cur_orbit)->GetWorld());
				temp = ((SimpleOrbit *)bo)->AppendToOrbitTable(mask, 5.0, p);
				if(temp != NULL) { 
					delete cur_orbit;
					cur_orbit = temp;
					return(TRUE);
				}								
			}
		}
		i++;
    }

	// fell thru, has to be a non-sat
	// if a sat, make it a world and delete it
	if(cur_orbit->GetOrbType() & ORBIT_IS_SAT) {
		unsigned short mask;
		BaseOrbit *temp;
		planet *p;

		mask = cur_orbit->GetOrbOrbitType() | ORBIT_IS_WORLD;
		p = new planet(((WorldOrbit *)cur_orbit)->GetWorld());
		temp = ((SimpleOrbit *)bo)->AppendToOrbitTable(mask, val, p);
		if(temp != NULL) { 
			delete cur_orbit;
			cur_orbit = temp;
			return(TRUE);
		}
	}
	// if not a sat, just move it
	else if(cur_orbit->GetOrbType() & ORBIT_IS_WORLD)
	    return(((SimpleOrbit *)bo)->AddOrbit(cur_orbit, val));

    // really fell thru, something went wrong
    return(FALSE);
}

// check a possible orbit
//  st - star that will have an orbit added to
//  orb - value of the orbit
//        if an orbit exists at there, will check if it can be a sat
//  ot - orbit type to be added
//  ig - ignore warning (OC_LOOSE_SAT)
ORBIT_CHECK_RESP
System::CheckOrbit(STAR_INDEX st, float orb, ORBIT_TYPE ot, bool ig)
{
int i=0;
BaseOrbit *o;
DetailStar *star;
BaseOrbit *p;
//SimpleOrbit *sio;

    p = GetOrbit(st, -1, -1);
	star = ((StarOrbit *)p)->GetStar();
	// check if a valid orbit for this star
	if((ot != OT_RING) && (ot != OT_SATILLITE))
	    {
		if(star->GetFirstOrbit() > orb)
		    return(OC_NO_ORB);
		if(p->GetOrbType() & ORBIT_IS_PRIMARY)
		    {
			if(orb > 19.5)
			    return(OC_NO_ORB);
			}
		else
		    { 
			if(orb > (p->GetOrbit() / 2.0))
			    return(OC_NO_ORB);
			}
		}
	
	// adding star?
	if(ot == OT_STAR)
	    {
		unsigned short typ;
		
		// check if there is a holding place for a star
		typ = p->GetOrbType();
		if(typ & ORBIT_IS_PRIMARY)
		    {
			if((((PrimOrbit *)p)->GetComp(0) != NULL) &&
			   (((PrimOrbit *)p)->GetComp(1) != NULL))
			    return(OC_TOO_MANY_STARS);
			}
		else if(typ & ORBIT_ORBITS_PRIMARY)
		    {
			if(((CompOrbit *)p)->GetTert() != NULL)
			    return(OC_TOO_MANY_STARS);
		    }
		else
		    return(OC_TOO_MANY_STARS);

		// ok, there will be a place for a star,
		//  check if it will conflict with any existing orbit(s)
		while((o = GetOrbit(st, i, -1)) != NULL)
		    {
			if((o->GetOrbit() > (orb / 2.0)) &&
			   (o->GetOrbit() < (orb + 2.0)))
			    return(OC_CONFLICTS);
			i++;
		    }
		}

	// now, walk thru ea. orbit and see:
	//  1) if this exists with an existing star orbit,
	//  2) if this will end up as a satillite.....
	i = 0;
	while((o = GetOrbit(st, i, -1)) != NULL)
	    {
		// see if a star is in the way
		if((o->IsStar()) && (o->GetOrbit() >= 0.0))
		    {
			if((orb > (o->GetOrbit() / 2.0)) &&
			   (orb < (o->GetOrbit() + 2.0)))
			    return(OC_STAR_CONFLICTS);
			}

		// check if the one in question should orbit here
		// (the floating point may need some work....)
		if(o->GetOrbit() == orb)
		    {
			if(o->IsStar() || o->IsRing() || o->IsBelt())
			    return(OC_NO_SAT);

			switch(ot)
			    {
				case OT_UNKNOWN:
				case OT_STAR:
				case OT_GG:
				case OT_BELT:
				    return(OC_NOT_A_WORLD);
					break;
				case OT_WORLD:
				    if(!ig)
					    return(OC_LOOSE_SAT);
					break;
				case OT_SATILLITE:
				case OT_RING:
				    break;
				}
		    }

		i++;
	    }

    return(OC_OK);
}

bool
System::DeleteOrbit(BaseOrbit *bo)
{
SimpleOrbit *po;
		
    if((po = (SimpleOrbit *)bo->GetParent()) == NULL)
	    return(FALSE);
	
    if(bo == cur_orbit)
	    cur_orbit = po;

	return(po->DeleteOrbit(bo));
}

// ---------------------------------------------------------------------------
// parsing routines
/*
#define PLANET_FLAG_FARMING			0x01
#define PLANET_FLAG_MINING			0x02
#define PLANET_FLAG_COLONY			0x04
#define PLANET_FLAG_LAB				0x08
#define PLANET_FLAG_MILITARY		0x10
*/
// trade, other, system
void
System::ParseCodes(char *src, unsigned long *co, unsigned long *cx, 
	unsigned long *cs)
{
unsigned long i=0;
char *ptr;
CodeTableEntry *ct;
ListNode *n;

    *co = *cx = *cs = 0l;
	if((NULL == src) || (strlen(src) == 0)) return;

    n = sect->GetTradeTable()->First();
    i = 1l;
    while((n != NULL) && (i != 0))
        {
        ct = (CodeTableEntry *) n->Data();
        if(((ptr = ct->GetCode()) != NULL) && (strcasestr(ptr, src)))
            *co |= i;
        i *= 2;
        n = n->Next();
        }

    n = sect->GetCodeTable()->First();
    i = 1l;
    while((n != NULL) && (i != 0))
        {
        ct = (CodeTableEntry *) n->Data();
        if(((ptr = ct->GetCode()) != NULL) && (strcasestr(ptr, src)))
            *cx |= i;
        i *= 2;
        n = n->Next();
        }

    n = sect->GetSystTable()->First();
    i = 1l;
    while((n != NULL) && (i != 0))
        {
        ct = (CodeTableEntry *) n->Data();
        if(((ptr = ct->GetCode()) != NULL) && (strcasestr(ptr, src)))
            *cs |= i;
        i *= 2;
        n = n->Next();
        }
}

char *
System::ParseTwo(char *buff, int *one, int *two)
{
	buff = ParseOne(buff, one);
	buff = ParseOne(buff, two);
	return(buff);
}

char *
System::ParseOne(char *buff, int *one)
{
char lbuff[20];
int i=0;

	while(((*buff >= '0') && (*buff <= '9')) || isspace(*buff))
		{
		lbuff[i] = *buff;
		buff++;
		i++;
		lbuff[i] = 0;
		}

	sscanf(lbuff, "%d", one);
	return(++buff);
}

char *System::ParseString(char *buff)
{
	while(*buff != ':')
		{
		if(*buff == 0)
			return(buff);
		buff++;
		}

	buff++;
	while((*buff != 0) && (isspace(*buff)))
	    buff++;
	
	return(buff);
}

// WARNING!!!!
// sndx cannot indicate a second tert. or comp. orbit!
bool
System::ValidOrbit(char *buff, unsigned short *sndx, float *orb)
{
//int i;

// quick check
	if(strchr(buff, ':') != NULL)
		return(FALSE);
// check for primary
	*sndx = 0;
	if(strcasestr(buff, "primary") != 0) {
		*orb = 0.0;
		*sndx = ORBIT_IS_PRIMARY;
		return(TRUE);
// FIX: close companion.....
	} if(strcasestr(buff, "close") != 0) {
		*orb = -1.0;
		if(!isspace(buff[orb_parse[ORB_PARSE_PRIMARY]-4]))
		    *sndx = ORBIT_ORBITS_PRIMARY;
		else if(!isspace(buff[orb_parse[ORB_PARSE_SECONDARY]-4]))
		    *sndx = ORBIT_ORBITS_COMP;
		else
		    *sndx = ORBIT_ORBITS_TERT;

		return(TRUE);
	}
// is it even a number?
	if(sscanf(buff, " %f", orb) != 1)
		return(FALSE);

	if(isdigit(buff[orb_parse[ORB_PARSE_PRIMARY]]) &&
		isspace(buff[orb_parse[ORB_PARSE_PRIMARY]+1]))
		*sndx = ORBIT_ORBITS_PRIMARY;
	if(isdigit(buff[orb_parse[ORB_PARSE_PRIMARY_SAT]]) &&
		isspace(buff[orb_parse[ORB_PARSE_PRIMARY_SAT]+1]))
		*sndx = (ORBIT_ORBITS_PRIMARY | ORBIT_IS_SAT);
	if(isdigit(buff[orb_parse[ORB_PARSE_SECONDARY]]) &&
		isspace(buff[orb_parse[ORB_PARSE_SECONDARY]+1]))
		*sndx = ORBIT_ORBITS_COMP;
	if(isdigit(buff[orb_parse[ORB_PARSE_SECONDARY_SAT]]) &&
		isspace(buff[orb_parse[ORB_PARSE_SECONDARY_SAT]+1]))
		*sndx = (ORBIT_ORBITS_COMP | ORBIT_IS_SAT);
	if(isdigit(buff[orb_parse[ORB_PARSE_TERTIARY]]) &&
		isspace(buff[orb_parse[ORB_PARSE_TERTIARY]+1]))
		*sndx = ORBIT_ORBITS_TERT;
	if(isdigit(buff[orb_parse[ORB_PARSE_TERTIARY_SAT]]) &&
		isspace(buff[orb_parse[ORB_PARSE_TERTIARY_SAT]+1]))
		*sndx = (ORBIT_ORBITS_TERT | ORBIT_IS_SAT);

// quick check: did sndx get a good value?
	if(*sndx == 0)
		return(FALSE);
	return(TRUE);
}

bool
System::ParseAndAddOrbit(char *buff)
{
char n[120];		// name
//short sndx;
unsigned short mask;
float orbit;
char u[13];			// uwp
char b;				// base
unsigned long co;	// codes (sector)
unsigned long cx;	// codes (other)
unsigned long cs;	// codes (system)
char z;				// zone
char a[3];			// alleg
int i,j;
char lbuff[120];
planet *p;
DetailStar *primary=NULL;

// quick check
	if((int)strlen(buff) < (start_parse[SYS_PARSE_UWP] + 2))
		return(FALSE);

// check the orbit
	j = 0;
	for(i = start_parse[SYS_PARSE_ORBIT];i < end_parse[SYS_PARSE_ORBIT];i++) {
		n[j] = buff[i];
		j++;
	}
	n[j] = 0;
	if(!ValidOrbit(n, &mask, &orbit))
		return(FALSE);

// last check (is this the primary or do we have a primary?)
	if((prim_orbit == NULL) && ((mask & ORBIT_IS_PRIMARY) == 0))
		return(FALSE);
	
// fell thru, must be good
// start w/ the name
	j = 0;
	for(i = start_parse[SYS_PARSE_NAME];i <= end_parse[SYS_PARSE_NAME];i++) {
		n[j] = buff[i];
		j++;
	}
	n[j] = 0;
	
// grab the UWP
	j = 0;
	for(i = start_parse[SYS_PARSE_UWP];i <= end_parse[SYS_PARSE_UWP];i++) {
		u[j] = buff[i];
		j++;
	}
	u[j] = 0;

// the base
	b = buff[start_parse[SYS_PARSE_BASE]];

// any codes.....
	j = 0;
	for(i = start_parse[SYS_PARSE_CODE];i <= end_parse[SYS_PARSE_CODE];i++) {
		lbuff[j] = buff[i];
		j++;
	}
	lbuff[j] = 0;
	ParseCodes(lbuff, &co, &cx, &cs);

// zone
	z = buff[start_parse[SYS_PARSE_ZONE]];

// alleg
    j = start_parse[SYS_PARSE_ALLEG];
    for(i = 0;i < 2;i++) {
        a[i] = buff[j];
        j++;
    }
    a[i] = 0;

// add it
// can't think of a better way to flag a star.......
//	if(isspace(u[2]))
	if((u[2] == 'V') || (u[2] == 'I') ||
		(u[3] == 'V') || (u[3] == 'I')) {
		mask |= ORBIT_IS_STAR;
		if(mask & ORBIT_IS_PRIMARY) {			
			primary = new DetailStar(n, u);
			prim_orbit = new PrimOrbit(mask, primary);
		} else {
			DetailStar *star;
			star = new DetailStar(n, u);
			prim_orbit->AppendToOrbitTable(mask, orbit, star); 
		}
	} else {
		if((mask & ORBIT_IS_SAT) == 0)
		    mask |= ORBIT_IS_WORLD;
		p = new planet(n, u, b, co, cx, cs, z, a);
		prim_orbit->AppendToOrbitTable(mask, orbit, p);
		//	dump_parse(start_parse, end_parse);
	}

//// initialize this to the begining
//	cur_orbit = prim_orbit;

	return(TRUE);
}

int
System::LoadFile(char *n)
{
FILE *fpx;
int count=0;
char buff[BIG_BUFFER_SIZE];
int i,j;
//int have_name_flag=0;
char *ptr;

	if(file_name != NULL)
		{
		delete file_name;
		file_name = NULL;
		}
		
	if((fpx = fopen(n, "r")) == NULL)
		return(-1);

	Clear();
	while(fgets(buff, BIG_BUFFER_SIZE, fpx) != NULL) {
		// strip off the ^%^*&*%^&* new line
		if(buff[strlen(buff) - 1] == '\n')
			buff[strlen(buff) - 1] = 0;
		strstrip(buff, TRUE);
		// do anything?
		if((strlen(buff) == 0) || (buff[0] == '#')) {
			// make sure it's clean for next pass
			for(i = 0;i < BIG_BUFFER_SIZE;i++)
			    buff[i] = ' ';
			continue;
		}

		// try to parse it
		if(ParseAndAddOrbit(buff))
			count++;

		// didn't parse, is it info?
		// system name?
		else if(strstr(buff, "System")) {
			ptr = ParseString(buff);
			name = new char[strlen(ptr)+1];
			strcpy(name, ptr);
		// ss name?
		} else if(strstr(buff, "SubSector")) {
			ptr = ParseString(buff);
			ss_name = new char[strlen(ptr) + 1];
			strcpy(ss_name, ptr);
		// Sector name?
		} else if(strstr(buff, "Sector")) {
			ptr = ParseString(buff);
			sect_name = new char[strlen(ptr) + 1];
			strcpy(sect_name, ptr);
		// loc?
		} else if(strstr(buff, "Location")) {
			ptr = ParseString(buff);
			loc = new char[strlen(ptr) + 1];
			strcpy(loc, ptr);
		// is it format info?
		// Orbits?
		} else if(strstr(buff, "Orbit(s)")) {
			ParseTwo(buff, &i, &j);
			start_parse[SYS_PARSE_ORBIT] = i - 1;
			end_parse[SYS_PARSE_ORBIT] = j - 1;
		// Name?
		} else if(strstr(buff, "Name")) {
			ParseTwo(buff, &i, &j);
			start_parse[SYS_PARSE_NAME] = i - 1;
			end_parse[SYS_PARSE_NAME] = j - 1;
		// UWP?
		} else if(strstr(buff, "UWP")) {
			ParseTwo(buff, &i, &j);
			start_parse[SYS_PARSE_UWP] = i - 1;
			end_parse[SYS_PARSE_UWP] = j - 1;
		// Bases?
		} else if(strstr(buff, "Bases")) {
			ParseOne(buff, &i);
			start_parse[SYS_PARSE_BASE] = i - 1;
		// Codes?
		} else if(strstr(buff, "Codes")) {
			ParseTwo(buff, &i, &j);
			start_parse[SYS_PARSE_CODE] = i - 1;
			end_parse[SYS_PARSE_CODE] = j - 1;
		// Zone?
		} else if(strstr(buff, "Zone")) {
			ParseOne(buff, &i);
			end_parse[SYS_PARSE_ZONE] = start_parse[SYS_PARSE_ZONE] = i - 1;
		// Alleg?
		} else if(strstr(buff, "Alleg")) {
			ParseTwo(buff, &i, &j);
			start_parse[SYS_PARSE_ALLEG] = i - 1;
			end_parse[SYS_PARSE_ALLEG] = j - 1;
		// Primary Orbit?
		} else if(strstr(buff, "Primary Orbit")) {
			ParseOne(buff, &i);
			orb_parse[ORB_PARSE_PRIMARY] = i - 1;
		// Primary Satillite Orbit?
		} else if(strstr(buff, "Primary Satillite Orbit")) {
			ParseOne(buff, &i);
			orb_parse[ORB_PARSE_PRIMARY_SAT] = i - 1;
		// Primary Companion Orbit?
		} else if(strstr(buff, "Primary Companion Orbit")) {
			ParseOne(buff, &i);
			orb_parse[ORB_PARSE_SECONDARY] = i - 1;
		// Primary Companion Satillite Orbit?
		} else if(strstr(buff, "Primary Companion Satillite Orbit")) {
			ParseOne(buff, &i);
			orb_parse[ORB_PARSE_SECONDARY_SAT] = i - 1;
		// Secondary Companion Orbit?
		} else if(strstr(buff, "Secondary Companion Orbit")) {
			ParseOne(buff, &i);
			orb_parse[ORB_PARSE_TERTIARY] = i - 1;
		// Secondary Companion Satillite Orbit?
		} else if(strstr(buff, "Secondary Companion Satillite Orbit")) {
			ParseOne(buff, &i);
			orb_parse[ORB_PARSE_TERTIARY_SAT] = i - 1;
		// unknown!
		}

		// make sure it's clean for next pass
		for(i = 0;i < BIG_BUFFER_SIZE;i++)
			buff[i] = ' ';
	}

	fclose(fpx);
	if(count > 0) {
		file_name = new char[strlen(n) + 1];
		strcpy(file_name, n);
	}

// initialize this to the begining
	cur_orbit = prim_orbit;

//	dump_parse(start_parse, end_parse);
	
	return(count);
}

// ---------------------------------------------------------------------------
bool
System::SaveSystem(char *s)
{
FILE *fpx;

    if((fpx = fopen(s, "w")) != NULL)
	    {
		int i;
		char *ptr;
		char buff[BIG_BUFFER_SIZE]; //,buff2[BIG_BUFFER_SIZE];

		if((ptr = GetName()) != NULL)
		    fprintf(fpx, "    %s ", ptr);
		else
		    fprintf(fpx, "    Unnamed ");
		if((ptr = GetLoc()) != NULL)
		    fprintf(fpx, ptr);
		fprintf(fpx, "\nSystem: ");
		if((ptr = GetName()) != NULL)
		    fprintf(fpx, ptr);
		else
		    fprintf(fpx, "Unnamed");
		fprintf(fpx, "\nSector: ");
		if((ptr = GetSectName()) != NULL)
		    fprintf(fpx, ptr);
		else
		    fprintf(fpx, "Unknown");
		fprintf(fpx, "\nSubSector: ");
		if((ptr = GetSSName()) != NULL)
		    fprintf(fpx, ptr);
		else
		    fprintf(fpx, "Unknown");
		fprintf(fpx, "\nLocation: ");
		if((ptr = GetLoc()) != NULL)
		    fprintf(fpx, ptr);
		else
		    fprintf(fpx, "Unknown");
		
		fprintf(fpx, "\n\nThe data in the system text files is laid out in column format:\n");
		fprintf(fpx, " %d-%d: Orbit(s)\n",
				start_parse[0] + 1, end_parse[0] + 1);
		fprintf(fpx, "%d-%d: Name\n",
				start_parse[1] + 1, end_parse[1] + 1);
		fprintf(fpx, "%d-%d: UWP\n",
				start_parse[2] + 1, end_parse[2] + 1);
		fprintf(fpx, "   %d: Bases\n",
				start_parse[3] + 1);
		fprintf(fpx, "%d-%d: Codes & Comments\n",
				start_parse[4] + 1, end_parse[4] + 1);
		fprintf(fpx, "   %d: Travel Zone (optional)\n",
				start_parse[5] + 1);
		fprintf(fpx, "%d-%d: Allegiance (optional)\n",
				start_parse[6] + 1, end_parse[6] + 1);


		fprintf(fpx, "\nThe orbits are formated by the last position of it's number as follows:\n");
		fprintf(fpx, "%d: Primary Orbit\n",
				orb_parse[0] + 1);
		fprintf(fpx, "%d: Primary Satillite Orbit",
				orb_parse[1] + 1);
		fprintf(fpx, "%d: Primary Companion Orbit\n",
				orb_parse[2] + 1);
		fprintf(fpx, "%d: Primary Companion Satillite Orbit\n",
				orb_parse[3] + 1);
		fprintf(fpx, "%d: Secondary Companion Orbit\n",
				orb_parse[4] + 1);
		fprintf(fpx, "%d: Secondary Companion Satillite Orbit\n",
				orb_parse[5] + 1);

		fprintf(fpx, "\n\n....+....1....+....2....+....3....+....4....+....5....+....6....+....7....+....8\n12345678901234567890123456789012345678901234567890123456789012345678901234567890\n");

		i = 0;
		while(GetListing(i, buff) != NULL)
		    {
			strstrip(buff, TRUE);
			fprintf(fpx, "%s\n", buff);
			i++;
		    }

		fclose(fpx);
		return(TRUE);
		}

    return(FALSE);
}

