//
// File:		mtu_file.cpp
// Purpose:		handle all dirs and files
// Author:		Mark A. Nordstrand
// Created:		Sun Mar 17 07:02:29 CST 2002
// Updated:
// Copyright:	LGPL
// Traveller is a registered trademark of Far Future Enterprises.
// Portions based upon material Copyright 1977-2002 Far Future Enterprises.
//
// rcsid[] = "$RCSfile: mtu_file.cpp,v $ $Revision: 1.13 $ $Author: man $ $Date: 2002/10/09 02:19:02 $"

#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <libgen.h>
#include "wx/confbase.h"
#include "wx/fileconf.h"
#include "mtu_file.h"
#include "t_res.h"

/*
typical layout (and stuff we'll need to find):

<data dir>
	|
	+<mtu dir>
		|
		+- mtu.dat 					(main mtu file)
		+- <universe>
		 	+- <unverse>.dat		(main universe file)
		 	+- <sector>
		 	  	+- <sector>.sec	(main sector file)
			  	+- <sector>.dat	(aux sector file)
			  	+- desc
			  	  	+- sector.txt	(sector desc)
			  	  	+- <ssec>.txt	(subsector desc)
			  	  	+- <loc>.txt	(location desc)
			  	+- enc
			  	  	+- <loc>.enc	(location enc table)
			  	+- sys
			  	  	+- <loc>.sys	(location system)
			  	+- world
			  	  	+- <loc>.wld	(location world detail)
			  	+- trade
			  	  	+- <loc>.trd	(location trade detail)
		+- data
		 	+- (data files)

BLOAT WARNING:
the most hideous/insideous thing about this is having to drag all the
pieces around in every executable.  world, for instance, has no need
for the encounter table dir.  on the other hand, I'm tired of writing 
and fixing each individual program's handling of resources and strings,
so I've shoved all the common stuff into one location.  on the other
other hand, these strings shouldn't be longer than MAX_FILE_LENGTH (or
whatever it's called), and all totaled shouldn't be much bigger than
a typical sector/data file

one other note:  one may wonder why there isn't a write or save method.
  fact is, there is only one place this needs to be done (mtu_dlg.cpp),
  and it has already been written (and debugged), and it ain't broke so,
  it won't be 'fixed.'

*/

// XXX fairly unix specific values
#define DIR_DELIM		'/'
#define EXT_DELIM		'.'
#define NOTES_DIR		"desc"
#define NOTES_EXT		"txt"
#define ENC_DIR			"enc"
#define ENC_EXT			"enc"
#define TRADE_DIR		"trade"
#define TRADE_EXT		"trd"
#define SYS_DIR			"sys"
#define SYS_EXT			"sys"
#define WORLD_DIR		"world"
#define WORLD_EXT		"wld"
#define DATA_DIR		"data"
#define DATA_EXT		"dat"
#define SECTOR_EXT		"sec"
#define MTU_DATA_FILE	"mtu.dat"

#define MIN_STRLEN		2
#define DEFAULT_PERM	0755

// ============================================================================
MTUFile::MTUFile()
{
	data_dir = NULL;
	exec_dir = NULL;
	help_dir = NULL;
	mtu_dir = NULL;
	sector = NULL;
	system = NULL;
	sysgen = NULL;
	world = NULL;
	enc_tbl = NULL;
	word = NULL;
	trade = NULL;
	secgen = NULL;
	editor = NULL;
	graphics = NULL;
	viewer = NULL;
	mtu_data = NULL;
	univ_dir = NULL;
	univ_file = NULL;
	sect_dir = NULL;
	sect_sec = NULL;
	sect_dat = NULL;
	sect_text = NULL;
	sect_enc = NULL;
	sect_sys = NULL;
	sect_world = NULL;
	sect_trade = NULL;

	config = new wxConfig((const wxString) RESOURCE_FILE);
	wxConfigBase::Set(config);
	wxConfigBase::Get();

	Load();
}

MTUFile::~MTUFile()
{
	if(data_dir != NULL)
		delete data_dir;
	if(exec_dir != NULL)
		delete exec_dir;
	if(help_dir != NULL)
		delete help_dir;
	if(mtu_dir != NULL)
		delete mtu_dir;
	if(sector != NULL)
		delete sector;
	if(system != NULL)
		delete system;
	if(sysgen != NULL)
		delete sysgen;
	if(world != NULL)
		delete world;
	if(enc_tbl != NULL)
		delete enc_tbl;
	if(word != NULL)
		delete word;
	if(trade != NULL)
		delete trade;
	if(secgen != NULL)
		delete secgen;
	if(editor != NULL)
		delete editor;
	if(graphics != NULL)
		delete graphics;
	if(viewer != NULL)
		delete viewer;
	if(mtu_data != NULL)
		delete mtu_data;
	if(univ_dir != NULL)
		delete univ_dir;
	if(univ_file != NULL)
		delete univ_file;
	if(sect_dir != NULL)
		delete sect_dir;
	if(sect_sec != NULL)
		delete sect_sec;
	if(sect_dat != NULL)
		delete sect_dat;
	if(sect_text != NULL)
		delete sect_text;
	if(sect_enc != NULL)
		delete sect_enc;
	if(sect_sys != NULL)
		delete sect_sys;
	if(sect_world != NULL)
		delete sect_world;
	if(sect_trade != NULL)
		delete sect_trade;

	delete wxConfigBase::Set((wxConfigBase *) NULL);
}

// ----------------------------------------------------------------------------
void
MTUFile::Load(void)
{
char *ptr;

	// XXX should the config file be re-loaded too?
	GetRString(&data_dir, SECTION_DATA, ENTRY_DATA_DIR);
	GetRString(&exec_dir, SECTION_EXECUTE, ENTRY_DIR);
	GetRString(&help_dir, SECTION_HELP, ENTRY_HELP_DIR);
	// special case:  normally, mtu_dir is only a subdir under data_dir.
	//   however, if the user gives a full path name, don't prepend data_dir.
	ptr = NULL;
	GetRString(&ptr, SECTION_DATA, ENTRY_MTU_DIR);
	if(ptr != NULL) {
		if(DIR_DELIM == ptr[0])
			mtu_dir = ptr;
		else {
			mtu_dir = BuildString(data_dir, ptr);
			delete ptr;
		}
	}

	// grab the executables
	GetEString(&sector, SECTION_EXECUTE, ENTRY_SECTOR);
	GetEString(&sysgen, SECTION_EXECUTE, ENTRY_SYSGEN);
	GetEString(&system, SECTION_EXECUTE, ENTRY_SYSTEM);
	GetEString(&world, SECTION_EXECUTE, ENTRY_WORLD);
	GetEString(&enc_tbl, SECTION_EXECUTE, ENTRY_ENC_TBL);
	GetEString(&word, SECTION_EXECUTE, ENTRY_WORD);
	GetEString(&trade, SECTION_EXECUTE, ENTRY_TRADE);
	GetEString(&secgen, SECTION_EXECUTE, ENTRY_SECGEN);
	// another special case.  editor should be available via the user's
	//  PATH=, so it doesn't need an to be built with the path (although
	//  the user could put it in)
	GetRString(&editor, SECTION_EXECUTE, ENTRY_EDITOR);
	GetRString(&graphics, SECTION_EXECUTE, ENTRY_GRAPHICS);
	GetRString(&viewer, SECTION_HELP, ENTRY_HELP_VIEWER);

	// this is the only one of these we can be sure of right now
	if(mtu_data != NULL) {			// shouldn't be...
		delete mtu_data;
		mtu_data = NULL;
	}
	mtu_data = BuildString(mtu_dir, DATA_DIR);

	ClearExtra(TRUE);
}

// read a string from the resources
void
MTUFile::GetRString(char **dest, char *section, char *entry)
{
char buffer[256];
wxString str;

	if(*dest != NULL) {
		delete *dest;
		*dest = NULL;
	}

	sprintf(buffer, "/%s/%s", section, entry);
	if(config->Read((const wxString) buffer, &str)) {
		*dest = new char[strlen(str.GetData()) + 1];
		strcpy(*dest, str.GetData());
	}
}

// like GetRString, but prepends exec_dir (if it exists)
void
MTUFile::GetEString(char **dest, char *section, char *entry)
{
char *ptr=NULL;

	GetRString(&ptr, section, entry);
	if(ptr != NULL) {
		*dest = BuildString(exec_dir, ptr);
		delete ptr;
	}
}

// cat some strings together (w/ checking and other features)
char *
MTUFile::BuildString(char *ptr1, char *ptr2)
{
char *ret=NULL,lbuff[4];
int len=0;

	if((ptr1 != NULL) && (ptr1[0] != 0)) {
		len += strlen(ptr1);
		if(ptr1[len-1] != DIR_DELIM) {
			len++;
			sprintf(lbuff, "%c", DIR_DELIM);
		} else
			lbuff[0] = 0;
	}

	if((ptr2 != NULL) && (ptr2[0] != 0))
		len += strlen(ptr2);

	if(len > 0) {
		ret = new char[len + 1];
		ret[0] = 0;
		if((ptr1 != NULL) && (ptr1[0] != 0))
			sprintf(ret, "%s%s", ptr1, lbuff);
		if((ptr2 != NULL) && (ptr2[0] != 0))
			strcat(ret, ptr2);
	}

	return(ret);
}

// fill in 'extra' strings
void 
MTUFile::BuildUniv(char *u_dir_name)
{
char lbuff[4];

	ClearExtra(TRUE);
	if(u_dir_name != NULL) {
		lbuff[0] = 0;
		if(mtu_dir != NULL) {
			if(mtu_dir[strlen(mtu_dir) - 1] != DIR_DELIM)
				sprintf(lbuff, "%c", DIR_DELIM);
			univ_dir = new char[strlen(mtu_dir) + strlen(lbuff) + 
					strlen(u_dir_name) + 1];
			sprintf(univ_dir, "%s%s%s", mtu_dir, lbuff, u_dir_name);
			univ_file = new char[strlen(univ_dir) +
					strlen(u_dir_name) + strlen(DATA_EXT) + 3];
			sprintf(univ_file, "%s%c%s%c%s", univ_dir, DIR_DELIM, u_dir_name,
					EXT_DELIM, DATA_EXT);
		} else {
			// XXX
			univ_dir = new char[strlen(u_dir_name) + 1];
			strcpy(univ_dir, u_dir_name);
			univ_file = new char[strlen(MTU_DATA_FILE) + 1];
			strcpy(univ_file, MTU_DATA_FILE);
		}
	}
}

// NOTE:  the sector stuff needs to work a little bit differently.
//   the capability to run the programs 'stand-alone' needs to be
//   preserved.  to accomplish this, look if the string passed to
//   BuildSect() is an absolute pathname.  if it is, build the full
//   strings using univ_dir.  otherwise, just build the dirs as "".
void 
MTUFile::BuildSect(bool abs, char *s_dir_name)
{
	ClearExtra(FALSE);
	if(s_dir_name != NULL) {
//		if(DIR_DELIM == s_dir_name[0]) {
		if(abs) {
			char *ptr1,lbuff[4];

			ptr1 = univ_dir;
			if(NULL == ptr1)
				ptr1 = "";
			lbuff[0] = 0;
			if(ptr1[strlen(ptr1) - 1] != DIR_DELIM)
				sprintf(lbuff, "%c", DIR_DELIM);

			sect_dir = new char[strlen(ptr1) + strlen(s_dir_name) + 
					strlen(lbuff) + 1];
			sprintf(sect_dir, "%s%s%s", ptr1, lbuff, s_dir_name);
			sect_sec = new char[strlen(sect_dir) + strlen(s_dir_name) +
					strlen(SECTOR_EXT) + 3];
			sprintf(sect_sec, "%s%c%s%c%s", sect_dir, DIR_DELIM, s_dir_name,
					EXT_DELIM, SECTOR_EXT);
			sect_dat = new char[strlen(sect_dir) + strlen(s_dir_name) +
					strlen(DATA_EXT) + 3];
			sprintf(sect_dat, "%s%c%s%c%s", sect_dir, DIR_DELIM, s_dir_name,
					EXT_DELIM, DATA_EXT);
			sect_text = new char[strlen(sect_dir) + strlen(NOTES_DIR) + 2];
			sprintf(sect_text, "%s%c%s", sect_dir, DIR_DELIM, NOTES_DIR);
			sect_enc = new char[strlen(sect_dir) + strlen(ENC_DIR) + 2];
			sprintf(sect_enc, "%s%c%s", sect_dir, DIR_DELIM, ENC_DIR);
			sect_sys = new char[strlen(sect_dir) + strlen(SYS_DIR) + 2];
			sprintf(sect_sys, "%s%c%s", sect_dir, DIR_DELIM, SYS_DIR);
			sect_world = new char[strlen(sect_dir) + strlen(WORLD_DIR) + 2];
			sprintf(sect_world, "%s%c%s", sect_dir, DIR_DELIM, WORLD_DIR);
			sect_trade = new char[strlen(sect_dir) + strlen(WORLD_DIR) + 2];
			sprintf(sect_trade, "%s%c%s", sect_dir, DIR_DELIM, TRADE_DIR);
		} else {
			sect_dir = new char[MIN_STRLEN];
			sect_dir[0] = 0;
			sect_sec = new char[strlen(s_dir_name) + strlen(SECTOR_EXT) + 2];
			sprintf(sect_sec, "%s%c%s", s_dir_name, DIR_DELIM, SECTOR_EXT);
			sect_dat = new char[strlen(s_dir_name) + strlen(SECTOR_EXT) + 2];
			sprintf(sect_dat, "%s%c%s", s_dir_name, DIR_DELIM, DATA_EXT);
			sect_text = new char[MIN_STRLEN];
			sect_text[0] = 0;
			sect_enc = new char[MIN_STRLEN];
			sect_enc[0] = 0;
			sect_sys = new char[MIN_STRLEN];
			sect_sys[0] = 0;
			sect_world = new char[MIN_STRLEN];
			sect_world[0] = 0;
			sect_trade = new char[MIN_STRLEN];
			sect_trade[0] = 0;
		}
	}
}

void
MTUFile::ClearExtra(bool u)
{
	if(u) {
		if(univ_dir != NULL) {
			delete univ_dir;
			univ_dir = NULL;
		}
		if(univ_file != NULL) {
			delete univ_file;
			univ_file = NULL;
		}
	}
	if(sect_dir != NULL) {
		delete sect_dir;
		sect_dir = NULL;
	}
	if(sect_sec != NULL) {
		delete sect_sec;
		sect_sec = NULL;
	}
	if(sect_dat != NULL) {
		delete sect_dat;
		sect_dat = NULL;
	}
	if(sect_text != NULL) {
		delete sect_text;
		sect_text = NULL;
	}
	if(sect_enc != NULL) {
		delete sect_enc;
		sect_enc = NULL;
	}
	if(sect_sys != NULL) {
		delete sect_sys;
		sect_sys = NULL;
	}
	if(sect_world != NULL) {
		delete sect_world;
		sect_world = NULL;
	}
	if(sect_trade != NULL) {
		delete sect_trade;
		sect_trade = NULL;
	}
}

// ----------------------------------------------------------------------------
// given a sector or system name, pull out the universe and sector portion
void
MTUFile::CrackNames(char *n)
{
char *univ=NULL,*sect=NULL;

	ClearExtra(TRUE);
	if((n != NULL) && (DIR_DELIM == n[0]) && 
				(strncmp(mtu_dir, n, strlen(mtu_dir)) == 0)) {
		char *ptr1,*ptr2,*ptr3;
		int i;

		ptr1 = new char[strlen(n) + 1];
		strcpy(ptr1, n);
		// we really, really should have this:
		i = strlen(mtu_dir);
		ptr2 = &ptr1[i];
		while(DIR_DELIM == *ptr2)
			ptr2++;
		ptr3 = strchr(ptr2, DIR_DELIM);
		while(DIR_DELIM == *ptr3) {
			*ptr3 = 0;
			ptr3++;
		}

		univ = new char[strlen(ptr2) + 1];
		strcpy(univ, ptr2);
		ptr2 = strchr(ptr3, DIR_DELIM);
		*ptr2 = 0;
		sect = new char[strlen(ptr3) + 1];
		strcpy(sect, ptr3);
		delete ptr1;
	}

	BuildUniv(univ);
	BuildSect(DIR_DELIM == n[0], sect);
	if(univ != NULL)
		delete univ;
	if(sect != NULL)
		delete sect;
}

// ===========================================================================
//  blindly set up required dirs
void
MTUFile::VerifyDirs(unsigned int which)
{
	if((which & MTUD_NOTES) && (sect_text != NULL)) {
		mkdir(sect_text, DEFAULT_PERM);
	}

	if((which & MTUD_SYS) && (sect_sys != NULL)) {
		mkdir(sect_sys, DEFAULT_PERM);
	}

	if((which & MTUD_ENC) && (sect_enc != NULL)) {
		mkdir(sect_enc, DEFAULT_PERM);
	}

	if((which & MTUD_WORLD) && (sect_world != NULL)) {
		mkdir(sect_world, DEFAULT_PERM);
	}

	if((which & MTUD_TRADE) && (sect_trade != NULL)) {
		mkdir(sect_trade, DEFAULT_PERM);
	}
}

char *
MTUFile::CheckDir(char *dir)
{
	if((dir != NULL) && (dir[0] != 0)) {
		mkdir(dir, DEFAULT_PERM);
	}

	return(dir);
}

