/*
 * File:      gal2mtu.c
 * Purpose:   an attemt to transfer (some of) galactic data files into
 *				Mark's Traveller Utilities (MTU)
 * Author:	
 * Created:	
 * Updated:	
 * Copyright:	LGPL
Traveller is a registered trademark of Far Future Enterprises.
Portions based upon material Copyright 1977-2002 Far Future Enterprises.
 */

/* rcsid[] = "$RCSfile: gal2mtu.c,v $ $Revision: 1.5 $ $Author: man $ $Date: 2003/02/03 03:03:37 $" */

#include "conv.h"
#include <errno.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/types.h>
#include <fcntl.h>
#include "str_util.h"

#define HEADER_LINE "################################################################################\n"

/*
 * convert a sector
 */
int
conv_sector(char *name, char *gal_dir, char *mtu_dir)
{
FILE *gal_fp,*sec_fp,*dat_fp;
char buffer1[BUFFER_SIZE];
char buffer2[BUFFER_SIZE];
int i,j;
int ret;

	sprintf(buffer1, "%s/%s.dat", gal_dir, name);
	if((gal_fp = fopen(buffer1, "r")) == NULL) {
//		fprintf(stderr, "Cannot open %s for reading(%d).\n", buffer1, errno);
		return(0);
	}
	
	if(mkdir(mtu_dir, 0775) < 0) {
		if(EEXIST != errno) {
			fprintf(stderr, "Cannot create %s (%d).\n", mtu_dir, errno);
			return(0);
		}
	}

	sprintf(buffer1, "%s/%s.sec", mtu_dir, name);
	if((sec_fp = fopen(buffer1, "a+")) == NULL) {
		fprintf(stderr, "Cannot open %s for writing(%d).\n", buffer1, errno);
		fclose(gal_fp);
		return(0);
	}
	
	sprintf(buffer1, "%s/%s.dat", mtu_dir, name);
	if((dat_fp = fopen(buffer1, "a+")) == NULL) {
		fprintf(stderr, "Cannot open %s for writing(%d).\n", buffer1, errno);
		fclose(gal_fp);
		fclose(sec_fp);
		return(0);
	}

	if(parse_galactic(gal_dir, gal_fp, sec_fp, dat_fp, mtu_dir, name) == 0) {
		fprintf(stderr, "Problems processing %s.\n", gal_dir);
		return(0);
	}

/* thsese should be handled properly by the menu conversion
	sprintf(buffer1, "%s/credits.txt", gal_dir);
	sprintf(buffer2, "%s/credits.txt", mtu_dir);
	if(!conv_text_file(buffer1, buffer2, 1))
		return(0);
*/
	sprintf(buffer2, "%s/desc", mtu_dir);
	mkdir(buffer2, 0775);
/*
	for(i = 0;i < 32;i++) {
		for(j = 0;j < 40;j++) {
			sprintf(buffer1, "%s/loc/p%02d%02d.txt", gal_dir, i, j);
			sprintf(buffer2, "%s/desc/%02d%02d.txt", mtu_dir, i, j);
			conv_text_file(buffer1, buffer2, 0);
		}
	}
*/
	for(i = 0;i < 16;i++) {
		/* this seems common */
		sprintf(buffer1, "%s/loc/sub-%c.txt", gal_dir, i + 'a');
		sprintf(buffer2, "%s/desc/sub-%c.txt", mtu_dir, i + 'a');
		conv_text_file(buffer1, buffer2, 0);
	}
	
	sprintf(buffer2, "%s/old_sys", mtu_dir);
	mkdir(buffer2, 0775);
	for(i = 0;i < 32;i++) {
		for(j = 0;j < 40;j++) {
			sprintf(buffer1, "%s/sys/s%02d%02d.txt", gal_dir, i, j);
			sprintf(buffer2, "%s/old_sys/%02d%02d.txt", mtu_dir, i, j);
			conv_text_file(buffer1, buffer2, 0);
		}
	}

	fclose(gal_fp);
	fclose(sec_fp);
	fclose(dat_fp);

	sprintf(buffer1, "%s/gen/sector.mnu", gal_dir);
	if((gal_fp = fopen(buffer1, "r")) == NULL) {
		fprintf(stderr, "Cannot open %s for reading(%d).\n", buffer1, errno);
	} else {
		char gal_buff[BUFFER_SIZE],mtu_buff[BUFFER_SIZE],*ptr;
		FILE *mtu_fp;

//fprintf(stderr, "--><%s> <%s>\n", gal_dir, mtu_dir);
		strcpy(gal_buff, gal_dir);
		ptr = strrchr(gal_buff, '/');
		*ptr = 0;
		ptr = strrchr(gal_buff, '/');
		*ptr = 0;
		strcpy(mtu_buff, mtu_dir);
		ptr = strrchr(mtu_buff, '/');
		*ptr = 0;
		ptr = strrchr(mtu_buff, '/');
		*ptr = 0;
		sprintf(buffer1, "%s/menu/", mtu_dir);
		mkdir(buffer1, 0775);
		sprintf(buffer1, "%s/menu/sector.menu", mtu_dir);
		if((mtu_fp = fopen(buffer1, "a+")) == NULL) {
			fprintf(stderr, "Cannot open %s for writing(%d).\n", buffer1,errno);
			fclose(gal_fp);
			return(0);
		}

//fprintf(stderr, "----><%s> <%s>\n", gal_buff, mtu_buff);
		convert_menu(gal_fp, gal_buff, mtu_fp, mtu_buff);
		fclose(gal_fp);
		fclose(mtu_fp);
	}

	return(1);
}

/*
 * convert a galaxy
 */
int 
conv_dir(char *name, char *gal_dir, char *mtu_dir)
{
FILE *gal_fp,*mtu_fp;
char buffer[BUFFER_SIZE];
int line_count=0;

	sprintf(buffer, "%s/%s.lst", gal_dir, name);
	if((gal_fp = fopen(buffer, "r")) == NULL) {
		fprintf(stderr, "Cannot open %s for reading(%d).\n", buffer, errno);
		return(0);
	}

	/* this failing is not an error */
	mkdir(mtu_dir, 0775);
	sprintf(buffer, "%s/%s.dat", mtu_dir, name);
	/* but, this is */
	if((mtu_fp = fopen(buffer, "a+")) == NULL) {
		fprintf(stderr, "Cannot open %s for writing(%d).\n", buffer, errno);
		fclose(gal_fp);
		return(0);
	}

	fprintf(mtu_fp, HEADER_LINE);
	fprintf(mtu_fp, "## MTU Sectors for the %s universe\n", name);

	while(read_line(gal_fp, buffer) != NULL) {
		char *dir,*menu,*x,*y,code,gal_buff[BUFFER_SIZE],mtu_buff[BUFFER_SIZE];

		line_count++;
		if(line_count < 3)
			continue;
		code = buffer[60];
		buffer[59] = 0;
		y = &buffer[55];
		buffer[54] = 0;
		x = &buffer[50];
		buffer[49] = 0;
		menu = &buffer[13];
		buffer[12] = 0;
		dir = buffer;
		strstrip(x, 1);
		strstrip(y, 1);
		strstrip(menu, 1);
		strstrip(dir, 1);
		sprintf(gal_buff, "%s/%s", gal_dir, dir);
		sprintf(mtu_buff, "%s/%s", mtu_dir, dir);
		conv_sector(dir, gal_buff, mtu_buff);
		fprintf(mtu_fp, "%s:%s:%s:%s:%c\n", dir, menu, x, y, code);
	}
	fclose(gal_fp);
	fclose(mtu_fp);

	/* try the menus (reuse fp) */
	sprintf(buffer, "%s/gen/galaxy.mnu", gal_dir);
	if((gal_fp = fopen(buffer, "r")) == NULL) {
		fprintf(stderr, "Cannot open %s for reading(%d).\n", buffer, errno);
	} else {
		char gal_buff[BUFFER_SIZE],mtu_buff[BUFFER_SIZE],*ptr;

		strcpy(gal_buff, gal_dir);
		ptr = strrchr(gal_buff, '/');
		*ptr = 0;
		strcpy(mtu_buff, mtu_dir);
		ptr = strrchr(mtu_buff, '/');
		*ptr = 0;
		sprintf(buffer, "%s/menu/", mtu_dir);
		mkdir(buffer, 0775);
		sprintf(buffer, "%s/menu/universe.menu", mtu_dir);
		if((mtu_fp = fopen(buffer, "a+")) == NULL) {
			fprintf(stderr, "Cannot open %s for writing(%d).\n", buffer, errno);
			fclose(gal_fp);
			return(0);
		}

		convert_menu(gal_fp, gal_buff, mtu_fp, mtu_buff);
		fclose(gal_fp);
		fclose(mtu_fp);
	}

	return(1);
}

/* convert the galactic star generation file
 * NOTE: most of the fields originally described are redundant 
 * 		(handled differently?)
 */
int
conv_star_table(FILE *gal_fp, char *mtu_dir)
{
FILE *mtu_fp;
int line_count=0;
char buffer[BUFFER_SIZE];

	sprintf(buffer, "%s/%s/%s", mtu_dir, "data", "stargen.dat");
	if((mtu_fp = fopen(buffer, "a+")) == NULL) {
		fprintf(stderr, "Cannot open %s for writing(%d).\n", buffer, errno);
		return(0);
	}
	fprintf(mtu_fp, HEADER_LINE);
	fprintf(mtu_fp, "##\n## stargen table\n##  format:\n");
	fprintf(mtu_fp, "## TS:P:N\n##  TS is type and size\n");
	fprintf(mtu_fp, "##  P is the probablility (0 is ok)\n");
	fprintf(mtu_fp, "##  N is the number of planets ##\n");
	fprintf(mtu_fp, HEADER_LINE);

	while(read_line(gal_fp, buffer) != NULL) {
		if(buffer[0] != '#') {
			char *ptr1,*ptr2;

			ptr1 = &buffer[26];
			buffer[34] = 0;
			strstrip(ptr1, 0);
			ptr2 = &buffer[42];
			buffer[51] = 0;
			strstrip(ptr2, 0);

			fprintf(mtu_fp, "%c%c:%s:%s\n", buffer[0], buffer[1], ptr1, ptr2);
			line_count++;
		}
	}

	fclose(mtu_fp);
	return(line_count);
}

/* convert the planet generation file
 */
int
conv_planet_table(FILE *gal_fp, char *mtu_dir)
{
FILE *mtu_fp;
int line_count=0;
char buffer[BUFFER_SIZE];

	sprintf(buffer, "%s/%s/%s", mtu_dir, "data", "sysgen.dat");
	if((mtu_fp = fopen(buffer, "a+")) == NULL) {
		fprintf(stderr, "Cannot open %s for writing(%d).\n", buffer, errno);
		return(0);
	}
	fprintf(mtu_fp, HEADER_LINE);
	fprintf(mtu_fp, "## planet table\n##  format:\n");
	fprintf(mtu_fp, "## desc:I:B:O:Moons\n##  desc is a description\n");
	fprintf(mtu_fp, "##  I B O are chances for Inner Bio and Outer zones\n");
	fprintf(mtu_fp, "##  Moons are the number of moons\n");
	fprintf(mtu_fp, "##\n");
	fprintf(mtu_fp, HEADER_LINE);

	while(read_line(gal_fp, buffer) != NULL) {
		if(buffer[0] != '#') {
			char *ptr1,*ptr2,*ptr3,*ptr4;

			buffer[26] = 0;
			strstrip(buffer, 1);
			ptr1 = &buffer[27];
			buffer[29] = 0;
			strstrip(ptr1, 0);
			ptr2 = &buffer[34];
			buffer[37] = 0;
			strstrip(ptr2, 0);
			ptr3 = &buffer[42];
			buffer[45] = 0;
			strstrip(ptr3, 0);
			ptr4 = &buffer[46];
			strstrip(ptr4, 0);

			fprintf(mtu_fp, "%s:%s:%s:%s:%s\n", &buffer[6], ptr1, ptr2,
				ptr3, ptr4);
			line_count++;
		}
	}

	fclose(mtu_fp);
	return(line_count);
}

/* convert the galactic jump table (for sect gen)
 */
int
conv_jump_table(FILE *gal_fp, char *mtu_dir)
{
FILE *mtu_fp;
int line_count=0;
char buffer[BUFFER_SIZE];

	sprintf(buffer, "%s/%s/%s", mtu_dir, "data", "secgen.dat");
	if((mtu_fp = fopen(buffer, "a+")) == NULL) {
		fprintf(stderr, "Cannot open %s for writing(%d).\n", buffer, errno);
		return(0);
	}
	fprintf(mtu_fp, HEADER_LINE);
	fprintf(mtu_fp, "##\n## Jump probability table\n##  format:\n");
	fprintf(mtu_fp, "## J:X-X:1:2:3:4\n");
	fprintf(mtu_fp, "##   X is relative ports\n");
	fprintf(mtu_fp, 
		"##   the numbers are the roll on 1d6 for a route to exist.\n##\n");
	fprintf(mtu_fp, HEADER_LINE);

	while(read_line(gal_fp, buffer) != NULL) {
		if('-' == buffer[1]) {
			fprintf(mtu_fp, "J:%c-%c:%c:%c:%c:%c\n", buffer[0], buffer[2], 
				buffer[11], buffer[20], buffer[29], buffer[38]);
			line_count++;
		}
	}

	fclose(mtu_fp);
	return(line_count);
}

/* convert gal.lst in the galactic dir
 *  then walk thru each 'galaxy'
 */
int
conv_gal(char *gal_dir, char *mtu_dir)
{
FILE *gal_fp,*mtu_fp;
int line_count=0;
char buffer[BUFFER_SIZE];

	/* get the data files we can use not many really */
	sprintf(buffer, "%s/%s", mtu_dir, "data");
	/* create the data dir */
	if(mkdir(buffer, 0775) < 0) {
		if(EEXIST != errno) {
			fprintf(stderr, "Cannot create %s (%d).\n", buffer, errno);
/*			return(0);	don't give up yet */
		}
	} else {
		/* start w/ the jump table... */
		sprintf(buffer, "%s/data/j1.dat", gal_dir);
		if((gal_fp = fopen(buffer, "r")) != NULL) {
			conv_jump_table(gal_fp, mtu_dir);
			fclose(gal_fp);
		} else
			fprintf(stderr, "Cannot open %s for reading(%d).\n", buffer, errno);

		/* convert the star generation table */
		sprintf(buffer, "%s/data/sysmap/stars.dat", gal_dir);
		if((gal_fp = fopen(buffer, "r")) != NULL) {
			conv_star_table(gal_fp, mtu_dir);
			fclose(gal_fp);
		} else
			fprintf(stderr, "Cannot open %s for reading(%d).\n", buffer, errno);

		/* convert the system generation table */
		sprintf(buffer, "%s/data/sysmap/planets.dat", gal_dir);
		if((gal_fp = fopen(buffer, "r")) != NULL) {
			conv_planet_table(gal_fp, mtu_dir);
			fclose(gal_fp);
		} else
			fprintf(stderr, "Cannot open %s for reading(%d).\n", buffer, errno);
	}

	/* finally, walk through the galaxies */
	sprintf(buffer, "%s/%s/%s", gal_dir, "gals", "gal.lst");
	if((gal_fp = fopen(buffer, "r")) == NULL) {
		fprintf(stderr, "Cannot open %s for reading(%d).\n", buffer, errno);
		return(0);
	}

	sprintf(buffer, "%s/%s", mtu_dir, "mtu.dat");
	if((mtu_fp = fopen(buffer, "a+")) == NULL) {
		fprintf(stderr, "Cannot open %s for writing(%d).\n", buffer, errno);
		fclose(gal_fp);
		return(0);
	}

	fprintf(mtu_fp, HEADER_LINE);
	fprintf(mtu_fp, "## Universes for MTU\n");

	while(read_line(gal_fp, buffer) != NULL) {
		char *dir,*menu;
		char gal_buff[BUFFER_SIZE];
		char mtu_buff[BUFFER_SIZE];

		line_count++;
		if(line_count < 3)
			continue;
		dir = buffer;
		menu = strchr(buffer, ' ');
		*menu = 0;
		menu++;
		while(isspace(*menu))
			menu++;

		sprintf(gal_buff, "%s/gals/%s", gal_dir, dir);
		sprintf(mtu_buff, "%s/%s", mtu_dir, dir);
		if(conv_dir(dir, gal_buff, mtu_buff))
			fprintf(mtu_fp, "%s:%s\n", dir, menu);
	}

	fclose(gal_fp);
	fclose(mtu_fp);
	return(1);
}

main(int argc, char *argv[])
{
char buffer[256],*n;
char *gal_dir=NULL,*mtu_dir=NULL;
struct stat stat_struct;
int err=1;
int i;

	if(3 == argc) {
		for(i = 1;i < 3;i++) {
			if('-' == argv[i][0]) {
				if('i' == argv[i][1]) {
					if(NULL == gal_dir) {
						gal_dir = &argv[i][2];
					} else
						fprintf(stderr, "Cannot use multiple input dirs.\n");
				} else if('o' == argv[i][1]) {
					n = &argv[i][2];
					if(NULL == mtu_dir) {
						mtu_dir = &argv[i][2];
					} else
						fprintf(stderr, "Cannot use multiple output dirs.\n");
				}
			}
		}

		if(NULL == gal_dir) {
			fprintf(stderr, "No input dir.\n");
			err = 0;
		} else if(stat(gal_dir, &stat_struct) != 0) {
			perror(gal_dir);
			err = 0;
		}
		if(NULL == mtu_dir) {
			fprintf(stderr, "No output dir.\n");
			err = 0;
		} else if(stat(mtu_dir, &stat_struct) != 0) {
			perror(mtu_dir);
			err = 0;
		}

		if(err) {
			if(conv_gal(gal_dir, mtu_dir))
				err = 0;
		}
	}


	if(err) {
		fprintf(stderr, "%s: Usage:\n\t%s -i<galactic dir> -o<mtu dir>\n",
			argv[0], argv[0]);
		fprintf(stderr, "where:\n");
		fprintf(stderr, "  galactic dir is the galactic dir.\n");
		fprintf(stderr, "  mtu dir is the MTU dir.\n");
		exit(1);
	}
}

